﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Management.Automation.Runspaces;
using System.Text.RegularExpressions;

namespace Curse.MSBuild.Deployment
{

    internal class AppPool
    {
        public string Name { get; private set; }

        public bool IsRunning { get; private set; }

        private AppPool(string name, bool isRunning)
        {
            Name = name;
            IsRunning = isRunning;
        }

        public void Start(Runspace runspace)
        {
            var results = RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"& {0} start apppool {1}", LoadTestWebAdminHelper.AppCmdLocation, Name));
            IsRunning = results.Any(r => r.Contains(string.Format(@"""{0}"" successfully started", Name)));
        }

        public void Stop(Runspace runspace)
        {
            var results = RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"& {0} stop apppool {1}", LoadTestWebAdminHelper.AppCmdLocation, Name));
            IsRunning = !results.Any(r => r.Contains(string.Format(@"""{0}"" successfully stopped", Name)) || r.Contains(string.Format(@"""{0}"" already stopped", Name)));
        }

        public static AppPool Get(Runspace runspace, string appPoolName)
        {
            var results = RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"& {0} list apppool {1}", LoadTestWebAdminHelper.AppCmdLocation, appPoolName), false);
            return results.Any() ? new AppPool(appPoolName, results[0].Contains("state:Started")) : null;
        }

        public static AppPool Create(Runspace runspace, string appPoolName)
        {
            RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"& {0} add apppool /name:""{1}""", LoadTestWebAdminHelper.AppCmdLocation, appPoolName));
            return Get(runspace, appPoolName);
        }

        public void Configure(Runspace runspace, bool enableManagedCode, string managedRuntimeVersion)
        {
            var scriptLines = new List<string>();
            var configArgs = new List<string> {"/autoStart:true"};

            if (enableManagedCode)
            {
                // Clear the schedule before configuring the rest
                scriptLines.Add(string.Format("& {0} set apppool {1} /-recycling.periodicRestart.schedule", LoadTestWebAdminHelper.AppCmdLocation, Name));

                configArgs.Add(string.Format("/managedRuntimeVersion:\"{0}\"", managedRuntimeVersion));
                configArgs.Add("/recycling.periodicRestart.time:\"00:00:00\"");
                configArgs.Add("/recycling.logEventOnRecycle:\"ConfigChange,OnDemand,Requests,Schedule,Time,Memory,IsapiUnhealthy\"");

                var firstRecycle = GetFirstRecycle(runspace.ConnectionInfo.ComputerName);
                configArgs.Add(string.Format("\"/+recycling.periodicRestart.schedule.[value='{0}']\"", firstRecycle.ToString(@"hh\:mm\:ss")));
                configArgs.Add(string.Format("\"/+recycling.periodicRestart.schedule.[value='{0}']\"", firstRecycle.Add(TimeSpan.FromHours(12)).ToString(@"hh\:mm\:ss")));
            }
            else
            {
                configArgs.Add("/managedRuntimeVersion:");
            }

            scriptLines.Add(string.Format(@"{0} set apppool ""{1}"" {2}", LoadTestWebAdminHelper.AppCmdLocation, Name, string.Join(" ", configArgs)));

            RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Join("\n", scriptLines));
            // TODO: Throw exception on failure
        }

        public void Recycle(Runspace runspace)
        {
            RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"& {0} recycle apppool {1}", LoadTestWebAdminHelper.AppCmdLocation, Name));
        }

        private TimeSpan GetFirstRecycle(string serverName)
        {
            var match = Regex.Match(serverName, @"(?<Number>\d{1,3})(?<Letter>a|b|c)");
            if (match.Success)
            {
                int firstHour = int.Parse(match.Groups["Number"].Value)%12;
                int minutes = 0;
                switch (match.Groups["Letter"].Value.ToLower())
                {
                    case "a":
                        minutes = 0;
                        break;
                    case "b":
                        minutes = 20;
                        break;
                    case "c":
                        minutes = 40;
                        break;
                }

                return TimeSpan.FromHours(firstHour).Add(TimeSpan.FromMinutes(minutes));
            }
            else
            {
                return  TimeSpan.FromHours(11);
            }
        }
    }


    internal class HostedSite
    {
        public bool IsRunning { get; private set; }

        public string Name { get; private set; }

        private readonly List<string> _bindingInfos;

        private HostedSite(string name, bool isRunning, string[] bindingInfos)
        {
            Name = name;
            IsRunning = isRunning;
            _bindingInfos = new List<string>(bindingInfos);
        }

        public void Start(Runspace runspace)
        {
            var results = RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"& {0} start site ""{1}""", LoadTestWebAdminHelper.AppCmdLocation, Name));
            IsRunning = results.Any(r => r.Contains(String.Format(@"""{0}"" successfully started", Name)));
        }

        public static HostedSite Get(Runspace runspace, string siteName)
        {
            var results = RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"& {0} list site ""{1}""", LoadTestWebAdminHelper.AppCmdLocation, siteName), false);
            var result = results.FirstOrDefault();
            if (result == null)
            {
                return null;
            }

            return new HostedSite(siteName, result.Contains("state:Started"), ParseBindingInfos(result));
        }

        private static string[] ParseBindingInfos(string siteInfo)
        {
            return Regex.Matches(siteInfo, @"(?<=https?/).*?:\d+:.*?(?=,)").Cast<Match>().Select(m => m.Value).ToArray();
        }

        public static HostedSite Create(Runspace runspace, string siteName, string physicalPath)
        {
            var createSiteScript = string.Format("{0}\n{1}",
                string.Format(@"& {0} add site /name:""{1}"" /physicalPath:""{2}""",LoadTestWebAdminHelper.AppCmdLocation,siteName,physicalPath),
                string.Format(@"& {0} set app ""{1}/"" /applicationPool:""{1}""",LoadTestWebAdminHelper.AppCmdLocation,siteName));

            RemoteScriptHelper.ExecuteRemoteScript(runspace, createSiteScript);
            return Get(runspace, siteName);
            // TODO: Set app pool for all apps? - Load Testing has no multi-app sites currently
        }

        public void Configure(Runspace runspace, string[] bindings)
        {

            var configArgs = new List<string> {"/serverAutoStart:true"};

            foreach(var binding in bindings)
            {
                var bindingInfo = string.Format("*:80:{0}", binding);
                if(!_bindingInfos.Contains(bindingInfo))
                {
                    configArgs.Add(string.Format("\"/+bindings.[protocol='http',bindingInformation='{0}']\"", bindingInfo));
                    _bindingInfos.Add(bindingInfo);
                }
            }

            RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"& {0} set site ""{1}"" {2}", LoadTestWebAdminHelper.AppCmdLocation, Name, string.Join(" ", configArgs)));
            // TODO: Handle Failures
        }

        public void ChangePhysicalPath(Runspace runspace, string path)
        {
            RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"& {0} set site ""{1}"" ""/[path='/'].[path='/'].physicalPath:`""{2}`""""", LoadTestWebAdminHelper.AppCmdLocation, Name, path));
        }
    }


    public static class LoadTestWebAdminHelper
    {
        public const string AppCmdLocation = @"C:\Windows\System32\inetsrv\appcmd.exe";

        private static AppPool ConfigureAppPool(Runspace runspace, string serverName, string siteName, bool enableManagedCode, string managedRuntimeVersion = "", bool resetConfig = false)
        {
            try
            {
                AppPool applicationPool = AppPool.Get(runspace, siteName);
                if (applicationPool == null || resetConfig)
                {
                    // Create the application pool
                    applicationPool = applicationPool ?? AppPool.Create(runspace, siteName);
                    applicationPool.Configure(runspace, enableManagedCode, managedRuntimeVersion);
                }
                return AppPool.Get(runspace, siteName);
            }
            catch (Exception)
            {
                Console.ForegroundColor = ConsoleColor.Cyan;
                Console.WriteLine("{0} - {1} - {2} - {3} - {4}", serverName, siteName, managedRuntimeVersion, enableManagedCode, resetConfig);
                Console.ForegroundColor = ConsoleColor.White;
                throw;
            }
        }


        public static void CreateWebSite(Runspace runspace, string serverName, string siteName, string physicalPath, string[] siteBindings, bool enableManagedCode, string managedRuntimeVersion = "")
        {
            try
            {
                var applicationPool = ConfigureAppPool(runspace, serverName, siteName, enableManagedCode, managedRuntimeVersion);

                if (!applicationPool.IsRunning)
                {
                    applicationPool.Start(runspace);
                }

                var mySite = HostedSite.Get(runspace, siteName) ?? HostedSite.Create(runspace, siteName, physicalPath);
                mySite.Configure(runspace, siteBindings);

                // Get fresh data and check state
                if (!HostedSite.Get(runspace, siteName).IsRunning)
                {
                    Console.Write("Starting site...");
                    mySite.Start(runspace);
                    Console.WriteLine("Done");
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("Error Creating Web Site: " + ex.Message);
                Console.WriteLine("Stack: " + ex.StackTrace);
                throw;
            }
        }
    }
}
