﻿using System;
using System.Collections.Specialized;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Mail;
using System.Net.Sockets;
using System.Threading;
using Microsoft.Build.Framework;

namespace Curse.MSBuild.Deployment
{
    public class LoadTestWebFarmDeploy : ITask
    {
        private string AspNetTempFolderPath
        {
            get { return @"C:\Windows\Microsoft.NET\Framework\v4.0.30319\Temporary ASP.NET Files\root"; }
        }

        private string SixtyFourBitAspNetTempFolderPath
        {
            get { return @"C:\Windows\Microsoft.NET\Framework64\v4.0.30319\Temporary ASP.NET Files\root"; }
        }

        private string TempFolderPath
        {
            get { return string.Format(@"c:\Windows\Temp\Deploys\{0}\{1}", (WebAppName ?? ApplicationName), ApplicationVersion ?? "Unversioned"); }
        }

        private string GetApplicationPath(string node, bool returnLocalPath = false)
        {

            return returnLocalPath ? @"c:\inetpub\" + (WebAppName ?? ApplicationName) : @"\\" + node + @"\c$\inetpub\" + (WebAppName ?? ApplicationName);
        }

        private string GetStaticPath(string node, bool returnLocalPath = false)
        {
            return returnLocalPath ? @"c:\inetpub\" + (StaticAppName ?? ApplicationName) + "Static" : @"\\" + node + @"\c$\inetpub\" + (StaticAppName ?? ApplicationName) + "Static";
        }


        private HostFile HostFile { get; set; }


        private bool SkipRpcCalls { get; set; }


        public bool Execute()
        {
            if (ServerFilteringPrompt)
            {
                WebNodeNames = DeployUtils.PromptServerNames(WebNodeNames);
            }

            DeployStep.Initialize();

            DeployStep.RegisterStep("Predeploy Options", () => !AutoCommit, () =>
            {

                if (PromptUser("Do you want to skip web, static and media site setup for this deploy? (This can speed up deploys of existing sites)", new[] {ConsoleKey.Y, ConsoleKey.N}) ==
                    ConsoleKey.Y)
                {
                    SkipRpcCalls = true;
                }
                else
                {
                    SkipRpcCalls = false;
                }

            });


            DeployStep.RegisterStep("Creating Temp Folder", () => true, () =>
            {
                DirectoryInfo tempFolder = new DirectoryInfo(TempFolderPath);
                if (!tempFolder.Exists)
                {
                    tempFolder.Create();
                }
            });

            DeployStep.RegisterStep("Parsing Host File", () => true, () =>
            {
                HostFile = HostFile.Parse();
            });

            DeployStep.RegisterStep("Validating Configuration", () => true, ValidateConfiguration);

            DeployStep.RegisterStep("Create Media Sites", () => CreateMediaSites && !SkipRpcCalls, SetupMediaSites);

            if (WebAppZipPath == null)
            {
                DeployStep.RegisterStep("Zip Application Files", () => CopyAppFiles, () =>
                {
                    WebAppZipPath = Path.Combine(TempFolderPath, "Application.zip");
                    if (File.Exists(WebAppZipPath))
                    {
                        File.Delete(WebAppZipPath);
                    }

                    ZipHelper.CreateZip(WebAppFolderPath, WebAppZipPath);
                });
            }


            DeployStep.RegisterStep("Copy and Unzip Application Files", () => CopyAppFiles, () =>
            {
                foreach (var node in WebNodeNames)
                {

                    Console.Write("Copying app zip file to " + node + "...");
                    ZipHelper.CopyZipToTemp(WebAppZipPath, node, ApplicationVersion);
                    Console.WriteLine("Done");

                    Console.Write("Unzipping app zip file to " + node + "...");
                    string applicationPath = GetApplicationPath(node);
                    DirectoryInfo di = new DirectoryInfo(applicationPath);
                    if (!di.Exists)
                    {
                        di.Create();
                    }
                    using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                    {
                        ZipHelper.UnzipRemotelyWithPowershell(runspace, PathHelper.GetRemoteTempFilePath(WebAppZipPath, node, ApplicationVersion, true), GetApplicationPath(node, true), node,
                            UnzipExePath);
                    }
                    Console.WriteLine("Done");
                }
            });


            DeployStep.RegisterStep("Creating Application Web Sites", () => CreateWebSites && !SkipRpcCalls, () =>
            {
                foreach (var node in WebNodeNames)
                {
                    SetupWebSite(node);
                }
            });

            if (StaticZipPath == null)
            {
                DeployStep.RegisterStep("Zip Static Files", () => CopyStaticFiles, () =>
                {
                    StaticZipPath = Path.Combine(TempFolderPath, "Static.zip");
                    ZipHelper.CreateZip(StaticFolderPath, StaticZipPath);
                });
            }

            DeployStep.RegisterStep("Copy and Unzip Static Files", () => CopyStaticFiles, () =>
            {
                foreach (var node in WebNodeNames)
                {
                    Console.WriteLine("Copying static zip file to production...");
                    ZipHelper.CopyZipToTemp(StaticZipPath, node, ApplicationVersion);

                    Console.WriteLine("Unzipping static zip file to file server...");
                    DirectoryInfo di = new DirectoryInfo(GetStaticPath(node));
                    if (!di.Exists)
                    {
                        di.Create();
                    }
                    using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                    {
                        ZipHelper.UnzipRemotelyWithPowershell(runspace, PathHelper.GetRemoteTempFilePath(StaticZipPath, node, ApplicationVersion, true), GetStaticPath(node, true), node, UnzipExePath);
                    }

                    if (CopyStaticToCurrent)
                    {
                        Console.WriteLine("Copying static content to current folder...");
                        string newStaticPath = Path.Combine(GetStaticPath(node, true), ApplicationVersion);
                        string currentStaticPath = Path.Combine(GetStaticPath(node, true), "current");
                        RunRemoteCommand(node, string.Format(@"xcopy /e /y /q /r ""{0}"" ""{1}\""", newStaticPath, currentStaticPath));
                    }

                    // Copy the static web.config
                    FileInfo fi = new FileInfo(StaticConfigPath);
                    string destinationPath = Path.Combine(di.FullName, "web.config");
                    //if (!File.Exists(destinationPath))
                    //{
                    fi.CopyTo(Path.Combine(di.FullName, "web.config"), true);
                    //}
                }
            });

            DeployStep.RegisterStep("Create Static Web Sites", () => CreateStaticSites && !SkipRpcCalls, () =>
            {
                foreach (var node in WebNodeNames)
                {
                    SetupStaticSite(node);
                }
            });


            DeployStep.RegisterStep("Deploy Options", () => !AutoCommit && SendNotificationEmail, () =>
            {
                if (PromptUser("Do you want to send an email notification for this deploy?", new[] {ConsoleKey.Y, ConsoleKey.N}) == ConsoleKey.Y)
                {
                    Console.WriteLine("You have elected to send notification email.");
                    SendNotificationEmail = true;
                }
                else
                {
                    Console.WriteLine("You have elected NOT to send notification email.");
                    SendNotificationEmail = false;
                }

            });

            DeployStep.RegisterStep("Notification Email", () => SendNotificationEmail && Commit, () =>
            {
                MailMessage mailMessage = new MailMessage();
                mailMessage.Subject = NotificationEmailSubject;
#if DEBUG
                mailMessage.To.Add("rmuzzey@curse.com");
#else
                    mailMessage.To.Add(NotificationEmailRecipient);
#endif
                mailMessage.From = new MailAddress(NotificationEmailSender);
                mailMessage.Body = NotificationEmailBody;
                using (SmtpClient client = new SmtpClient(SmtpServerHost, 25))
                {
                    client.Credentials = new NetworkCredential(SmtpServerUsername, SmtpServerPassword);
                    client.Send(mailMessage);
                }
            });

            DeployStep.RegisterStep("Commit Deploy", () => Commit, CommitDeploy);

            DeployStep.RegisterStep("Notify New Relic Deploy", () => (Commit && NotifyNewRelic && NewRelicAppName != null), () =>
            {
                using (WebClient client = new WebClient())
                {
                    client.Headers.Add("x-api-key", NewRelicApiKey);

                    NameValueCollection vals = new NameValueCollection();
                    vals.Add("deployment[app_name]", NewRelicAppName);
                    vals.Add("deployment[revision]", ApplicationVersion);
                    vals.Add("deployment[user]", Environment.UserName);
                    Console.Write("Notifiying New Relic of deploy to " + NewRelicAppName);
                    try
                    {
                        client.UploadValues("https://rpm.newrelic.com/deployments.xml", vals);
                        Console.WriteLine("Done");
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine("Failed: " + ex.Message);
                    }


                }
            });

            DeployStep.RegisterStep("Notification Email Confirmation", () => Commit && SendNotificationEmail, () =>
            {
                MailMessage mailMessage = new MailMessage();
                mailMessage.Subject = "Re: " + NotificationEmailSubject;
#if DEBUG
                mailMessage.To.Add("rmuzzey@curse.com");
#else
                mailMessage.To.Add(NotificationEmailRecipient);
#endif
                mailMessage.From = new MailAddress(NotificationEmailSender);
                mailMessage.Body = NotificationEmailSuccessBody;
                using (SmtpClient client = new SmtpClient(SmtpServerHost))
                {
                    client.Credentials = new NetworkCredential(SmtpServerUsername, SmtpServerPassword);
                    client.Send(mailMessage);
                }
            });

            DeployStep.RegisterStep("Cleanup", () => CleanupAfterDeploy, () =>
            {
                // Cleanup any application 
                foreach (var webNodeName in WebNodeNames)
                {
                    if (CreateWebSites)
                    {
                        DirectoryInfo di = new DirectoryInfo(GetApplicationPath(webNodeName));

                        // Get all folders
                        var oldFolders = di.GetDirectories().OrderByDescending(p => p.Name).Skip(10).ToArray();

                        Console.WriteLine("Deleting " + oldFolders.Count() + " old application folders...");

                        foreach (var folder in oldFolders)
                        {
                            string localPath = Path.Combine(GetApplicationPath(webNodeName, true), folder.Name);
                            Console.WriteLine("Deleting " + folder.FullName + "...");
                            RunRemoteCommand(webNodeName, string.Format(@"cmd /c rmdir /s /q ""{0}""", localPath));
                        }
                    }

                    if (CreateStaticSites)
                    {
                        DirectoryInfo di = new DirectoryInfo(GetStaticPath(webNodeName));

                        // Get all folders
                        var oldFolders = di.GetDirectories().OrderByDescending(p => p.Name).Skip(10).ToArray();

                        Console.WriteLine("Deleting " + oldFolders.Count() + " old static folders...");

                        foreach (var folder in oldFolders)
                        {
                            string localPath = Path.Combine(GetStaticPath(webNodeName, true), folder.Name);
                            Console.WriteLine("Deleting " + folder.FullName + "...");
                            RunRemoteCommand(webNodeName, string.Format(@"cmd /c rmdir /s /q ""{0}""", localPath));
                        }
                    }

                }
            });

            return DeployStep.RunAll();
        }


        public static ConsoleKey PromptUser(string promptText, ConsoleKey[] options)
        {
            Console.ForegroundColor = ConsoleColor.Cyan;
            Console.Write(promptText);
            Console.ForegroundColor = ConsoleColor.Yellow;
            Console.Write(" [");

            for (int i = 0; i < options.Length; i++)
            {
                if (i > 0)
                {
                    Console.Write("/");
                }
                Console.Write(options[i].ToString());
            }
            Console.WriteLine("]");
            Console.ResetColor();
            var resp = Console.ReadKey(true);


            while (true)
            {
                if (options.Contains(resp.Key))
                {
                    break;
                }

                Console.WriteLine("Unknown Response. Please enter a valid option.");
                resp = Console.ReadKey(true);
            }

            return resp.Key;
        }


        private void RunRemoteCommand(string nodeName, string command)
        {
            using (var runspace = RemoteScriptHelper.CreateRunspace(nodeName))
            {
                RemoteScriptHelper.ExecuteRemoteScript(runspace, command);
            }
        }

        private void DeleteRemoteFolder(string nodeName, string command)
        {

        }

        private bool ValidateHost(string host)
        {
            IPHostEntry hostEntry = null;
            try
            {
                hostEntry = Dns.GetHostEntry(host);
            }
            catch (Exception ex)
            {
                return false;
            }

            return hostEntry != null && hostEntry.AddressList.Any(p => p.AddressFamily == AddressFamily.InterNetwork);
        }

        private void ValidateConfiguration()
        {

            if (!DisableBindingValidation)
            {
                for (int i = 0; i < SiteBindings.Length; i++)
                {

                    Console.Write("Validating site binding '" + SiteBindings[i] + "'...");

                    if (SiteBindings[i].Equals("Default", StringComparison.InvariantCultureIgnoreCase))
                    {
                        SiteBindings[i] = "";
                    }
                    else if (!ValidateHost(SiteBindings[i]))
                    {
                        Console.WriteLine("Failed");
                        throw new Exception("Unable to resolve DNS for host: " + SiteBindings[i]);
                    }
                    Console.WriteLine("Done");

                }
            }

            // Ensure that each node can be connected to
            foreach (string name in WebNodeNames)
            {
                string unc = @"\\" + name + @"\c$\";
                Console.Write("Validating web node access at: " + unc + "...");

                if (!ValidateFolder(unc))
                {
                    Console.WriteLine("Failed");
                    throw new Exception("Invalid folder: " + unc);
                }
                Console.WriteLine(" Passed");
            }

            if (CopyStaticFiles)
            {
                Console.Write("Validating static host '" + StaticSiteBinding + "'...");

                if (!ValidateHost(StaticSiteBinding))
                {
                    Console.WriteLine("Failed");
                    throw new Exception("Invalid static host: " + StaticSiteBinding);
                }
                else
                {
                    Console.WriteLine("Passed");
                }

                Console.Write("Validating static zip file at: " + StaticZipPath + "...");
                if (StaticZipPath != null)
                {
                    if (!ValidateFile(StaticZipPath))
                    {
                        Console.WriteLine(" Failed");
                        throw new Exception("Invalid static zip path: " + StaticZipPath);
                    }
                }
                else
                {
                    if (!ValidateFolder(StaticFolderPath))
                    {
                        Console.WriteLine(" Failed");
                        throw new Exception("Invalid static folder path: " + StaticFolderPath);
                    }
                }

                if (!ValidateFile(StaticConfigPath))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Invalid static file config: " + StaticConfigPath);
                }

                Console.WriteLine(" Passed");
            }

            if (CopyAppFiles)
            {
                Console.Write("Validating web app zip file at: " + WebAppZipPath + "...");
                if (WebAppZipPath != null)
                {
                    if (!ValidateFile(WebAppZipPath))
                    {
                        Console.WriteLine(" Failed");
                        throw new Exception("Invalid file: " + WebAppZipPath);
                    }
                }
                else
                {
                    if (!ValidateFolder(WebAppFolderPath))
                    {
                        Console.WriteLine(" Failed");
                        throw new Exception("Invalid file: " + WebAppZipPath);
                    }
                }

                Console.WriteLine(" Passed");
            }

            if (MediaNodeNames == null || !MediaNodeNames.Any(p => !string.IsNullOrEmpty(p)))
            {
                MediaNodeNames = _defaultMediaServerNames;
            }

            if (string.IsNullOrEmpty(MediaApplicationPath))
            {
                MediaApplicationPath = DefaultMediaApplicationPath;
            }

            if (string.IsNullOrEmpty(ManagedRuntimeVersion))
            {
                ManagedRuntimeVersion = _defaultManagedRuntimeVersion;
            }
            else if (ManagedRuntimeVersion == "None")
            {
                ManagedRuntimeVersion = "";
            }

            if (String.IsNullOrEmpty(ApplicationTestUrl))
            {
                ApplicationTestUrl = _defaultApplicationTestUrl;
            }

            if (string.IsNullOrEmpty(UnzipExePath))
            {
                string currentDirectory = Directory.GetCurrentDirectory();
                UnzipExePath = Path.Combine(currentDirectory, "unzip.exe");
                if (!ValidateFile(UnzipExePath))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Invalid unzeip exe path: " + UnzipExePath);
                }
            }

            if (CreateMediaSites)
            {
                Console.Write("Validating static host '" + MediaSiteBinding + "'...");

                if (!ValidateHost(MediaSiteBinding))
                {
                    Console.WriteLine("Failed");
                    throw new Exception("Invalid media host: " + MediaSiteBinding);
                }
                else
                {
                    Console.WriteLine("Passed");
                }

                foreach (var mediaNode in MediaNodeNames)
                {
                    string unc = @"\\" + mediaNode + @"\c$\";
                    Console.Write("Validating media node access at: " + unc + "...");

                    if (!ValidateFolder(unc))
                    {
                        Console.WriteLine(" Failed");
                        throw new Exception("Invalid media share: " + unc);
                    }
                    Console.WriteLine(" Passed");
                }

                Console.Write("Validating media share access at: " + MediaApplicationPath + "...");
                if (!ValidateFolder(MediaApplicationPath))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Invalid media share: " + MediaApplicationPath);
                }
                Console.WriteLine(" Passed");
            }

            if (SendNotificationEmail)
            {
                Console.Write("Validating notification email settings...");
                if (string.IsNullOrEmpty(NotificationEmailBody))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Missing NotificationEmailBody");
                }

                if (string.IsNullOrEmpty(NotificationEmailRecipient))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Missing NotificationEmailRecipient");
                }

                if (string.IsNullOrEmpty(NotificationEmailSender))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Missing NotificationEmailSender");
                }

                if (string.IsNullOrEmpty(NotificationEmailSubject))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Missing NotificationEmailSubject");
                }

                if (string.IsNullOrEmpty(SmtpServerHost))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Missing SmtpServerHost");
                }
                Console.WriteLine(" Passed");
            }

        }

        private static bool ValidateFolder(string folderPath)
        {
            try
            {
                var di = new DirectoryInfo(folderPath);
                if (!di.Exists)
                {
                    Console.WriteLine("Unable to open folder at: " + folderPath);
                    return false;
                }
            }
            catch (Exception ex)
            {

                Console.WriteLine("Unable to open folder share at: " + folderPath + ". Exception: " + ex.Message);
                return false;
            }

            return true;
        }

        private static bool ValidateFile(string filePath)
        {
            if (string.IsNullOrEmpty(filePath))
            {
                return false;
            }

            try
            {
                var fi = new FileInfo(filePath);
                if (!fi.Exists)
                {
                    Console.WriteLine("Unable to open file at: " + filePath);
                    return false;
                }
            }
            catch (Exception ex)
            {

                Console.WriteLine("Unable to open folder share at: " + filePath + ". Exception: " + ex.Message);
                return false;
            }

            return true;
        }

        public string PhysicalApplicationPath
        {
            get
            {
                return (!string.IsNullOrEmpty(ApplicationVersion) && !VersioningDisabled)
                    ? @"c:\inetpub\" + (WebAppName ?? ApplicationName) + @"\" + ApplicationVersion
                    : @"c:\inetpub\" + (WebAppName ?? ApplicationName);
            }
        }

        public string DefaultMediaApplicationPath
        {
            get { return string.Format(@"\\{0}\{1}\media", FileServerName, (MediaAppName ?? ApplicationName)); }
        }

        public string StaticApplicationPath
        {
            get { return string.Format(@"c:\inetpub\{0}Static", (StaticAppName ?? ApplicationName)); }
        }

        private void SetupMediaSites()
        {
            foreach (var node in MediaNodeNames)
            {
                using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                {
                    Console.WriteLine("Setting up media site on " + node + "...");
                    LoadTestWebAdminHelper.CreateWebSite(runspace, node, (MediaAppName ?? ApplicationName) + "Media", MediaApplicationPath, new[] {MediaSiteBinding}, false);
                }
            }
        }

        private void SetupStaticSite(string nodeName)
        {
            using (var runspace = RemoteScriptHelper.CreateRunspace(nodeName))
            {
                Console.WriteLine("Setting up static site on " + nodeName + "...");
                LoadTestWebAdminHelper.CreateWebSite(runspace, nodeName, (StaticAppName ?? ApplicationName) + "Static", StaticApplicationPath, new[] {StaticSiteBinding}, false);
            }
        }

        private void SetupWebSite(string nodeName)
        {
            using (var runspace = RemoteScriptHelper.CreateRunspace(nodeName))
            {
                Console.WriteLine("Setting up web site on: " + nodeName + "...");
                LoadTestWebAdminHelper.CreateWebSite(runspace, nodeName, (WebAppName ?? ApplicationName), PhysicalApplicationPath, SiteBindings, true, ManagedRuntimeVersion);
            }
        }

        private void CommitDeploy()
        {
            if (!AutoCommit)
            {
                var r = PromptUser("Press enter to begin the commit process (run any database scripts before doing this).", new[] {ConsoleKey.Enter});
            }

            bool isReleaseCandidateNode = true;
            int counter = 0;
            foreach (var node in WebNodeNames)
            {
                using(var runspace = RemoteScriptHelper.CreateRunspace(node)){
                if (!isReleaseCandidateNode && !AutoCommit)
                {
                    if (ConfirmCommitToSecondaries)
                    {
                        PromptUser("Press enter to commit to " + node, new[] {ConsoleKey.Enter});
                    }
                    else if (NodeRolloutDelay > 0)
                    {
                        Console.WriteLine("Waiting " + NodeRolloutDelay + " seconds before next node rollout...");
                        Thread.Sleep(TimeSpan.FromSeconds(NodeRolloutDelay));
                    }
                }

                Console.WriteLine("Commiting to node " + (++counter).ToString() + " of " + WebNodeNames.Count());
                IPHostEntry host = Dns.GetHostEntry(node);
                if (host == null || !host.AddressList.Any(p => p.AddressFamily == AddressFamily.InterNetwork))
                {
                    Console.WriteLine("Unable to resolve IP Address for " + node + ". Deploy cancelled!");
                    return;
                }

                var address = host.AddressList.FirstOrDefault(p => p.AddressFamily == AddressFamily.InterNetwork);

                AppPool healthCheckAppPool = null;
                // Sleep
                if (!AutoCommit)
                {
                    Console.Write("Getting health check app pool...");
                    healthCheckAppPool = AppPool.Get(runspace, "HealthMonitor");

                    if (healthCheckAppPool == null)
                    {
                        Console.WriteLine("Unable to find  " + node + ". Deploy cancelled!");
                        throw new Exception("Deploy cancelled, due to missing health check app pool");
                    }
                    Console.WriteLine("Done");

                    if (healthCheckAppPool.IsRunning)
                    {
                        Console.Write("Stopping health monitor...");
                        healthCheckAppPool.Stop(runspace);
                        Console.WriteLine("Done");
                    }

                    Console.Write("Waiting " + LoadBalancerWait.TotalSeconds.ToString("##,##0") + " seconds for load balancer...");
                    Thread.Sleep(LoadBalancerWait);
                    Console.WriteLine("Done");

                }

                var site = HostedSite.Get(runspace, WebAppName ?? ApplicationName);
                    var appPool = AppPool.Get(runspace, site.Name);

                // Update the physical path
                if (!SkipUpdateSitePath)
                {
                    if (appPool.IsRunning)
                    {
                        Console.Write("Stopping app pool...");
                        appPool.Stop(runspace);
                        int attempts = 0;
                        while (appPool.IsRunning)
                        {
                            Thread.Sleep(1000);
                            if (++attempts > 120)
                            {
                                Console.WriteLine("Failed to stop app pool after 2 minutes!");
                                throw new Exception("Deploy cancelled, due to a malfunctioning app pool");
                            }
                        }
                        Console.WriteLine("Done");
                    }

                    // Delete the contents of the ASP.Net Temp folders before updating the site path.
                    Console.WriteLine("Deleting ASP.Net Temp Files...");

                    if (Directory.Exists(string.Format(@"\\{0}\{1}", node, AspNetTempFolderPath.Replace(@"C:\", @"c$\"))))
                    {
                        RemoteScriptHelper.ExecuteRemoteScript(node, string.Format(@"rmdir -Recurse -Force ""{0}""", AspNetTempFolderPath));
                    }
                    if (Directory.Exists(string.Format(@"\\{0}\{1}", node, SixtyFourBitAspNetTempFolderPath.Replace(@"C:\", @"c$\"))))
                    {
                        RemoteScriptHelper.ExecuteRemoteScript(node, string.Format(@"rmdir -Recurse -Force ""{0}""", SixtyFourBitAspNetTempFolderPath));
                    }
                    Console.WriteLine("Done");

                    Console.Write("Updating site path...");
                    site.ChangePhysicalPath(runspace, PhysicalApplicationPath);
                    Console.WriteLine("Done");

                    Console.Write("Starting app pool...");
                    appPool.Start(runspace);
                    Console.WriteLine("Done");
                }

                if (SkipUpdateSitePath || !SkipApplicationPoolRecycle)
                {
                    Console.Write("Recycling app pool...");

                    appPool.Recycle(runspace);
                    Console.WriteLine("Done");
                }


                    if (!AutoCommit)
                    {
                        // Try to load the root URL
                        if (!TestUrl(node, ApplicationTestUrl, true))
                        {
                            Console.WriteLine("The default web page at " + SiteBindings.First() + " failed to load on " + node + ". The deploy has been cancelled!");
                            throw new Exception("Deploy cancelled, due to page error");
                        }

                        // Prompt for testing
                        if (isReleaseCandidateNode)
                        {


                            isReleaseCandidateNode = false;

                            Console.Write("Updating your host file...");
                            HostFile.AddOrReplaceEntry(SiteBindings.First(), address.ToString());
                            HostFile.AddOrReplaceEntry(StaticSiteBinding, address.ToString());
                            HostFile.Save();
                            Console.WriteLine("Done");

                            Console.ForegroundColor = ConsoleColor.White;
                            Console.WriteLine("Launching the release candidate of this site in your browser. Please confirm it is functioning properly, before committing.");
                            Console.ResetColor();

                            LaunchCleanBrowser("http://" + SiteBindings.First() + ApplicationTestUrl);

                            if (PromptUser("Are you ready to commit this build to " + node + "?", new[] {ConsoleKey.Y, ConsoleKey.N}) == ConsoleKey.N)
                            {
                                throw new Exception("Deploy cancelled, by user.");
                            }


                            Console.Write("Starting health check app pool... ");
                            healthCheckAppPool.Start(runspace);
                            Console.WriteLine("Done");

                            HostFile.RemoveEntry(SiteBindings.First());
                            HostFile.RemoveEntry(StaticSiteBinding);
                            HostFile.Save();

                            Console.ForegroundColor = ConsoleColor.White;
                            Console.WriteLine("Launching load balanced version of this site. This is what the public is currently seeing.");
                            Console.ResetColor();

                            LaunchCleanBrowser("http://" + SiteBindings.First());

                        }
                        else
                        {
                            if (ConfirmCommitToSecondaries)
                            {
                                PromptUser("Press ENTER to commit this build to " + node + "?", new[] {ConsoleKey.Enter});
                            }

                            if (!TestUrl(node, ApplicationTestUrl, true))
                            {
                                Console.WriteLine("The default web page at " + SiteBindings.First() + " failed to load on " + node + ".");
                                if (PromptUser("Do you wish to cancel the deploy?", new ConsoleKey[] {ConsoleKey.Y, ConsoleKey.N}) == ConsoleKey.Y)
                                {
                                    throw new Exception("Deploy cancelled, due to page error");
                                }
                                else
                                {
                                    continue;
                                }
                            }

                            healthCheckAppPool.Start(runspace);
                        }
                    }
                }
                Console.WriteLine("-------------------------------------------------------------------------------------");
            }

            Console.WriteLine("This deploy is now live!");
        }


        public static void LaunchCleanBrowser(string url)
        {
            try
            {
                var running = Process.GetProcessesByName("firefox");

                if (running.Any())
                {
                    PromptUser("To clear your DNS cache, we will now kill all running instances of Firefox. Press ENTER to continue.", new[] { ConsoleKey.Enter });
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("We were unable to determine if Firefox is running.");
            }
            

            try
            {
                var stillRunning = Process.GetProcessesByName("firefox");
                foreach (var process in stillRunning)
                {
                    process.Kill();
                    process.WaitForExit();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("We were unable to shut down Firefox. This may result in stale DNS lookups.");
            }


            Process.Start("firefox.exe", url);
        }

        private bool TestUrl(string nodeName, string relativeUrlPath, bool isInitialRequest)
        {
            if (!relativeUrlPath.StartsWith("/"))
            {
                relativeUrlPath = "/" + relativeUrlPath;
            }

            int timeout = (isInitialRequest && PageTestTimeoutSeconds > 0) ? PageTestTimeoutSeconds*1000 : 60000;

            HttpWebRequest webRequest = (HttpWebRequest) WebRequest.Create("http://" + nodeName + relativeUrlPath);
            webRequest.UserAgent = "googlebot (Curse Health Monitor)";
            webRequest.Proxy = null;
            webRequest.Host = SiteBindings.First();
            webRequest.Timeout = timeout; // 1 Minute
            webRequest.AllowAutoRedirect = true;

            string result = null;
            HttpWebResponse response = null;
            try
            {
                Console.Write("Loading web page from " + nodeName + " at " + SiteBindings.First() + relativeUrlPath + "...");
                response = (HttpWebResponse) webRequest.GetResponse();
                using (var reader = new StreamReader(response.GetResponseStream()))
                {
                    result = reader.ReadToEnd();
                }

            }
            catch (WebException ex)
            {
                Console.WriteLine(ex.Status.ToString());
                return false;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return false;
            }

            if (result.Length == 0)
            {
                Console.WriteLine("Empty Response");
                return false;
            }
            Console.WriteLine("Passed! Response status: " + response.StatusCode + ". Response size: " + result.Length + " bytes");
            return true;
        }


        public IBuildEngine BuildEngine { get; set; }
        public ITaskHost HostObject { get; set; }

        [Required]
        public string ApplicationName { get; set; }

        public string MediaAppName { get; set; }

        public string StaticAppName { get; set; }
        public string WebAppName { get; set; }

        private string _defaultApplicationTestUrl = "/";

        public string ApplicationTestUrl { get; set; }

        [Required]
        public string ApplicationVersion { get; set; }

        [Required]
        public bool VersioningDisabled { get; set; }

        private string _defaultManagedRuntimeVersion = "v4.0";
        public string ManagedRuntimeVersion { get; set; }

        public bool SkipApplicationPoolRecycle = true;
        public bool SkipUpdateSitePath = false;

        [Required]
        public string[] WebNodeNames { get; set; }

        [Required]
        public string FileServerName { get; set; }


        public string WebAppZipPath { get; set; }

        [Required]
        public string WebAppFolderPath { get; set; }

        public string StaticZipPath { get; set; }

        [Required]
        public string StaticFolderPath { get; set; }

        [Required]
        public string StaticConfigPath { get; set; }

        [Required]
        public string UnzipExePath { get; set; }

        [Required]
        public string[] SiteBindings { get; set; }

        [Required]
        public bool CopyAppFiles { get; set; }

        [Required]
        public bool CopyStaticFiles { get; set; }

        [Required]
        public bool CreateWebSites { get; set; }

        [Required]
        public bool CreateMediaSites { get; set; }

        [Required]
        public string MediaSiteBinding { get; set; }

        public string MediaApplicationPath { get; set; }

        private string[] _defaultMediaServerNames = new[] {"media01a-live.curse.us", "media01b-live.curse.us", "media01c-live.curse.us"};
        public string[] MediaNodeNames { get; set; }

        [Required]
        public bool CreateStaticSites { get; set; }

        public bool CopyStaticToCurrent { get; set; }

        [Required]
        public string StaticSiteBinding { get; set; }

        [Required]
        public bool CleanupAfterDeploy { get; set; }

        [Required]
        public bool SendNotificationEmail { get; set; }

        public string NotificationEmailRecipient { get; set; }

        public string NotificationEmailSender { get; set; }

        public string NotificationEmailSubject { get; set; }

        public string NotificationEmailBody { get; set; }

        public string NotificationEmailSuccessBody { get; set; }

        public string SmtpServerHost { get; set; }
        public string SmtpServerUsername { get; set; }
        public string SmtpServerPassword { get; set; }

        public bool NotifyNewRelic { get; set; }
        public string NewRelicApiKey = "113588c977a0f88ff09b4d8bb2f2577479f2385d39b5f45";
        public string NewRelicAppName { get; set; }

        public bool Commit = true;

        public bool AutoCommit { get; set; }
        public bool ConfirmCommitToSecondaries { get; set; }

        public bool DisableBindingValidation { get; set; }

        public int PageTestTimeoutSeconds { get; set; }

        public int NodeRolloutDelay { get; set; }

        public int LoadBalancerWaitSeconds { get; set; }

        public bool ServerFilteringPrompt { get; set; }

        private TimeSpan LoadBalancerWait
        {
            get
            {
                if (LoadBalancerWaitSeconds > 0)
                {
                    return TimeSpan.FromSeconds(LoadBalancerWaitSeconds);
                }

                return TimeSpan.FromSeconds(10);
            }
        }
    }
}
