﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Management.Automation;
using System.Management.Automation.Runspaces;
using System.Security;
using System.Threading;

namespace Curse.MSBuild.Deployment
{
    public static class RemoteScriptHelper
    {
        private static PSCredential _credential;

        public static void Initialize(string username, string password)
        {
            char[] cpwd = password.ToCharArray();
            var ss = new SecureString();
            foreach (char c in cpwd)
            {
                ss.AppendChar(c);
            }

            _credential = new PSCredential(username, ss);
        }

        const string shellUri = "http://schemas.microsoft.com/powershell/Microsoft.PowerShell";

        public static Runspace CreateRunspace(string ipAddress)
        {

            var connectionInfo = new WSManConnectionInfo(false, ipAddress, 5985, "/wsman", shellUri, _credential);
            connectionInfo.SkipCACheck = true;
            connectionInfo.SkipCNCheck = true;
            if (_credential != null)
            {
                connectionInfo.AuthenticationMechanism = AuthenticationMechanism.Basic;
            }

            var runspace = RunspaceFactory.CreateRunspace(connectionInfo);
            runspace.Open();
            return runspace;
        }

        public static void ExecuteRemoteScript(string ipAddress, string script)
        {
            using (var runspace = CreateRunspace(ipAddress))
            {
                ExecuteRemoteScript(runspace, script);
            }
        }

        public static void DeleteFolder(Runspace runspace, string directoryPath)
        {
            var pipeline = runspace.CreatePipeline();
            var command = new Command("Remove-Item");
            command.Parameters.Add(new CommandParameter("Recurse"));
            command.Parameters.Add(new CommandParameter("Force"));
            command.Parameters.Add(new CommandParameter(null, directoryPath));

            pipeline.Commands.Add(command);
            var results = pipeline.Invoke();
            foreach (var r in results)
            {
                Console.WriteLine(r);
            }

        }

        public static void RenameFolder(Runspace runspace, string originalPath, string newPath)
        {
            var pipeline = runspace.CreatePipeline();
            var command = new Command("Rename-Item");
            command.Parameters.Add(new CommandParameter(null, originalPath));
            command.Parameters.Add(new CommandParameter(null, newPath));

            pipeline.Commands.Add(command);
            var results = pipeline.Invoke();
            foreach (var r in results)
            {
                Console.WriteLine(r);
            }

        }

        public static string[] ExecuteRemoteScript(Runspace runspace, string script, bool printResults = true)
        {
            var pipeline = runspace.CreatePipeline();
            var command = new Command(script, true);
            pipeline.Commands.Add(command);
            var results = pipeline.Invoke();
            var output = results.Select(r => r?.ToString() ?? string.Empty).ToArray();

            if (printResults)
            {
                foreach (var r in output)
                {
                    Console.WriteLine(r);
                }
            }

            return output;
        }

        public static dynamic[] ExecuteRemoteScriptDynamic(Runspace runspace, string script, bool printResults = true)
        {
            var pipeline = runspace.CreatePipeline();
            var command = new Command(script, true);
            pipeline.Commands.Add(command);
            var res = pipeline.Invoke();            
            var results = res.Select(p => (dynamic)p).ToArray();
            return results;
        }

        public static string GetAppPoolState(Runspace runspace, string appPoolName)
        {
            var results = ExecuteRemoteScriptDynamic(runspace, "Get-WebAppPoolState " + appPoolName);

            var result = results.FirstOrDefault();

            if (result == null)
            {
                return null;
            }

            return result.Value;
        }

        public static bool StartAppPool(Runspace runspace, string appPoolName, bool waitForCompletion = true, int maxAttempts = 90)
        {
            try
            {
                var appPoolState = GetAppPoolState(runspace, appPoolName);

                if (appPoolState == "Started")
                {
                    Console.WriteLine("App pool '" + appPoolState + "' is already started.");
                    return true;
                }

                ExecuteRemoteScript(runspace, "Start-WebAppPool " + appPoolName);

                var attempt = 0;
                do
                {
                    appPoolState = GetAppPoolState(runspace, appPoolName);
                    if (appPoolState != "Started")
                    {
                        if (++attempt == maxAttempts)
                        {
                            Console.WriteLine("Failed to start app pool '" + appPoolName + "' after waiting a while. Current state is: " + appPoolState);
                            return false;
                        }
                        Console.WriteLine("Waiting 1 second...");
                        Thread.Sleep(1000);
                    }


                } while (appPoolState != "Started");


                Console.WriteLine("App pool '" + appPoolState + "' has been started.");
                return true;
            }
            catch (Exception ex)
            {
                return false;
            }
        }

        public static bool StopAppPool(Runspace runspace, string appPoolName, bool waitForCompletion = true, int maxAttempts = 90)
        {
            try
            {
                var appPoolState = GetAppPoolState(runspace, appPoolName);

                if (appPoolState == "Stopped")
                {
                    Console.WriteLine("App pool '" + appPoolName + "' is already stopped.");
                    return true;
                }

                ExecuteRemoteScript(runspace, "Stop-WebAppPool " + appPoolName);

                var attempt = 0;
                do
                {

                    appPoolState = GetAppPoolState(runspace, appPoolName);
                    if (appPoolState != "Stopped")
                    {
                        if (++attempt == maxAttempts)
                        {
                            Console.WriteLine("Failed to stop app pool '" + appPoolName + "' after a while. Current state is: " + appPoolState);
                            return false;
                        }
                        Console.WriteLine("Waiting 1 second...");
                        Thread.Sleep(1000);
                    }


                } while (appPoolState != "Stopped");


                Console.WriteLine("App pool '" + appPoolName + "' has been stopped.");
                return true;
            }
            catch (Exception ex)
            {
                return false;
            }

        }

        public static string GetWebsiteState(Runspace runspace, string siteName)
        {
            var results = ExecuteRemoteScriptDynamic(runspace, string.Format(@"(Get-Website {0}).State", siteName));

            var result = results.FirstOrDefault();

            if (result == null)
            {
                return null;
            }

            return result.ToString();
        }

        public static bool StartWebsite(Runspace runspace, string siteName, int maxAttempts = 90)
        {
            try
            {
                var siteStatus = GetWebsiteState(runspace, siteName);

                if (siteStatus == "Started")
                {
                    Console.WriteLine("Website '" + siteName + "' is already started.");
                    return true;
                }

                ExecuteRemoteScript(runspace, "Start-Website " + siteName);

                var attempt = 0;
                do
                {
                    siteStatus = GetWebsiteState(runspace, siteName);
                    if (siteStatus != "Started")
                    {
                        Console.WriteLine("Site status is {0}", siteStatus);
                        if (++attempt == maxAttempts)
                        {
                            Console.WriteLine("Failed to start Website '" + siteName + "' after waiting a while. Current state is: " + siteStatus);
                            return false;
                        }
                        Console.WriteLine("Waiting 1 second...");
                        Thread.Sleep(1000);
                    }


                } while (siteStatus != "Started");


                Console.WriteLine("Website '" + siteName + "' has been started.");
                return true;
            }
            catch (Exception ex)
            {
                return false;
            }
        }

        public static bool ChangePhysicalPath(Runspace runspace, string siteName, string physicalPath)
        {
            try
            {
                var currentPath = GetPhysicalPath(runspace, siteName);
                if (currentPath == physicalPath)
                {
                    Console.WriteLine("No changes needed to the physical path");
                    return true;
                }

                ExecuteRemoteScript(runspace, string.Format(@"Set-ItemProperty 'IIS:\Sites\{0}\' -Name physicalPath -Value ""{1}""", siteName, physicalPath));
                currentPath = GetPhysicalPath(runspace, siteName);
                return currentPath == physicalPath;
            }
            catch (Exception ex)
            {
                return false;
            }
        }

        private static string GetPhysicalPath(Runspace runspace, string siteName)
        {
            var results = ExecuteRemoteScriptDynamic(runspace, string.Format(@"(Get-Website ""{0}"").physicalPath", siteName));

            var result = results.FirstOrDefault();
            if (result == null)
            {
                return null;
            }

            return result.Value;
        }

        public static string GetServiceState(Runspace runspace, string serviceName)
        {
            var pipeline = runspace.CreatePipeline();
            var command = new Command("Get-Service");
            command.Parameters.Add(new CommandParameter("name", serviceName));
            pipeline.Commands.Add(command);
            var resp = pipeline.Invoke();
            if (resp == null)
            {
                return null;
            }

            var results = resp.FirstOrDefault() as dynamic;
            if (results == null)
            {
                return null;
            }

            return results.Status;
        }

        public static bool IsServiceInstalled(Runspace runspace, string serviceName)
        {
            return GetServiceState(runspace, serviceName) != null;
        }

        public static bool StopService(Runspace runspace, string serviceName, bool waitForCompletion = true, int maxAttempts = 90)
        {
            return ChangeServiceState("Stop-Service", "Stopped", runspace, serviceName, waitForCompletion, maxAttempts);
        }

        public static bool StartService(Runspace runspace, string serviceName, bool waitForCompletion = true, int maxAttempts = 90)
        {
            return ChangeServiceState("Start-Service", "Running", runspace, serviceName, waitForCompletion, maxAttempts);
        }

        private static bool ChangeServiceState(string command, string desiredState, Runspace runspace, string serviceName, bool waitForCompletion, int maxAttempts)
        {
            try
            {
                var currentState = GetServiceState(runspace, serviceName);

                if (currentState.Equals(desiredState, StringComparison.InvariantCultureIgnoreCase))
                {
                    Console.WriteLine(" '" + serviceName + "' is already in the " + desiredState + " state...");
                    return true;
                }

                ExecuteRemoteScript(runspace, command + " " + serviceName);

                var attempt = 0;
                do
                {
                    currentState = GetServiceState(runspace, serviceName);

                    if (!currentState.Equals(desiredState, StringComparison.InvariantCultureIgnoreCase))
                    {
                        if (++attempt == maxAttempts)
                        {
                            Console.WriteLine("Failed to start service '" + serviceName + "' after waiting a while. Current state is: " + currentState);
                            return false;
                        }
                        Console.WriteLine("Waiting 1 second...");
                        Thread.Sleep(1000);
                    }


                } while (!currentState.Equals(desiredState, StringComparison.InvariantCultureIgnoreCase));


                Console.WriteLine(serviceName + " is now in the '" + desiredState + "' state.");
                return true;
            }
            catch (Exception ex)
            {
                Console.WriteLine("Failed to change service state: " + ex.Message);
                return false;
            }
        }

        public static void CreateOrUpdateWebSite(Runspace runspace, string siteName, string physicalPath, string[] siteBindings, bool updatePhysicalPath=true)
        {
            Console.WriteLine("Updating " + siteName);
            var siteResults = ExecuteRemoteScript(runspace, "Get-Website " + siteName, false);
            if (siteResults.Length == 0)
            {
                // Create
                Console.WriteLine("Creating site " + siteName);
                ExecuteRemoteScript(runspace, string.Format(@"New-Website -Name ""{0}"" -Port 80 -HostHeader ""{1}"" -PhysicalPath ""{2}""", siteName, siteBindings.First(), physicalPath));
            }

            Console.WriteLine("Examining site bindings...");
            var existingBindings = new HashSet<string>(
                ExecuteRemoteScriptDynamic(runspace, string.Format(@"(Get-Website ""{0}"").Bindings.Collection", siteName), false).Select(binding => (string) binding.bindingInformation));

            Console.WriteLine("Existing Bindings:");
            foreach(var binding in existingBindings)
            {
                Console.WriteLine($"\"{ binding}\"");
            }
            Console.WriteLine();
            foreach (var binding in siteBindings)
            {
                var bindingInformation = "*:80:" + binding;
                Console.WriteLine($"Checking for binding: \"{bindingInformation}\"");

                if (!existingBindings.Contains(bindingInformation))
                {
                    Console.WriteLine($"Creating binding \"{binding}\"");
                    ExecuteRemoteScript(runspace, string.Format(@"New-WebBinding -Name ""{0}"" -HostHeader ""{1}"" -Port 80 -IPAddress ""*""", siteName, binding));
                }
            }

            if (updatePhysicalPath)
            {
                ChangePhysicalPath(runspace, siteName, physicalPath);
            }

            // Get fresh data and check state
            var state = GetWebsiteState(runspace, siteName);
            if (state != "Started")
            {
                Console.Write("Starting site...");
                StartWebsite(runspace, siteName);
                Console.WriteLine("Done");
            }
        }

        public static string GetRemoteTempFolderPath(Runspace runspace, string version, bool createIfMissing = true)
        {
            string localPath = @"c:\Windows\Temp\Deploys\" + version;
            if (createIfMissing)
            {
                EnsureFolderExists(runspace, localPath);
            }

            return localPath;
        }

        public static void EnsureFolderExists(Runspace runspace, string path)
        {
            var res = ExecuteRemoteScript(runspace, "test-path " + path, false);
            var exists = res.FirstOrDefault(r => r.ToLower() == "true") != null;

            if (!exists)
            {
                ExecuteRemoteScript(runspace, $"new-item {path} -type directory", false);
            }
        }

        public static string GetRemoteTempFilePath(Runspace runspace, string zipFilePath, string version, bool createIfMissing = true)
        {
            var filename = Path.GetFileName(zipFilePath);
            return GetRemoteTempFolderPath(runspace, version, createIfMissing) + @"\" + filename;
        }

        public static void Unzip(Runspace runspace, string zipFilePath, string destinationPath)
        {
            ExecuteRemoteScript(runspace, $"Add-Type -AssemblyName System.IO.Compression.FileSystem; [System.IO.Compression.ZipFile]::ExtractToDirectory('{zipFilePath}', '{destinationPath}');");
        }

    }
}
