﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Http;
using System.Text;
using Curse.Extensions;
using Curse.MSBuild.Chef;
using Newtonsoft.Json;

namespace Curse.MSBuild.Deployment
{
    public class SensuHelper
    {
        private const long _stashTimeout = 420L;

        public static void SilenceWorkerAlerts(string nodeName, IEnumerable<string> chefEnvironments)
        {
            try
            {
                Console.WriteLine("Silencing worker node alerts on {0}", nodeName);

                var hostName = GetHostName(nodeName);
                var stashPath = string.Format("silence/{0}/CurseFriendsWorkerProcess", hostName);
                DoSilenceAlerts(nodeName, chefEnvironments, stashPath, "Scheduled Worker Deploy");
            }
            catch (Exception ex)
            {
                Console.WriteLine("Failed to silence Sensu alerts for worker deploy. " + ex.Message);
                Console.WriteLine("Continuing anyway");
            }
        }

        public static void SilenceHealthCheckAlerts(string nodeName, IEnumerable<string> chefEnvironments)
        {
            try
            {
                Console.WriteLine("Silencing health check alerts on {0}", nodeName);

                var hostName = GetHostName(nodeName);
                var stashPath = string.Format("silence/{0}/iis-http-healthcheck", hostName);
                DoSilenceAlerts(nodeName, chefEnvironments, stashPath, "Scheduled AppPool recycle");
            }
            catch (Exception ex)
            {
                Console.WriteLine("Failed to silence Sensu alerts for health-check. " + ex.Message);
                Console.WriteLine("Continuing anyway");
            }
        }

        private static string GetHostName(string nodeName)
        {
            return nodeName.Split(new[] {'.'}, 2)[0].ToLower();
        }

        private static void DoSilenceAlerts(string nodeName, IEnumerable<string> chefEnvironments, string path, string reason)
        {
            var sensuNode = GetSensuNode(nodeName, chefEnvironments);
            if (sensuNode == null)
            {
                throw new InvalidOperationException("Unable to determine Sensu URL");
            }

            var url = string.Format("http://{0}:4567/stashes", sensuNode);

            using (var client = new HttpClient())
            {
                try
                {
                    var res = client.GetAsync(url).Result;
                    if (!res.IsSuccessStatusCode)
                    {
                        throw new InvalidOperationException("GET REST request unsuccessful");
                    }
                    if (res.IsSuccessStatusCode)
                    {
                        var stashes = JsonConvert.DeserializeObject<Stash[]>(res.Content.ReadAsStringAsync().Result);
                        if (stashes == null || stashes.Length == 0)
                        {
                            throw new InvalidOperationException("GET REST response was empty");
                        }


                        var stash = stashes.FirstOrDefault(s => s.path == path);
                        if (stash != null && (stash.expire == -1 || stash.expire >= _stashTimeout))
                        {
                            Console.WriteLine("Skipping silence step, a stash already exists");
                            return;
                        }
                    }
                }
                catch (Exception ex)
                {
                    Console.WriteLine("Could not retrieve existing stashes. " + ex.Message);
                    Console.WriteLine("Assuming no stash exists and creating one");
                }

                Console.WriteLine("Creating a new stash to silence alerts");
                var content = new StringContent(JsonConvert.SerializeObject(new Stash
                {
                    path = path,
                    content = new StashContent
                    {
                        reason = reason,
                        message = reason,
                        timestamp = DateTime.UtcNow.SafeToEpochMilliseconds() / 1000,
                    },
                    expire = _stashTimeout

                }, Formatting.None, new JsonSerializerSettings { NullValueHandling = NullValueHandling.Ignore }), Encoding.UTF8, "application/json");
                var postRes = client.PostAsync(url, content).Result;
                if (!postRes.IsSuccessStatusCode)
                {
                    throw new InvalidOperationException("POST REST request unsuccessful");
                }
            }
        }

        private static string GetSensuNode(string nodeName, IEnumerable<string> chefEnvironments)
        {
            foreach (var environment in chefEnvironments)
            {
                try
                {
                    dynamic node = SousChef.GetDynamicNode(environment, nodeName);
                    return node.@override.sensu.common_config.api.host;
                }
                catch (Exception ex)
                {
                    Console.WriteLine("Error retrieving things! {0}", ex.Message);
                }
            }

            return null;
        }

        private class Stash
        {
            public string path { get; set; }

            public long expire { get; set; }

            public StashContent content { get; set; }
        }

        private class StashContent
        {
            public string reason { get; set; }

            public string message { get; set; }

            public string source { get; set; }

            public long timestamp { get; set; }
        }
    }
}
