﻿using Microsoft.Web.Administration;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace Curse.MSBuild.Deployment
{
    public static class WebAdminHelper
    {

        public static ApplicationPool ConfigureAppPool(string serverName, string siteName, bool enableManagedCode, string managedRuntimeVersion = "", bool resetConfiguration = false)
        {
            using (ServerManager serverManager = ServerManager.OpenRemote(serverName))
            {
                return ConfigureAppPool(serverName, serverManager, siteName, enableManagedCode, managedRuntimeVersion, resetConfiguration);
            }
        }

        public static void SetRecycleTimes(string serverName, string applicationPoolName, int additionalMinutes = 0, bool enableManagedCode = true, string managedRuntimeVersion = "v4.0", int recycleFrequencyHours = 1)
        {
            using (ServerManager serverManager = ServerManager.OpenRemote(serverName))
            {
                ApplicationPool applicationPool = serverManager.ApplicationPools.FirstOrDefault(p => p.Name == applicationPoolName);
                if (applicationPool == null)
                {
                    throw new Exception("Unknown Application: " + applicationPoolName);
                }
                
                applicationPool.AutoStart = true;
                applicationPool.Recycling.PeriodicRestart.Schedule.Clear();
                applicationPool.ManagedRuntimeVersion = managedRuntimeVersion;
                applicationPool.Recycling.PeriodicRestart.Time = TimeSpan.Zero;
                applicationPool.Recycling.LogEventOnRecycle = RecyclingLogEventOnRecycle.ConfigChange
                    | RecyclingLogEventOnRecycle.OnDemand
                    | RecyclingLogEventOnRecycle.Requests
                    | RecyclingLogEventOnRecycle.Schedule
                    | RecyclingLogEventOnRecycle.Time
                    | RecyclingLogEventOnRecycle.Memory
                    | RecyclingLogEventOnRecycle.IsapiUnhealthy
                    | RecyclingLogEventOnRecycle.ConfigChange;

                applicationPool.Recycling.PeriodicRestart.Schedule.Clear();

                int minutes = GetMinutesFromName(serverName) + additionalMinutes;

                Console.WriteLine("Recycles will occur " + minutes + " minutes after the hour");

                for (int i = 1; i < 24; i += recycleFrequencyHours)
                {
                    TimeSpan firstRecycle = TimeSpan.FromHours(i).Add(TimeSpan.FromMinutes(minutes));
                    applicationPool.Recycling.PeriodicRestart.Schedule.Add(firstRecycle);
                }

                serverManager.CommitChanges();
            }
        }


        private static int GetMinutesFromName(string serverName)
        {
            var match = Regex.Match(serverName, @"(?<Number>\d{1,3})(?<Letter>a|b|c)");
            int minutes = 0;
            if (match.Success)
            {
                int number = int.Parse(match.Groups["Number"].Value);
                string letter = match.Groups["Letter"].Value.ToLower();

                
                switch (letter)
                {
                    case "a":
                        minutes = 0;
                        break;
                    case "b":
                        minutes = 20;
                        break;
                    case "c":
                        minutes = 40;
                        break;

                }

               
            }
            return minutes;
        }
        public static ApplicationPool ConfigureAppPool(string serverName, ServerManager serverManager, string siteName, bool enableManagedCode, string managedRuntimeVersion = "", bool resetConfig = false)
        {
            ApplicationPool applicationPool = serverManager.ApplicationPools.FirstOrDefault(p => p.Name == siteName);            
            if (applicationPool == null || resetConfig)
            {                
                // Create the application pool
                applicationPool = applicationPool ?? serverManager.ApplicationPools.Add(siteName);                
                applicationPool.AutoStart = true;

                if (enableManagedCode)
                {
                    applicationPool.ManagedRuntimeVersion = managedRuntimeVersion;
                    applicationPool.Recycling.PeriodicRestart.Time = TimeSpan.Zero;
                    applicationPool.Recycling.LogEventOnRecycle = RecyclingLogEventOnRecycle.ConfigChange
                        | RecyclingLogEventOnRecycle.OnDemand
                        | RecyclingLogEventOnRecycle.Requests
                        | RecyclingLogEventOnRecycle.Schedule
                        | RecyclingLogEventOnRecycle.Time
                        | RecyclingLogEventOnRecycle.Memory
                        | RecyclingLogEventOnRecycle.IsapiUnhealthy
                        | RecyclingLogEventOnRecycle.ConfigChange;

                    applicationPool.Recycling.PeriodicRestart.Schedule.Clear();


                  
                    var match = Regex.Match(serverName, @"(?<Number>\d{1,3})(?<Letter>a|b|c)");
                    if (match.Success)
                    {
                        int number = int.Parse(match.Groups["Number"].Value);
                        string letter = match.Groups["Letter"].Value.ToLower();

                        int minutes = 0;
                        switch (letter)
                        {
                            case "a":
                                minutes = 0;
                                break;
                            case "b":
                                minutes = 20;
                                break;
                            case "c":
                                minutes = 40;
                                break;

                        }
                        
                        int firstHour = number % 12;

                        TimeSpan firstRecycle = TimeSpan.FromHours(firstHour).Add(TimeSpan.FromMinutes(minutes));
                        TimeSpan secondRecycle = firstRecycle.Add(TimeSpan.FromHours(12));

                        applicationPool.Recycling.PeriodicRestart.Schedule.Add(firstRecycle);
                        applicationPool.Recycling.PeriodicRestart.Schedule.Add(secondRecycle);
                    }
                    else
                    {
                        applicationPool.Recycling.PeriodicRestart.Schedule.Add(TimeSpan.FromHours(11));
                        applicationPool.Recycling.PeriodicRestart.Schedule.Add(TimeSpan.FromHours(23));
                    }
                    
                }
                else
                {
                    applicationPool.ManagedRuntimeVersion = "";
                }
            }
            serverManager.CommitChanges();
            return serverManager.ApplicationPools.FirstOrDefault(p => p.Name == siteName);            
        }


        public static void CreateWebSite(string serverName, string siteName, string physicalPath, string[] siteBindings, bool enableManagedCode, string managedRuntimeVersion = "")
        {
            try
            {
                using (ServerManager serverManager = ServerManager.OpenRemote(serverName))
                {
                    var applicationPool = ConfigureAppPool(serverName, serverManager, siteName, enableManagedCode, managedRuntimeVersion);

                    if (applicationPool.State != ObjectState.Started)
                    {
                        applicationPool.Start();
                    }

                    Site mySite = serverManager.Sites.FirstOrDefault(p => p.Name == siteName);

                    // See if the site already exists
                    if (mySite == null)
                    {
                        Console.WriteLine("Creating web site '" + siteName + "'...");

                        // Create a new site                        
                        
                        // Validate the site name
                        var invalid = SiteCollection.InvalidSiteNameCharacters();
                        if (siteName.IndexOfAny(invalid) > -1)
                        {                                                    
                            throw new ArgumentException("Invalid site name: " + siteName);
                        }

                        mySite = serverManager.Sites.Add(siteName, "http", "*:80:" + siteBindings.FirstOrDefault(), physicalPath);                                                 
                        mySite.ServerAutoStart = true;
                        mySite.Bindings.Clear();

                        foreach (string binding in siteBindings)
                        {
                            Console.Write("Creating binding site '" + binding + "'...");
                            mySite.Bindings.Add("*:80:" + binding, "http");
                            Console.WriteLine("Done");
                        }

                        foreach (Application app in mySite.Applications)
                        {
                            app.ApplicationPoolName = applicationPool.Name;
                        }

                        Console.Write("Committing changes to IIS...");
                        serverManager.CommitChanges();
                        mySite = serverManager.Sites.FirstOrDefault(p => p.Name == siteName);
                        Console.WriteLine("Done");
                    }
                    else
                    {
                        Console.WriteLine("Examining site bindings...");                        
                        var existingBindings = mySite.Bindings.ToArray();
                        foreach (var binding in siteBindings)
                        {
                            var bindingInfo = "*:80:" + binding;
                            if (!existingBindings.Any(p => p.BindingInformation.Equals(bindingInfo, StringComparison.InvariantCultureIgnoreCase)))
                            {
                                Console.Write("Creating binding site '" + binding + "'...");
                                mySite.Bindings.Add(bindingInfo, "http");
                                Console.WriteLine("Done");    
                            }
                        }
                        Console.Write("Committing changes to IIS...");
                        serverManager.CommitChanges();
                        mySite = serverManager.Sites.FirstOrDefault(p => p.Name == siteName);
                        Console.WriteLine("Done");
                    }

                    if (mySite.State != ObjectState.Started)
                    {
                        Console.Write("Starting site...");
                        mySite.Start();
                        Console.WriteLine("Done");
                    }


                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("Error Creating Web Site: " + ex.Message);
                Console.WriteLine("Stack: " + ex.StackTrace);
                throw;
            }
            
        }
    }
}
