﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Sockets;
using Microsoft.Build.Framework;
using System.IO;
using System.Diagnostics;
using System.Threading;
using Microsoft.Web.Administration;
using System.Net;
using System.Collections.Specialized;
using System.Net.Mail;
using Curse.MSBuild.Chef;
using System.Threading.Tasks;
using System.Collections.Concurrent;

namespace Curse.MSBuild.Deployment
{
    public class WebFarmDeploy : ITask
    {
        private string AspNetTempFolderPath
        {
            get
            {
                return @"C:\Windows\Microsoft.NET\Framework\v4.0.30319\Temporary ASP.NET Files\root";
            }
        }

        private string SixtyFourBitAspNetTempFolderPath
        {
            get
            {
                return @"C:\Windows\Microsoft.NET\Framework64\v4.0.30319\Temporary ASP.NET Files\root";
            }
        }

        private string TempFolderPath
        {
            get { return string.Format(@"c:\Windows\Temp\Deploys\{0}\{1}", (WebAppName ?? ApplicationName), ApplicationVersion ?? "Unversioned"); }
        }

        private string GetApplicationPath(string node, bool returnLocalPath = false, string appName = null)
        {
            return returnLocalPath ? @"c:\inetpub\" + (appName ?? WebAppName ?? ApplicationName) : @"\\" + node + @"\c$\inetpub\" + (WebAppName ?? ApplicationName);
        }

        private static string GetApplicationPath(string node, string appName, bool returnLocalPath)
        {
            return returnLocalPath ? @"c:\inetpub\" + appName : @"\\" + node + @"\c$\inetpub\" + appName;
        }

        private string GetVersionApplicationPath(string node, bool returnLocalPath = false)
        {
            return GetSuffixApplicationPath(node, returnLocalPath, ApplicationVersion);
        }

        private string GetSuffixApplicationPath(string node, bool returnLocalPath, string suffix)
        {
            return returnLocalPath ? @"c:\inetpub\" + (WebAppName ?? ApplicationName) + @"\" + suffix : @"\\" + node + @"\c$\inetpub\" + (WebAppName ?? ApplicationName) + @"\" + suffix;
        }

        private string GetStaticPath(string node, bool returnLocalPath = false)
        {
            return returnLocalPath ? @"c:\inetpub\" + (StaticAppName ?? ApplicationName) + "Static" : @"\\" + node + @"\c$\inetpub\" + (StaticAppName ?? ApplicationName) + "Static";
        }


        private HostFile HostFile { get; set; }

        private bool SkipRpcCalls { get; set; }




        public bool Execute()
        {
            ServicePointManager.ServerCertificateValidationCallback +=
                delegate
                {
                    return true; // **** Always accept
                };

            if (ApplicationName[0].ToString() == ApplicationName[0].ToString().ToLowerInvariant())
            {
                ApplicationName = ApplicationName[0].ToString().ToUpper() + ApplicationName.Substring(1);
                Console.WriteLine("Title casing application name to: " + ApplicationName);
            }

            var artifactUploadOnly = false;

            if(!string.IsNullOrEmpty(RemoteUsername) && !string.IsNullOrEmpty(RemotePasswordKey))
            {
                Console.WriteLine($"Using credentials for {RemoteUsername} for remote scripts using {CompilationMode}/{RemotePasswordKey} secrets.");
                var secretsHelper = SecretsHelper.GetSecretsHelper();
                var pw = secretsHelper.GetSecretValue(RemotePasswordKey, CompilationMode);                
                RemoteScriptHelper.Initialize(RemoteUsername, pw);
            }
            else
            {
                Console.WriteLine("Not using explicit credentials for remote scripts.");
            }

            if (!AutoCommit && UseS3)
            {
                var resp = DeployUtils.PromptUser("Would you like to build and upload an artifact to S3 without deploying?", new[] { ConsoleKey.Y, ConsoleKey.N });
                artifactUploadOnly = resp == ConsoleKey.Y;
                S3Helper.Initialize();
            }

            if (UseChef && !artifactUploadOnly)
            {
                if (string.IsNullOrEmpty(CompilationMode))
                {
                    Console.WriteLine("CompilationMode must be supplied.");
                    return false;
                }

                Console.WriteLine("Looking for nodes included in " + CompilationMode + " deploys...");

                var environments = DeployUtils.GetChefEnvironments(CompilationMode);
                if (environments.Length == 0)
                {
                    Console.WriteLine(CompilationMode + " compilations cannot be deployed using the Chef API.");
                    return false;
                }

                Console.WriteLine("Querying Chef for '" + ServiceName + "' service nodes in the following environments: " + string.Join(", ", environments));

                var iisNodes = SousChef.GetIISNodesForService(environments, ServiceName, false);

                WebNodeNames = iisNodes.Select(p => p.Name).ToArray();
                _siteBindingsByNode = iisNodes.ToDictionary(p => p.Name, p => p.HostName);

                if (!WebNodeNames.Any())
                {
                    Console.WriteLine("Unable to deploy. Chef found no nodes for this service: " + ServiceName);
                    return false;
                }

                SiteBindings = new[] { iisNodes.OrderByDescending(p => p.IsPreferred).Select(p => p.HostName).First() };

                Console.WriteLine("Using site binding: " + string.Join(", ", SiteBindings));
                Console.WriteLine("Found nodes: " + string.Join(", ", WebNodeNames));

                if (PromptUser("Shall we continue?", new[] { ConsoleKey.Y, ConsoleKey.N }, ConsoleKey.Y) != ConsoleKey.Y)
                {
                    return false;
                }
            }

            if (ServerFilteringPrompt && !artifactUploadOnly)
            {
                WebNodeNames = DeployUtils.PromptServerNames(WebNodeNames);
            }

            if (VersioningDisabled)
            {
                SkipUpdateSitePath = true;
            }

            var s3currentPath = String.Format("{0}/{0}_current.zip", ServiceName);
            var s3deployPath = String.Format("{0}/{0}_{1:s}.zip", ServiceName, DateTime.UtcNow);
            var s3StaticPath = String.Format("{0}/{0}_{1:s}_static.zip", ServiceName, DateTime.UtcNow);
            var s3StaticConfigPath = String.Format("{0}/{0}_{1:s}_config", ServiceName, DateTime.UtcNow);

            DeployStep.Initialize();

            DeployStep.RegisterStep("Replace Secret Placeholders", () => CopyAppFiles && !SkipSecrets, () =>
            {
                try
                {                    
                    var secretsHelper = SecretsHelper.GetSecretsHelper();
                    secretsHelper.ReplaceSecrets(WebAppFolderPath, CompilationMode);                    
                }
                catch (Exception ex)
                {
                    Console.WriteLine("Error: " + ex);
                    Console.WriteLine("Stack: " + ex.StackTrace);
                    throw;
                }
            });

            DeployStep.RegisterStep("Network Change Pause", () => PauseForNetworkChange, () =>
            {
                Console.WriteLine("Pausing for network change for deployment. Press enter when ready");
                Console.ReadLine();
            });

            DeployStep.RegisterStep("Predeploy Options", () => !artifactUploadOnly && !AutoCommit && !UseChef, () =>
            {

                if (PromptUser("Do you want to skip web, static and media site setup for this deploy? (This can speed up deploys of existing sites)", new[] { ConsoleKey.Y, ConsoleKey.N }, ConsoleKey.Y) == ConsoleKey.Y)
                {
                    SkipRpcCalls = true;
                }
                else
                {
                    SkipRpcCalls = false;
                }

            });


            DeployStep.RegisterStep("Creating Temp Folder", () => true, () =>
                {
                    DirectoryInfo tempFolder = new DirectoryInfo(TempFolderPath);
                    if (!tempFolder.Exists)
                    {
                        tempFolder.Create();
                    }
                });

            DeployStep.RegisterStep("Parsing Host File", () => true, () =>
                {
                    HostFile = HostFile.Parse();
                });

            DeployStep.RegisterStep("Validating Configuration", () => true, ValidateConfiguration);

            DeployStep.RegisterStep("Create Media Sites", () => !artifactUploadOnly && CreateMediaSites && !SkipRpcCalls, SetupMediaSites);

            if (WebAppZipPath == null)
            {
                DeployStep.RegisterStep("Zip Application Files", () => CopyAppFiles, () =>
                    {
                        WebAppZipPath = Path.Combine(TempFolderPath, "Application.zip");
                        if (File.Exists(WebAppZipPath))
                        {
                            File.Delete(WebAppZipPath);
                        }

                        Console.WriteLine("Creating zip from {0} to {1}", WebAppFolderPath, WebAppZipPath);
                        ZipHelper.CreateZip(WebAppFolderPath, WebAppZipPath);
                    });
            }

            string appZipUrl = null;
            DeployStep.RegisterStep("Upload artifact to S3", () => UseS3, () =>
            {
                Console.Write("Uploading app zip to S3... ");
                appZipUrl = S3Helper.SaveToS3(s3deployPath, WebAppZipPath);
                Console.WriteLine("Done");
            });

            DeployStep.RegisterStep("Copy and Unzip Application Files", () => !artifactUploadOnly && CopyAppFiles, () =>
            {
                if (UseS3)
                {
                    var failedNodes = new ConcurrentBag<string>();

                    Parallel.ForEach(WebNodeNames, node =>
                    {
                        try
                        {
                            using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                            {
                                Console.WriteLine("Downloading app zip on {0}...", node);
                                var tempPath = RemoteScriptHelper.GetRemoteTempFilePath(runspace, WebAppZipPath, ApplicationVersion);
                                RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"Invoke-WebRequest ""{0}"" -OutFile ""{1}""", appZipUrl, tempPath));

                                Console.WriteLine("Extracting app zip on {0}...", node);
                                string applicationPath = GetApplicationPath(node, true);
                                RemoteScriptHelper.EnsureFolderExists(runspace, applicationPath);
                                RemoteScriptHelper.Unzip(runspace, tempPath, GetApplicationPath(node, true));
                            }
                        }
                        catch (Exception ex)
                        {
                            failedNodes.Add(node);
                            Console.WriteLine("Failed on node " + node + ": " + ex.Message);
                            PrintExceptionDetails(ex);
                        }

                    });

                    if (failedNodes.Any())
                    {

                        Console.WriteLine("Failed on nodes: " + string.Join(", ", failedNodes));
                        var result = PromptUser("Do you want to stop the deploy?", new[] { ConsoleKey.Y, ConsoleKey.N }, ConsoleKey.N);
                        if (result == ConsoleKey.Y)
                        {
                            throw new Exception("Deploy cancelled!");
                        }

                        WebNodeNames = WebNodeNames.Where(p => !failedNodes.Contains(p)).ToArray();
                        Console.WriteLine("Deploy targets updated to: " + string.Join(", ", WebNodeNames));
                    }

                    Console.WriteLine("App files ready on all nodes.");
                }
                else
                {
                    foreach (var node in WebNodeNames)
                    {
                        Console.Write("Copying app zip file to " + node + "...");
                        ZipHelper.CopyZipToTemp(WebAppZipPath, node, ApplicationVersion);
                        Console.WriteLine("Done");

                        Console.Write("Unzipping app zip file to " + node + "...");
                        string applicationPath = GetApplicationPath(node);
                        DirectoryInfo di = new DirectoryInfo(applicationPath);
                        if (!di.Exists)
                        {
                            di.Create();
                        }

                        using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                        {
                            ZipHelper.UnzipRemotelyWithPowershell(runspace, PathHelper.GetRemoteTempFilePath(WebAppZipPath, node, ApplicationVersion, true), GetApplicationPath(node, true), node, UnzipExePath);
                        }

                        Console.WriteLine("Done");
                    }
                }
            });

            DeployStep.RegisterStep("Creating Application Web Sites", () => CreateWebSites && !SkipRpcCalls && !artifactUploadOnly, () =>
            {
                foreach (var node in WebNodeNames)
                {
                    SetupWebSite(node);
                }
            });

            if (StaticZipPath == null)
            {
                DeployStep.RegisterStep("Zip Static Files", () => CopyStaticFiles && !artifactUploadOnly, () =>
                {
                    StaticZipPath = Path.Combine(TempFolderPath, "Static.zip");
                    ZipHelper.CreateZip(StaticFolderPath, StaticZipPath);
                });
            }

            string staticZipUrl = null;
            string staticConfigUrl = null;
            DeployStep.RegisterStep("Upload static zip to S3", () => CopyStaticFiles && !artifactUploadOnly && UseS3, () =>
            {
                Console.Write("Uploading static zip to S3... ");
                staticZipUrl = S3Helper.SaveToS3(s3StaticPath, StaticZipPath);
                staticConfigUrl = S3Helper.SaveToS3(s3StaticConfigPath, StaticConfigPath);
                Console.WriteLine("Done");
            });

            DeployStep.RegisterStep("Copy and Unzip Static Files", () => CopyStaticFiles && !artifactUploadOnly, () =>
            {
                foreach (var node in WebNodeNames)
                {
                    Console.WriteLine("Copying static zip file to production...");
                    if (UseS3)
                    {
                        var staticZipPath = GetStaticPath(node, true);
                        using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                        {
                            var tempPath = RemoteScriptHelper.GetRemoteTempFilePath(runspace, StaticZipPath, ApplicationVersion);
                            RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"Invoke-WebRequest ""{0}"" -OutFile ""{1}""", staticZipUrl, tempPath));

                            Console.WriteLine("Unzipping static zip file to file server...");
                            RemoteScriptHelper.EnsureFolderExists(runspace, staticZipPath);
                            RemoteScriptHelper.Unzip(runspace, tempPath, staticZipPath);

                            Console.WriteLine("Copying static config file...");
                            var configPath = Path.Combine(staticZipPath, "web.config");
                            RemoteScriptHelper.ExecuteRemoteScript(runspace, string.Format(@"Invoke-WebRequest ""{0}"" -OutFile ""{1}""", staticConfigUrl, configPath));
                        }

                        if (CopyStaticToCurrent)
                        {
                            Console.WriteLine("Copying static content to current folder...");
                            string newStaticPath = Path.Combine(GetStaticPath(node, true), ApplicationVersion);
                            string currentStaticPath = Path.Combine(GetStaticPath(node, true), "current");
                            RunRemoteCommand(node, string.Format(@"xcopy /e /y /q /r ""{0}"" ""{1}\""", newStaticPath, currentStaticPath));
                        }
                    }
                    else
                    {
                        ZipHelper.CopyZipToTemp(StaticZipPath, node, ApplicationVersion);

                        Console.WriteLine("Unzipping static zip file to file server...");
                        DirectoryInfo di = new DirectoryInfo(GetStaticPath(node));
                        if (!di.Exists)
                        {
                            di.Create();
                        }


                        using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                        {
                            ZipHelper.UnzipRemotelyWithPowershell(runspace, PathHelper.GetRemoteTempFilePath(StaticZipPath, node, ApplicationVersion, true), GetStaticPath(node, true), node,
                                UnzipExePath);
                        }

                        if (CopyStaticToCurrent)
                        {
                            Console.WriteLine("Copying static content to current folder...");
                            string newStaticPath = Path.Combine(GetStaticPath(node, true), ApplicationVersion);
                            string currentStaticPath = Path.Combine(GetStaticPath(node, true), "current");
                            RunRemoteCommand(node, string.Format(@"xcopy /e /y /q /r ""{0}"" ""{1}\""", newStaticPath, currentStaticPath));
                        }

                        // Copy the static web.config
                        FileInfo fi = new FileInfo(StaticConfigPath);
                        string destinationPath = Path.Combine(di.FullName, "web.config");
                        //if (!File.Exists(destinationPath))
                        //{
                        fi.CopyTo(Path.Combine(di.FullName, "web.config"), true);
                        //}
                    }
                }
            });

            DeployStep.RegisterStep("Create Static Web Sites", () => CreateStaticSites && !SkipRpcCalls && !artifactUploadOnly, () =>
            {
                foreach (var node in WebNodeNames)
                {
                    SetupStaticSite(node);
                }
            });


            DeployStep.RegisterStep("Commit Deploy", () => Commit && !artifactUploadOnly, CommitDeploy);

            DeployStep.RegisterStep("Notify New Relic Deploy", () => (Commit && NotifyNewRelic && NewRelicAppName != null && !artifactUploadOnly), () =>
            {
                using (WebClient client = new WebClient())
                {
                    client.Headers.Add("x-api-key", NewRelicApiKey);

                    NameValueCollection vals = new NameValueCollection();
                    vals.Add("deployment[app_name]", NewRelicAppName);
                    vals.Add("deployment[revision]", ApplicationVersion);
                    vals.Add("deployment[user]", Environment.UserName);
                    Console.Write("Notifiying New Relic of deploy to " + NewRelicAppName);
                    try
                    {
                        client.UploadValues("https://rpm.newrelic.com/deployments.xml", vals);
                        Console.WriteLine("Done");
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine("Failed: " + ex.Message);
                        Console.WriteLine(ex.StackTrace);
                    }


                }
            });

            DeployStep.RegisterStep("Cleanup", () => CleanupAfterDeploy && !artifactUploadOnly && string.IsNullOrEmpty(RemoteUsername), () =>
                {
                    // Cleanup any application 
                    foreach (var webNodeName in WebNodeNames)
                    {
                        if (CreateWebSites)
                        {
                            DirectoryInfo di = new DirectoryInfo(GetApplicationPath(webNodeName));

                            // Get all folders
                            var oldFolders = di.GetDirectories().OrderByDescending(p => p.Name).Skip(10).ToArray();

                            Console.WriteLine("Deleting " + oldFolders.Count() + " old application folders...");

                            foreach (var folder in oldFolders)
                            {
                                string localPath = Path.Combine(GetApplicationPath(webNodeName, true), folder.Name);
                                Console.WriteLine("Deleteing " + folder.FullName + "...");
                                RunRemoteCommand(webNodeName, string.Format(@"cmd /c rmdir /s /q ""{0}""", localPath));
                            }
                        }

                        if (CreateStaticSites)
                        {
                            DirectoryInfo di = new DirectoryInfo(GetStaticPath(webNodeName));

                            // Get all folders
                            var oldFolders = di.GetDirectories().OrderByDescending(p => p.Name).Skip(10).ToArray();

                            Console.WriteLine("Deleting " + oldFolders.Count() + " old static folders...");

                            foreach (var folder in oldFolders)
                            {
                                string localPath = Path.Combine(GetStaticPath(webNodeName, true), folder.Name);
                                Console.WriteLine("Deleteing " + folder.FullName + "...");
                                RunRemoteCommand(webNodeName, string.Format(@"cmd /c rmdir /s /q ""{0}""", localPath));
                            }
                        }

                    }
                });


            if (UseS3)
            {
                DeployStep.RegisterStep("Flag As Current Build", () => true, () =>
                {
                    if (PromptUser("Flag as current build?", new[] { ConsoleKey.Y, ConsoleKey.N }, ConsoleKey.Y) == ConsoleKey.Y)
                    {
                        S3Helper.CopyWithinS3(s3deployPath, s3currentPath);
                    }
                });
            }

            DeployStep.RegisterStep("Completed", () => true, () =>
            {
                if (artifactUploadOnly)
                {
                    Console.WriteLine("Completed artifact upload-only deployment");
                }
                else
                {
                    Console.WriteLine("Completed deploy to: " + "http://" + SiteBindings.First() + ApplicationTestUrl);
                }
            });

            return DeployStep.RunAll();
        }

        private static void PrintExceptionDetails(Exception ex, bool isInner = false)
        {
            Console.WriteLine($"{(isInner ? "Inner Exception": "Exception")} {ex.GetType()}: ex.Message");
            Console.WriteLine(ex.StackTrace);

            if(ex.InnerException != null)
            {
                PrintExceptionDetails(ex);
            }
        }

        protected ConsoleKey PromptUser(string promptText, ConsoleKey[] options, ConsoleKey defaultChoice)
        {
            if (AutoCommit)
            {
                Console.WriteLine("Using default choice: " + defaultChoice);
                return defaultChoice;
            }

            return DeployUtils.PromptUser(promptText, options);
        }

        private void RunRemoteCommand(string nodeName, string command)
        {
            RemoteScriptHelper.ExecuteRemoteScript(nodeName, command);
        }

        private bool ValidateHost(string host)
        {
            IPHostEntry hostEntry = null;
            try
            {
                hostEntry = Dns.GetHostEntry(host);
            }
            catch (Exception ex)
            {
                return false;
            }

            return hostEntry != null && hostEntry.AddressList.Any(p => p.AddressFamily == AddressFamily.InterNetwork);
        }

        private void ValidateConfiguration()
        {

            if (!DisableBindingValidation)
            {
                for (int i = 0; i < SiteBindings.Length; i++)
                {

                    Console.Write("Validating site binding '" + SiteBindings[i] + "'...");

                    if (SiteBindings[i].Equals("Default", StringComparison.InvariantCultureIgnoreCase))
                    {
                        SiteBindings[i] = "";
                    }
                    else if (!ValidateHost(SiteBindings[i]))
                    {
                        Console.WriteLine("Failed");
                        throw new Exception("Unable to resolve DNS for host: " + SiteBindings[i]);
                    }
                    Console.WriteLine("Done");

                }
            }

            // Ensure that each node can be connected to
            if (!UseS3)
            {
                foreach (var name in WebNodeNames)
                {
                    var unc = @"\\" + name + @"\c$\";
                    Console.Write("Validating web node access at: " + unc + "...");

                    if (!ValidateFolder(unc))
                    {
                        Console.WriteLine("Failed");
                        throw new Exception("Invalid folder: " + unc);
                    }

                    Console.WriteLine(" Passed");
                }
            }

            if (CopyStaticFiles)
            {
                Console.Write("Validating static host '" + StaticSiteBinding + "'...");

                if (!ValidateHost(StaticSiteBinding))
                {
                    Console.WriteLine("Failed");
                    throw new Exception("Invalid static host: " + StaticSiteBinding);
                }
                else
                {
                    Console.WriteLine("Passed");
                }

                Console.Write("Validating static zip file at: " + StaticZipPath + "...");
                if (StaticZipPath != null)
                {
                    if (!ValidateFile(StaticZipPath))
                    {
                        Console.WriteLine(" Failed");
                        throw new Exception("Invalid static zip path: " + StaticZipPath);
                    }
                }
                else
                {
                    if (!ValidateFolder(StaticFolderPath))
                    {
                        Console.WriteLine(" Failed");
                        throw new Exception("Invalid static folder path: " + StaticFolderPath);
                    }
                }

                if (!ValidateFile(StaticConfigPath))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Invalid static file config: " + StaticConfigPath);
                }

                Console.WriteLine(" Passed");
            }

            if (CopyAppFiles)
            {
                Console.Write("Validating web app zip file at: " + WebAppZipPath + "...");
                if (WebAppZipPath != null)
                {
                    if (!ValidateFile(WebAppZipPath))
                    {
                        Console.WriteLine(" Failed");
                        throw new Exception("Invalid file: " + WebAppZipPath);
                    }
                }
                else
                {
                    if (!ValidateFolder(WebAppFolderPath))
                    {
                        Console.WriteLine(" Failed");
                        throw new Exception("Invalid file: " + WebAppZipPath);
                    }
                }

                Console.WriteLine(" Passed");
            }

            if (MediaNodeNames == null || !MediaNodeNames.Any(p => !string.IsNullOrEmpty(p)))
            {
                MediaNodeNames = _defaultMediaServerNames;
            }

            if (string.IsNullOrEmpty(MediaApplicationPath))
            {
                MediaApplicationPath = DefaultMediaApplicationPath;
            }

            if (string.IsNullOrEmpty(ManagedRuntimeVersion))
            {
                ManagedRuntimeVersion = _defaultManagedRuntimeVersion;
            }
            else if (ManagedRuntimeVersion == "None")
            {
                ManagedRuntimeVersion = "";
            }

            if (String.IsNullOrEmpty(ApplicationTestUrl))
            {
                ApplicationTestUrl = _defaultApplicationTestUrl;
            }

            if (string.IsNullOrEmpty(UnzipExePath))
            {
                string currentDirectory = Directory.GetCurrentDirectory();
                UnzipExePath = Path.Combine(currentDirectory, "unzip.exe");
                if (!ValidateFile(UnzipExePath))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Invalid unzeip exe path: " + UnzipExePath);
                }
            }

            if (CreateMediaSites)
            {
                Console.Write("Validating static host '" + MediaSiteBinding + "'...");

                if (!ValidateHost(MediaSiteBinding))
                {
                    Console.WriteLine("Failed");
                    throw new Exception("Invalid media host: " + MediaSiteBinding);
                }
                else
                {
                    Console.WriteLine("Passed");
                }

                foreach (var mediaNode in MediaNodeNames)
                {
                    string unc = @"\\" + mediaNode + @"\c$\";
                    Console.Write("Validating media node access at: " + unc + "...");

                    if (!ValidateFolder(unc))
                    {
                        Console.WriteLine(" Failed");
                        throw new Exception("Invalid media share: " + unc);
                    }
                    Console.WriteLine(" Passed");
                }

                Console.Write("Validating media share access at: " + MediaApplicationPath + "...");
                if (!ValidateFolder(MediaApplicationPath))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Invalid media share: " + MediaApplicationPath);
                }
                Console.WriteLine(" Passed");
            }

            if (SendNotificationEmail)
            {
                Console.Write("Validating notification email settings...");
                if (string.IsNullOrEmpty(NotificationEmailBody))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Missing NotificationEmailBody");
                }

                if (string.IsNullOrEmpty(NotificationEmailRecipient))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Missing NotificationEmailRecipient");
                }

                if (string.IsNullOrEmpty(NotificationEmailSender))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Missing NotificationEmailSender");
                }

                if (string.IsNullOrEmpty(NotificationEmailSubject))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Missing NotificationEmailSubject");
                }

                if (string.IsNullOrEmpty(SmtpServerHost))
                {
                    Console.WriteLine(" Failed");
                    throw new Exception("Missing SmtpServerHost");
                }
                Console.WriteLine(" Passed");
            }
        }

        private static bool ValidateFolder(string folderPath)
        {
            try
            {
                var di = new DirectoryInfo(folderPath);
                if (!di.Exists)
                {
                    Console.WriteLine("Unable to open folder at: " + folderPath);
                    return false;
                }
            }
            catch (Exception ex)
            {

                Console.WriteLine("Unable to open folder share at: " + folderPath + ". Exception: " + ex.Message);
                return false;
            }

            return true;
        }

        private static bool ValidateFile(string filePath)
        {
            if (string.IsNullOrEmpty(filePath))
            {
                return false;
            }

            try
            {
                var fi = new FileInfo(filePath);
                if (!fi.Exists)
                {
                    Console.WriteLine("Unable to open file at: " + filePath);
                    return false;
                }
            }
            catch (Exception ex)
            {

                Console.WriteLine("Unable to open folder share at: " + filePath + ". Exception: " + ex.Message);
                return false;
            }

            return true;
        }

        public string PhysicalApplicationPath
        {
            get
            {
                return UseCurrentFolder ?
                        @"c:\inetpub\" + (WebAppName ?? ApplicationName) + @"\Current" :
                    (!string.IsNullOrEmpty(ApplicationVersion) && !VersioningDisabled) ?
                        @"c:\inetpub\" + (WebAppName ?? ApplicationName) + @"\" + ApplicationVersion
                      : @"c:\inetpub\" + (WebAppName ?? ApplicationName);
            }
        }

        public string DefaultMediaApplicationPath
        {
            get { return string.Format(@"\\{0}\{1}\media", FileServerName, (MediaAppName ?? ApplicationName)); }
        }

        public string StaticApplicationPath
        {
            get { return string.Format(@"c:\inetpub\{0}Static", (StaticAppName ?? ApplicationName)); }
        }

        private void SetupMediaSites()
        {
            foreach (var node in MediaNodeNames)
            {
                Console.WriteLine("Setting up media site on " + node + "...");
                WebAdminHelper.CreateWebSite(node, (MediaAppName ?? ApplicationName) + "Media", MediaApplicationPath, new[] { MediaSiteBinding }, false);
            }
        }

        private void SetupStaticSite(string nodeName)
        {
            Console.WriteLine("Setting up static site on " + nodeName + "...");
            using (var runspace = RemoteScriptHelper.CreateRunspace(nodeName))
            {
                RemoteScriptHelper.CreateOrUpdateWebSite(runspace, (StaticAppName ?? ApplicationName) + "Static", StaticApplicationPath, new[] { StaticSiteBinding }, !UseCurrentFolder && !VersioningDisabled);
            }
        }

        private void SetupWebSite(string nodeName)
        {
            Console.WriteLine("Setting up web site on: " + nodeName + "...");
            using (var runspace = RemoteScriptHelper.CreateRunspace(nodeName))
            {
                RemoteScriptHelper.CreateOrUpdateWebSite(runspace, WebAppName ?? ApplicationName, PhysicalApplicationPath, SiteBindings, !UseCurrentFolder && !VersioningDisabled);
            }
        }


        private void FastCommitDeploy()
        {
            var counter = 0;

            Console.WriteLine("Fast Commit Mode Activated!");


            foreach (var node in WebNodeNames)
            {
                Console.WriteLine("Commiting to " + node + " (" + (++counter) + " of " + WebNodeNames.Length + ")");

                if (UseChef)
                {
                    SensuHelper.SilenceHealthCheckAlerts(node, DeployUtils.GetChefEnvironments(CompilationMode));
                }

                using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                {
                    Console.WriteLine("Stopping app pool '" + ApplicationName + "' ...");
                    RemoteScriptHelper.StopAppPool(runspace, ApplicationName);

                    if (UseCurrentFolder)
                    {
                        var newFolderPath = GetVersionApplicationPath(node, true);
                        var currentFolderPath = GetSuffixApplicationPath(node, true, "Current");

                        if (!UpdateCurrentFolder)
                        {
                            // Rename the current 'Current' folder to a timestamp
                            var script = "Rename-Item -path " + currentFolderPath + " " + DateTime.UtcNow.Ticks;
                            Console.WriteLine(script);
                            RemoteScriptHelper.ExecuteRemoteScript(runspace, script);

                            // Rename the versioned folder to 'Current'
                            script = "Rename-Item -path " + newFolderPath + " Current";
                            Console.WriteLine(script);
                            RemoteScriptHelper.ExecuteRemoteScript(runspace, script);
                        }
                        else
                        {
                            Console.WriteLine("Updating current folder rather than replacing it.");
                            RemoteScriptHelper.ExecuteRemoteScript(runspace,
                                String.Format(@"Copy-Item -Recurse -Force ""{0}\*"" -Destination ""{1}""", newFolderPath,
                                    currentFolderPath));

                            Console.WriteLine("Deleting deploy folder.");
                            RemoteScriptHelper.ExecuteRemoteScript(runspace,
                                String.Format(@"Remove-Item -Recurse -Force ""{0}""", newFolderPath));
                        }
                    }
                    else
                    {

                        Console.Write("Updating site path...");
                        RemoteScriptHelper.CreateOrUpdateWebSite(runspace, WebAppName ?? ApplicationName, PhysicalApplicationPath, SiteBindings, !VersioningDisabled);
                        Console.WriteLine("Done!");
                    }

                    Console.WriteLine("Waiting 1 second to start app pool...");
                    Thread.Sleep(1000);

                    RemoteScriptHelper.StartAppPool(runspace, ApplicationName);
                    Console.WriteLine("Committed to " + node);

                    // Try to load the root URL
                    TestUrl(node, ApplicationTestUrl, true);

                }
            }
        }

        private const string HealthCheckAppPoolName = "HealthMonitor";
        private const string HealthCheckPublishPath = @"C:\Projects\DeployTools\Publish\HealthMonitor";

        public static void DeployHealthMonitor(string nodeName)
        {
            Console.WriteLine("Validating health check app... ");
            // Check if the file exists
            var remoteAppPath = GetApplicationPath(nodeName, HealthCheckAppPoolName, false);
            var localAppPath = GetApplicationPath(nodeName, HealthCheckAppPoolName, true);

            if (Directory.Exists(remoteAppPath) && Directory.Exists(Path.Combine(remoteAppPath, "bin")))
            {
                Console.WriteLine("Health check exists.");
                return;
            }

            Console.WriteLine("Installing health check...");

            var zipPath = Path.Combine(@"C:\Windows\Temp\Deploys\HealthCheck", "HealthCheck.zip");
            ZipHelper.CreateZip(HealthCheckPublishPath, zipPath);

            Console.Write("Copying app zip file to " + nodeName + "...");
            ZipHelper.CopyZipToTemp(zipPath, nodeName, "HealthCheck");
            Console.WriteLine("Done");

            Console.Write("Unzipping app zip file to " + nodeName + "...");

            var di = new DirectoryInfo(remoteAppPath);

            if (!di.Exists)
            {
                di.Create();
            }

            using (var runspace = RemoteScriptHelper.CreateRunspace(nodeName))
            {
                var source = PathHelper.GetRemoteTempFilePath(zipPath, nodeName, "HealthCheck", true);
                Console.Write("Unzipping '" + source + "' to '" + localAppPath + "'... ");
                ZipHelper.UnzipRemotelyWithPowershell(runspace, source, localAppPath, nodeName);
                Console.WriteLine("Done");
            }

        }

        private void NewCommitDeploy()
        {
            var counter = 0;

            Console.WriteLine("New Commit Mode Activated!");

            var isReleaseCandidateNode = true;
            foreach (var node in WebNodeNames)
            {

                if (ConfirmCommitToSecondaries)
                {
                    PromptUser("Press enter to commit to " + node, new[] { ConsoleKey.Enter }, ConsoleKey.Enter);
                }
                else if (NodeRolloutDelay > 0 && !isReleaseCandidateNode)
                {
                    Console.WriteLine("Waiting " + NodeRolloutDelay + " seconds before next node rollout...");
                    Thread.Sleep(TimeSpan.FromSeconds(NodeRolloutDelay));
                }

                if (!SkipHealthMonitorDeploy)
                {
                    DeployHealthMonitor(node);
                }

                Console.WriteLine("Commiting to " + node + " (" + (++counter) + " of " + WebNodeNames.Length + ")");

                if (UseChef)
                {
                    SensuHelper.SilenceHealthCheckAlerts(node, DeployUtils.GetChefEnvironments(CompilationMode));
                }

                using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                {

                    // First let the service know it's about to shut down
                    if (!string.IsNullOrWhiteSpace(ServiceShutdownUrl))
                    {
                        Console.WriteLine("Stopping health check app pool '" + HealthCheckAppPoolName + "' ...");
                        RemoteScriptHelper.StopAppPool(runspace, HealthCheckAppPoolName);

                        Console.WriteLine("Waiting 5 seconds for load balancer to remove node from rotation...");
                        Thread.Sleep(5000);

                        Console.WriteLine("Triggering service shutdown...");

                        try
                        {
                            var secretsHelper = SecretsHelper.GetSecretsHelper();
                            var shutdownKey = secretsHelper.GetSecretValue("api-keys/service-shutdown", CompilationMode);                            
                            TestUrl(node, string.Format(ServiceShutdownUrl, Uri.EscapeDataString(shutdownKey)), true);
                            Console.WriteLine("Done!");
                        }
                        catch (Exception ex)
                        {
                            Console.WriteLine("Failed to trigger service shutdown: " + ex.Message);
                        }

                        Console.WriteLine("Press enter to continue the deploy (wait until connections are drained)...");
                        Console.ReadLine();
                    }
                    else
                    {
                        Console.WriteLine("Stopping health check app pool '" + HealthCheckAppPoolName + "' ...");
                        RemoteScriptHelper.StopAppPool(runspace, HealthCheckAppPoolName);
                    }


                    Console.Write("Waiting " + LoadBalancerWait.TotalSeconds.ToString("##,##0") + " seconds for load balancer...");
                    Thread.Sleep(LoadBalancerWait);
                    Console.WriteLine("Done");

                    Console.WriteLine("Stopping app pool '" + ApplicationName + "' ...");
                    RemoteScriptHelper.StopAppPool(runspace, ApplicationName);

                    if (UseCurrentFolder)
                    {
                        var newFolderPath = GetVersionApplicationPath(node, true);
                        var currentFolderPath = GetSuffixApplicationPath(node, true, "Current");

                        if (!UpdateCurrentFolder)
                        {
                            // Rename the current 'Current' folder to a timestamp
                            var script = "Rename-Item -path " + currentFolderPath + " " + DateTime.UtcNow.Ticks;
                            Console.WriteLine(script);
                            RemoteScriptHelper.ExecuteRemoteScript(runspace, script);

                            // Rename the versioned folder to 'Current'
                            script = "Rename-Item -path " + newFolderPath + " Current";
                            Console.WriteLine(script);
                            RemoteScriptHelper.ExecuteRemoteScript(runspace, script);
                        }
                        else
                        {
                            Console.WriteLine("Updating current folder rather than replacing it.");
                            RemoteScriptHelper.ExecuteRemoteScript(runspace,
                                String.Format(@"Copy-Item -Recurse -Force ""{0}\*"" -Destination ""{1}""", newFolderPath,
                                    currentFolderPath));

                            Console.WriteLine("Deleting deploy folder.");
                            RemoteScriptHelper.ExecuteRemoteScript(runspace,
                                String.Format(@"Remove-Item -Recurse -Force ""{0}""", newFolderPath));
                        }
                    }
                    else
                    {
                        Console.Write("Updating site path...");
                        RemoteScriptHelper.CreateOrUpdateWebSite(runspace, WebAppName ?? ApplicationName, PhysicalApplicationPath, SiteBindings, !VersioningDisabled);
                        Console.WriteLine("Done!");
                    }

                    Console.WriteLine("Waiting 5 seconds to start app pool...");
                    Thread.Sleep(5000);

                    RemoteScriptHelper.StartAppPool(runspace, ApplicationName);
                    Console.WriteLine("Committed to " + node);

                    // Try to load the root URL
                    if (!TestUrl(node, ApplicationTestUrl, true))
                    {
                        Console.WriteLine("Press any key to cancel the deploy...");
                        Console.ReadKey(true);
                        throw new Exception("Deploy cancelled, due to page error");
                    }

                    // Prompt for testing
                    if (isReleaseCandidateNode)
                    {

                        isReleaseCandidateNode = false;

                        IPAddress hostIP;
                        if(!IPAddress.TryParse(node, out hostIP))
                        {
                            var host = Dns.GetHostEntry(node);
                            if (host == null || host.AddressList.All(p => p.AddressFamily != AddressFamily.InterNetwork))
                            {
                                Console.WriteLine("Unable to resolve IP Address for " + node + ". Deploy cancelled!");
                                return;
                            }

                            hostIP = host.AddressList.FirstOrDefault(p => p.AddressFamily == AddressFamily.InterNetwork);
                            if (hostIP == null)
                            {
                                Console.WriteLine("Unable to continue deploy, the DNS could not be found for this node: " + node);
                                return;
                            }
                        }

                        Console.Write("Updating your host file...");
                        HostFile.AddOrReplaceEntry(SiteBindings.First(), hostIP.ToString());
                        HostFile.AddOrReplaceEntry(StaticSiteBinding, hostIP.ToString());
                        HostFile.Save();
                        Console.WriteLine("Done");

                        Console.ForegroundColor = ConsoleColor.White;
                        Console.WriteLine("Launching the release candidate of this site in your browser. Please confirm it is functioning properly, before committing.");
                        Console.ResetColor();

                        LaunchCleanBrowser("https://" + SiteBindings.First() + ApplicationTestUrl);

                        if (PromptUser("Are you ready to commit this build to " + node + "?", new[] { ConsoleKey.Y, ConsoleKey.N }, ConsoleKey.Y) == ConsoleKey.N)
                        {
                            throw new Exception("Deploy cancelled, by user.");
                        }


                        Console.Write("Starting health check app pool... ");
                        RemoteScriptHelper.StartAppPool(runspace, HealthCheckAppPoolName);
                        Console.WriteLine("Done");

                        HostFile.RemoveEntry(SiteBindings.First());
                        HostFile.RemoveEntry(StaticSiteBinding);
                        HostFile.Save();

                        Console.ForegroundColor = ConsoleColor.White;
                        Console.WriteLine("Launching load balanced version of this site. This is what the public is currently seeing.");
                        Console.ResetColor();

                        LaunchCleanBrowser("https://" + SiteBindings.First());

                    }
                    else
                    {
                        if (ConfirmCommitToSecondaries)
                        {
                            PromptUser("Press ENTER to commit this build to " + node + "?", new[] { ConsoleKey.Enter }, ConsoleKey.Enter);
                        }

                        if (!TestUrl(node, ApplicationTestUrl, true))
                        {
                            Console.WriteLine("The default web page at " + SiteBindings.First() + " failed to load on " + node + ".");

                            if (PromptUser("Do you wish to cancel the deploy?", new[] { ConsoleKey.Y, ConsoleKey.N }, ConsoleKey.N) == ConsoleKey.Y)
                            {
                                throw new Exception("Deploy cancelled, due to page error");
                            }
                        }

                        Console.WriteLine("Starting app pool '" + HealthCheckAppPoolName + "' ...");
                        RemoteScriptHelper.StartAppPool(runspace, HealthCheckAppPoolName);

                    }
                }
            }
        }

        private void CommitDeploy()
        {
            Console.WriteLine("Autocommit is {0}", AutoCommit);
            if (AutoCommit)
            {
                FastCommitDeploy();
                return;
            }


            PromptUser("Press enter to begin the commit process (run any database scripts before doing this).", new[] { ConsoleKey.Enter }, ConsoleKey.Enter);

            NewCommitDeploy();
            return;


            bool isReleaseCandidateNode = true;
            int counter = 0;
            foreach (var node in WebNodeNames)
            {
                if (!isReleaseCandidateNode && !AutoCommit)
                {
                    if (ConfirmCommitToSecondaries)
                    {
                        PromptUser("Press enter to commit to " + node, new[] { ConsoleKey.Enter }, ConsoleKey.Enter);
                    }
                    else if (NodeRolloutDelay > 0)
                    {
                        Console.WriteLine("Waiting " + NodeRolloutDelay + " seconds before next node rollout...");
                        Thread.Sleep(TimeSpan.FromSeconds(NodeRolloutDelay));
                    }
                }

                Console.WriteLine("Commiting to node " + (++counter).ToString() + " of " + WebNodeNames.Count());
                using (ServerManager serverManager = ServerManager.OpenRemote(node))
                {
                    IPHostEntry host = Dns.GetHostEntry(node);
                    if (host == null || !host.AddressList.Any(p => p.AddressFamily == AddressFamily.InterNetwork))
                    {
                        Console.WriteLine("Unable to resolve IP Address for " + node + ". Deploy cancelled!");
                        return;
                    }

                    var address = host.AddressList.FirstOrDefault(p => p.AddressFamily == AddressFamily.InterNetwork);

                    ApplicationPool healthCheckAppPool = null;
                    // Sleep
                    if (!AutoCommit)
                    {
                        Console.Write("Getting health check app pool...");
                        healthCheckAppPool = serverManager.ApplicationPools.FirstOrDefault(p => p.Name == "HealthMonitor");

                        if (healthCheckAppPool == null)
                        {
                            Console.WriteLine("Unable to find  " + node + ". Deploy cancelled!");
                            throw new Exception("Deploy cancelled, due to missing health check app pool");
                        }
                        Console.WriteLine("Done");

                        if (healthCheckAppPool.State != ObjectState.Stopped)
                        {
                            Console.Write("Stopping health monitor...");
                            healthCheckAppPool.Stop();
                            Console.WriteLine("Done");
                        }

                        Console.Write("Waiting " + LoadBalancerWait.TotalSeconds.ToString("##,##0") + " seconds for load balancer...");
                        Thread.Sleep(LoadBalancerWait);
                        Console.WriteLine("Done");

                    }

                    var site = serverManager.Sites.FirstOrDefault(p => p.Name == (WebAppName ?? ApplicationName));
                    var appPool = serverManager.ApplicationPools.FirstOrDefault(p => p.Name == site.Applications.FirstOrDefault().ApplicationPoolName);

                    // Update the physical path
                    if (!SkipUpdateSitePath)
                    {
                        if (appPool.State != ObjectState.Stopped)
                        {
                            Console.Write("Stopping app pool...");
                            appPool.Stop();
                            int attempts = 0;
                            while (appPool.State != ObjectState.Stopped)
                            {
                                Thread.Sleep(1000);
                                if (++attempts > 120)
                                {
                                    Console.WriteLine("Failed to stop app pool after 2 minutes!");
                                    throw new Exception("Deploy cancelled, due to a malfunctioning app pool");
                                }
                            }
                            Console.WriteLine("Done");
                        }

                        // Delete the contents of the ASP.Net Temp folders before updating the site path.
                        Console.WriteLine("Deleting ASP.Net Temp Files...");

                        if (Directory.Exists(string.Format(@"\\{0}\{1}", node, AspNetTempFolderPath.Replace(@"C:\", @"c$\"))))
                        {
                            RunRemoteCommand(node, string.Format(@"cmd /c rmdir /s /q ""{0}""", AspNetTempFolderPath));
                        }
                        if (Directory.Exists(string.Format(@"\\{0}\{1}", node, SixtyFourBitAspNetTempFolderPath.Replace(@"C:\", @"c$\"))))
                        {
                            RunRemoteCommand(node, string.Format(@"cmd /c rmdir /s /q ""{0}""", SixtyFourBitAspNetTempFolderPath));
                        }
                        Console.WriteLine("Done");

                        if (UseCurrentFolder)
                        {
                            using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                            {

                                // Rename the current 'Current' folder to a timestamp
                                Console.WriteLine("Backing up 'Current' folder...");
                                RemoteScriptHelper.RenameFolder(runspace, GetSuffixApplicationPath(node, true, "Current"), GetSuffixApplicationPath(node, true, DateTime.UtcNow.Ticks.ToString()));

                                // Rename the versioned folder to 'Current'
                                Console.WriteLine("Renaming deployed folder...");
                                RemoteScriptHelper.RenameFolder(runspace, GetVersionApplicationPath(node, true), GetSuffixApplicationPath(node, true, "Current"));
                            }

                            Console.Write("Updating site path (Current: " + site.Applications.FirstOrDefault().VirtualDirectories["/"].PhysicalPath + ")");
                            site.Applications.FirstOrDefault().VirtualDirectories["/"].PhysicalPath = PhysicalApplicationPath;
                            serverManager.CommitChanges();
                            Console.WriteLine("Done");
                        }
                        else
                        {
                            Console.Write("Updating site path...");
                            site.Applications.FirstOrDefault().VirtualDirectories["/"].PhysicalPath = PhysicalApplicationPath;
                            serverManager.CommitChanges();
                            Console.WriteLine("Done");
                        }


                        Console.Write("Starting app pool...");
                        appPool.Start();
                        Console.WriteLine("Done");
                    }



                    if (SkipUpdateSitePath && !SkipApplicationPoolRecycle)
                    {
                        Console.Write("Recycling app pool...");

                        appPool.Recycle();
                        Console.WriteLine("Done");
                    }


                    if (!AutoCommit)
                    {
                        // Try to load the root URL
                        if (!TestUrl(node, ApplicationTestUrl, true))
                        {
                            Console.WriteLine("The default web page at " + SiteBindings.First() + " failed to load on " + node + ". The deploy has been cancelled!");
                            throw new Exception("Deploy cancelled, due to page error");
                        }

                        // Prompt for testing
                        if (isReleaseCandidateNode)
                        {


                            isReleaseCandidateNode = false;

                            Console.Write("Updating your host file...");
                            HostFile.AddOrReplaceEntry(SiteBindings.First(), address.ToString());
                            HostFile.AddOrReplaceEntry(StaticSiteBinding, address.ToString());
                            HostFile.Save();
                            Console.WriteLine("Done");

                            Console.ForegroundColor = ConsoleColor.White;
                            Console.WriteLine("Launching the release candidate of this site in your browser. Please confirm it is functioning properly, before committing.");
                            Console.ResetColor();

                            LaunchCleanBrowser("http://" + SiteBindings.First() + ApplicationTestUrl);

                            if (PromptUser("Are you ready to commit this build to " + node + "?", new[] { ConsoleKey.Y, ConsoleKey.N }, ConsoleKey.Y) == ConsoleKey.N)
                            {
                                throw new Exception("Deploy cancelled, by user.");
                            }


                            Console.Write("Starting health check app pool... ");
                            healthCheckAppPool.Start();
                            Console.WriteLine("Done");

                            HostFile.RemoveEntry(SiteBindings.First());
                            HostFile.RemoveEntry(StaticSiteBinding);
                            HostFile.Save();

                            Console.ForegroundColor = ConsoleColor.White;
                            Console.WriteLine("Launching load balanced version of this site. This is what the public is currently seeing.");
                            Console.ResetColor();

                            LaunchCleanBrowser("http://" + SiteBindings.First());

                        }
                        else
                        {
                            if (ConfirmCommitToSecondaries)
                            {
                                PromptUser("Press ENTER to commit this build to " + node + "?", new[] { ConsoleKey.Enter }, ConsoleKey.Enter);
                            }

                            if (!TestUrl(node, ApplicationTestUrl, true))
                            {
                                Console.WriteLine("The default web page at " + SiteBindings.First() + " failed to load on " + node + ".");
                                if (PromptUser("Do you wish to cancel the deploy?", new ConsoleKey[] { ConsoleKey.Y, ConsoleKey.N }, ConsoleKey.N) == ConsoleKey.Y)
                                {
                                    throw new Exception("Deploy cancelled, due to page error");
                                }
                                else
                                {
                                    continue;
                                }
                            }

                            PrimeCaches(node);
                            healthCheckAppPool.Start();
                        }
                    }
                    Console.WriteLine("-------------------------------------------------------------------------------------");
                }
            }

            Console.WriteLine("This deploy is now live!");
        }


        public void LaunchCleanBrowser(string url)
        {
            try
            {
                var running = Process.GetProcessesByName("firefox");
                foreach (var process in running)
                {
                    process.Kill();
                    process.WaitForExit();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("We were unable to shut down Firefox. This may result in stale DNS lookups.");
            }


            Process.Start("firefox.exe", url);
        }

        private void PrimeCaches(string nodeName)
        {

        }

        private bool TestUrl(string nodeName, string relativeUrlPath, bool isInitialRequest)
        {
            if (!relativeUrlPath.StartsWith("/"))
            {
                relativeUrlPath = "/" + relativeUrlPath;
            }

            string host;

            if (!_siteBindingsByNode.TryGetValue(nodeName, out host))
            {
                host = SiteBindings.First();
            }
            var useSsl = !UseInsecureTestUrl;
            var testUrl = (useSsl ? "https://" : "http://") + nodeName + relativeUrlPath;

            Console.WriteLine("Loading web page from " + testUrl + " (Host: " + host + ")...");

            var timeout = (isInitialRequest && PageTestTimeoutSeconds > 0) ? PageTestTimeoutSeconds * 1000 : 60000;

            var webRequest = (HttpWebRequest)WebRequest.Create(testUrl);
            webRequest.UserAgent = "googlebot (Curse Health Monitor)";
            webRequest.Proxy = null;
            webRequest.Host = SiteBindings.First();
            webRequest.Timeout = timeout; // 1 Minute
            webRequest.AllowAutoRedirect = true;

            string result = null;
            HttpWebResponse response = null;
            try
            {
                response = (HttpWebResponse)webRequest.GetResponse();
                using (var reader = new StreamReader(response.GetResponseStream()))
                {
                    result = reader.ReadToEnd();
                }

            }
            catch (WebException ex)
            {

                Console.WriteLine("Failed to load page at " + testUrl + ". Status: " + ex.Status);
                return false;
            }
            catch (Exception ex)
            {
                Console.WriteLine("Failed to load page at " + testUrl + ". Error: " + ex.Message);
                return false;
            }

            if (result.Length == 0)
            {
                Console.WriteLine("Empty Response");
                return false;
            }

            Console.WriteLine("Passed! Response status: " + response.StatusCode + ". Response size: " + result.Length + " bytes");
            Console.WriteLine(result);
            return true;
        }


        public IBuildEngine BuildEngine { get; set; }
        public ITaskHost HostObject { get; set; }

        [Required]
        public string ApplicationName { get; set; }

        public string MediaAppName { get; set; }

        public string StaticAppName { get; set; }
        public string WebAppName { get; set; }

        private string _defaultApplicationTestUrl = "/";

        public string ApplicationTestUrl { get; set; }

        [Required]
        public string ApplicationVersion { get; set; }

        [Required]
        public bool VersioningDisabled { get; set; }

        private string _defaultManagedRuntimeVersion = "v4.0";
        public string ManagedRuntimeVersion { get; set; }

        public bool SkipApplicationPoolRecycle = true;
        public bool SkipUpdateSitePath = false;

        [Required]
        public string[] WebNodeNames { get; set; }

        [Required]
        public string FileServerName { get; set; }


        public string WebAppZipPath { get; set; }

        [Required]
        public string WebAppFolderPath { get; set; }

        public string StaticZipPath { get; set; }

        [Required]
        public string StaticFolderPath { get; set; }

        [Required]
        public string StaticConfigPath { get; set; }

        [Required]
        public string UnzipExePath { get; set; }

        [Required]
        public string[] SiteBindings { get; set; }

        [Required]
        public bool CopyAppFiles { get; set; }

        [Required]
        public bool CopyStaticFiles { get; set; }

        [Required]
        public bool CreateWebSites { get; set; }

        [Required]
        public bool CreateMediaSites { get; set; }

        [Required]
        public string MediaSiteBinding { get; set; }

        public string MediaApplicationPath { get; set; }

        private string[] _defaultMediaServerNames = new[] { "media01a-live.curse.us", "media01b-live.curse.us", "media01c-live.curse.us" };
        public string[] MediaNodeNames { get; set; }

        [Required]
        public bool CreateStaticSites { get; set; }

        public bool CopyStaticToCurrent { get; set; }

        [Required]
        public string StaticSiteBinding { get; set; }

        [Required]
        public bool CleanupAfterDeploy { get; set; }

        [Required]
        public bool SendNotificationEmail { get; set; }

        public string NotificationEmailRecipient { get; set; }

        public string NotificationEmailSender { get; set; }

        public string NotificationEmailSubject { get; set; }

        public string NotificationEmailBody { get; set; }

        public string NotificationEmailSuccessBody { get; set; }

        public string SmtpServerHost { get; set; }
        public string SmtpServerUsername { get; set; }
        public string SmtpServerPassword { get; set; }

        public bool NotifyNewRelic { get; set; }
        public string NewRelicApiKey = "113588c977a0f88ff09b4d8bb2f2577479f2385d39b5f45";
        public string NewRelicAppName { get; set; }

        public bool Commit = true;

        public bool AutoCommit { get; set; }
        public bool ConfirmCommitToSecondaries { get; set; }

        public bool DisableBindingValidation { get; set; }

        public int PageTestTimeoutSeconds { get; set; }

        public int NodeRolloutDelay { get; set; }

        public int LoadBalancerWaitSeconds { get; set; }

        public bool ServerFilteringPrompt { get; set; }

        [Required]
        public bool UseCurrentFolder { get; set; }

        public bool UpdateCurrentFolder { get; set; }

        public bool UseChef { get; set; }

        public bool UseS3 { get; set; }

        public string CompilationMode { get; set; }

        public string ServiceName { get; set; }

        public string ServiceShutdownUrl { get; set; }

        public bool SkipSecrets { get; set; }

        public string RemoteUsername { get; set; }

        public string RemotePasswordKey { get; set; }

        public bool UseInsecureTestUrl { get; set; }

        public bool PauseForNetworkChange { get; set; }

        public bool SkipHealthMonitorDeploy { get; set; }

        private TimeSpan LoadBalancerWait
        {
            get
            {
                if (LoadBalancerWaitSeconds > 0)
                {
                    return TimeSpan.FromSeconds(LoadBalancerWaitSeconds);
                }

                return TimeSpan.FromSeconds(10);
            }
        }

        private Dictionary<string, string> _siteBindingsByNode = new Dictionary<string, string>();
    }
}
