﻿using System.Data.SqlClient;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Management.Automation;
using System.Security;
using Microsoft.Build.Framework;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Management.Automation.Runspaces;

namespace Curse.MSBuild.Deployment
{
    public class WindowsServiceDeploy : ITask
    {
        public IBuildEngine BuildEngine { get; set; }
        public ITaskHost HostObject { get; set; }        

        public string DestinationDriveLetter
        {
            get;
            set;
        }

        public string DestinationFolderPath
        {
            get;
            set;
        }

        public string TempFolderPath
        {
            get;
            set;
        }

        private string GetApplicationPath(string node, bool returnLocalPath = false)
        {

            return returnLocalPath ? string.Format(@"{0}:\{1}", DestinationDriveLetter, DestinationFolderPath) :
                string.Format(@"\\{0}\{1}$\{2}", node, DestinationDriveLetter, DestinationFolderPath);
        }
        
        public bool Execute()
        {
            DeployStep.Initialize();

            DeployStep.RegisterStep("Validating Configuration", () => true, () =>
            {
                var fi = new FileInfo(@"C:\Program Files (x86)\PsExec\psexec.exe");

                if (!fi.Exists)
                {
                    Console.WriteLine(@"You are missing a local copy of PsExec. Please put it at C:\Program Files (x86)\PsExec\psexec.exe");
                    throw new Exception("Missing file.");
                }

                RemoteScriptHelper.Initialize(Username, Password);

            });

            DeployStep.RegisterStep("Whitelist Remote Servers", () => true, () =>
            {
                // Query list of servers
                var ipList = new List<string>();

                using (var conn = new SqlConnection("Data Source=sql07-live.curse.us;Initial Catalog=CurseVoice;Integrated Security=True;MultiSubnetFailover=true"))
                {
                    conn.Open();
                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "SELECT [IPAddress] from [VoiceHost]";

                        using (var reader = cmd.ExecuteReader())
                        {
                            while (reader.Read())
                            {
                                ipList.Add(reader.GetString(0));
                            }
                        }
                    }
                }

                foreach (var ipAddress in ipList)
                {
                    Console.WriteLine("White listing '" + ipAddress + "'...");
                    using (var ps = PowerShell.Create())
                    {
                        ps.AddScript(@"Set-Item wsman:\localhost\Client\TrustedHosts " + ipAddress + " -Concatenate -Force");
                        var result = ps.Invoke();
                    }
                }
            });

            DeployStep.RegisterStep("Creating Temp Folder", () => true, () =>
            {
                var tempFolder = new DirectoryInfo(TempFolderPath);
                if (!tempFolder.Exists)
                {
                    tempFolder.Create();
                }
            });

            
            DeployStep.RegisterStep("Zip Application Files", () => true, () =>
            {
                AppZipPath = Path.Combine(TempFolderPath, "Application.zip");

                if (File.Exists(AppZipPath))
                {
                    File.Delete(AppZipPath);
                }

                ZipHelper.CreateZip(ApplicationSourcePath, AppZipPath);
            });

            DeployStep.RegisterStep("Copy and Unzip App Files", () => true, () =>
            {
                foreach (var node in ServerNames)
                {
                    Console.WriteLine("Deploying to " + node);

                    using (NetworkShareAccesser.Access(node, node, Username, Password))
                    {
                        var di = new DirectoryInfo(GetApplicationPath(node));
                        if (!di.Exists)
                        {
                            Console.Write("Creating remote folder at: " + di.FullName + "...");
                            di.Create();
                            Console.WriteLine("Done");
                        }

                        using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                        {
                            
                            Console.WriteLine("Copying static zip file to production...");
                            ZipHelper.CopyZipToTemp(AppZipPath, node, ApplicationVersion);

                            Console.WriteLine("Stopping " + ApplicationName);
                            ToggleServiceState(runspace, ApplicationName, "stop");

                            Console.WriteLine("Unzipping zip file to file server...");
                            ZipHelper.UnzipRemotelyWithPowershell(runspace, PathHelper.GetRemoteTempFilePath(AppZipPath, node, ApplicationVersion, true), GetApplicationPath(node, true), node);

                            Console.WriteLine("Starting " + ApplicationName);
                            ToggleServiceState(runspace, ApplicationName, "start");
                            ToggleServiceState(runspace, "CurseVoice", "start");
                        }                        
                    }                    
                }
            });


            DeployStep.RunAll();
            return true;

        }

        


        private void ToggleServiceState(Runspace runspace, string serviceName, string state)
        {
            RemoteScriptHelper.ExecuteRemoteScript(runspace, "net " + state + " /y " + serviceName);
        }

        public string AppZipPath;

        [Required]
        public string ApplicationName { get; set; }

        [Required]
        public string ApplicationVersion { get; set; }

        [Required]
        public string ApplicationSourcePath { get; set; }

        [Required]
        public string[] ServerNames { get; set; }
        
        [Required]
        public string Username { get; set; }

        [Required]
        public string Password { get; set; }

    }
}
