﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using Microsoft.Build.Framework;
using System.Text.RegularExpressions;

namespace Curse.MSBuild.Packer
{
    public static class FileProcessor
    {       
        public static void Process(string outputFile, IEnumerable<string> inputFiles, IEnumerable<KeyValuePair<string,string>> substitutions, Func<string, string> procDelegate, bool verbose, string absolutePathBase, int outputFileCount = 1)
        {
            if (File.Exists(outputFile))
            {
                File.Delete(outputFile);
            }

            for (int i = 1; i <= outputFileCount; i++)
            {
                string outputFileName = Path.GetDirectoryName(outputFile) + Path.DirectorySeparatorChar + Path.GetFileNameWithoutExtension(outputFile) + "_" + i + Path.GetExtension(outputFile);
                if (File.Exists(outputFileName))
                {
                    File.Delete(outputFileName);
                }
            }
            DirectoryInfo destinationFolder = new DirectoryInfo(Path.GetDirectoryName(outputFile));

            if (!destinationFolder.Exists)
            {
                LogMessage("Creating new folder '{0}'\n", destinationFolder.FullName);
                destinationFolder.Create();
            }
 

            foreach (string inputFile in inputFiles)
            {
                if (verbose) {
                    LogMessage("Processing '{0}'\n", inputFile);
                }

                if (File.Exists(inputFile)) 
                {
                    
                    FileInfo info = new FileInfo(inputFile);
                    string fileContents;
                    using (StreamReader reader = info.OpenText())
                    {
                        fileContents = reader.ReadToEnd();
                    }

                    if (substitutions != null)
                    {
                        foreach (KeyValuePair<string, string> sub in substitutions)
                        {
                            if (verbose)
                            {
                                LogMessage("\tSubstituting {0} -> {1}\n", sub.Key, sub.Value);
                            }
                            fileContents = fileContents.Replace(sub.Key, sub.Value);
                        }
                    }
                    
                    if (procDelegate != null)
                    {
                        if (verbose)
                        {
                            LogMessage("Running delegate");
                        }

                        fileContents = procDelegate(fileContents);
                    }

                    fileContents = FileHeader(inputFile) + fileContents + "\n\n";

                    if (verbose)
                    {
                        LogMessage("Appending {0} to {1}\n", inputFile, outputFile);
                    }

                    File.AppendAllText(outputFile, fileContents);
                } else {
                    LogMessage("FILE NOT FOUND: '{0}'\n", inputFile);
                }

            }
        }
        private static string WrapUrl(string url)
        {
            return string.Format("url({0})", url);
        }
        private static string FileHeader(string fileName)
        {
            return string.Format("/* {0} */\n", fileName);
        }

        public static void LogMessage(string msg) {
            Console.WriteLine(msg);
        }

        public static void LogMessage(string format, params object[] args) {           
            Console.WriteLine(format, args);
        }
 
    }
}
