﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using Microsoft.Build.Framework;
using Microsoft.Build.Utilities;
using System.Xml;
using System.Xml.Linq;
using System.IO;
using System.Globalization;
using Yahoo.Yui.Compressor;

namespace Curse.MSBuild.Packer
{
    public class Packer : Task
    {

        /// <summary>
        /// Gets or sets the input files to pack.
        /// </summary>
        /// <value>The files to pack.</value>
        [Required]
        public ITaskItem[] InputFiles { get; set; }

        [Required]
        public string RootFolder { get; set; }

        /// Gets or sets the input files to pack.
        /// </summary>
        /// <value>The files to pack.</value>
        public string Substitutions { get; set; }

        private string _outputFileName;

        /// <summary>
        /// Gets or sets the output filename
        /// </summary>
        public string OutputFileName
        {
            get { return _outputFileName; }
            set { _outputFileName = value; }
        }

        /// <summary>
        /// Gets or sets the output file count.
        /// </summary>
        public int OutputFileCount { get; set; }

        /// <summary>
        /// Mode of processing to use:
        /// 
        /// CSSMinify
        /// JSMinify
        /// None
        /// </summary>
        public string Processing { get; set; }


        /// <summary>
        /// If set, converts all relative paths to an absolute path starting at this directory
        /// For example, if AbsoultePathBase is Content, then c:\projects\example\Content\Skins\MySkin\... will become /Skins/MySkin/...
        /// </summary>
        public string AbsolutePathBase { get; set; }

        /// <summary>
        /// Controls whether additional details are printed to the console
        /// </summary>
        public bool Verbose { get; set; }

        public override bool Execute()
        {
            Console.WriteLine("Creating asset package  '" + OutputFileName + "'...");

            List<string> inputFiles = new List<string>();
            foreach (ITaskItem item in InputFiles)
            {                
                inputFiles.Add(item.ItemSpec);
            }

            IEnumerable<KeyValuePair<string, string>> parsedSubs = null;
            if (!string.IsNullOrEmpty(Substitutions))
            {
                XDocument doc = XDocument.Parse(Substitutions);

                parsedSubs = from sub in doc.Root.Elements()
                             select new KeyValuePair<string, string>(sub.Attribute("From").Value, sub.Attribute("To").Value);

            }

            if (Processing == "Copy")
            {
                return FileCopier.CopyFiles(inputFiles.ToArray(), OutputFileName);
            }

            Func<string, string> procDelegate = null;

            if (!string.IsNullOrEmpty(Processing))
            {
                switch (Processing.ToLower())
                {
                    case "cssminify":
                        procDelegate = CSSMinifier.Minify;
                        break;
                    case "jsminify":
                        procDelegate = JavaScriptMinifier.Minify;
                        break;
                    case "googleclosurewhitespace":
                        procDelegate = GoogleClosure.MinifyWhitespace;
                        break;
                    case "googleclosuresimple":
                        procDelegate = GoogleClosure.MinifySimple;
                        break;
                    case "googleclosureadvanced":
                        procDelegate = GoogleClosure.MinifyAdvanced;
                        break;
                    case "jsyui":
                        JavaScriptCompressor compressor = new JavaScriptCompressor();
                        compressor.Encoding = Encoding.UTF8;
                        compressor.LoggingType = Verbose ? (LoggingType.Debug | LoggingType.Info) : LoggingType.None;
                        compressor.ThreadCulture = CultureInfo.CurrentCulture;
                        compressor.ErrorReporter = new YuiErrorReporter(Verbose);
                        compressor.ObfuscateJavascript = true;
                        procDelegate = text => string.IsNullOrEmpty(text) ? string.Empty : compressor.Compress(text);
                        //procDelegate = text => string.IsNullOrEmpty(text) ? string.Empty : new Yahoo.Yui.Compressor.JavaScriptCompressor(text, Verbose, Encoding.UTF8, CultureInfo.CurrentCulture, true, new YuiErrorReporter(true)).Compress();
                        break;
                    case "cssyui":
                        CssCompressor cssCompressor = new CssCompressor();
                        cssCompressor.RemoveComments = true;
                        procDelegate = text => string.IsNullOrEmpty(text) ? string.Empty : cssCompressor.Compress(text);
                        break;
                    default:
                        throw new InvalidOperationException(string.Format("Unknown Processing: {0}", Processing));

                }
            }

            FileProcessor.Process(OutputFileName, inputFiles, parsedSubs, procDelegate, Verbose, AbsolutePathBase, OutputFileCount);

            if (OutputFileCount > 1 && Processing.ToLower() == "cssyui")
            {
                // Split the compiled.css file.
                using (StreamReader stream = new StreamReader(OutputFileName))
                {
                    const int maxPartitionSize = 1048576; // 1MB
                    const int maxBufferSize = maxPartitionSize * 2; // ~2MB
                    long length = stream.BaseStream.Length;
                    int partitionSize = (int)Math.Ceiling(length / (double)OutputFileCount);
                    partitionSize = partitionSize > maxPartitionSize ? maxPartitionSize : partitionSize; // Ensure we don't create partitions larger than 1 MB.

                    for (int i = 1; i <= OutputFileCount; i++)
                    {
                        // Create a buffer that's twice the max size and fill it with empty characters. 
                        // This is to prevent out of range exceptions when writing to the buffer
                        // when the nearest end brace is beyond the partition size. If somehow 
                        // a partition winds up being larger than 2MB, we should look at refactoring a selector.
                        char[] buffer = new char[maxBufferSize].Select(c => ' ').ToArray();

                        stream.ReadBlock(buffer, 0, partitionSize);
                        string text = Encoding.ASCII.GetString(buffer.Select(b => (byte)b).ToArray()).Trim();
                        if (!text.EndsWith("}"))
                        {
                            bool stop = false;
                            while (!stop)
                            {
                                try
                                {
                                    if (stream.EndOfStream)
                                    {
                                        stop = true;
                                        continue;
                                    }

                                    char next = (char)stream.Read();
                                    // Seek to the next } so that we have a complete css statement.
                                    if (next == '}')
                                    {
                                        stop = true;
                                    }
                                    text += next;
                                }
                                catch
                                {
                                    // Must be the end of the file.
                                    stop = true;
                                }
                            }
                        }

                        // Write out the file with the text.
                        string outputFileName = Path.GetDirectoryName(OutputFileName) + Path.DirectorySeparatorChar + Path.GetFileNameWithoutExtension(OutputFileName) + "_" + i + Path.GetExtension(OutputFileName);
                        using (FileStream fs = File.Open(outputFileName, !File.Exists(outputFileName) ? FileMode.CreateNew : FileMode.Truncate, FileAccess.ReadWrite))
                        {
                            byte[] textBuffer = Encoding.ASCII.GetBytes(text);
                            fs.Write(textBuffer, 0, text.Length);
                            fs.Flush();
                        }
                    }
                }

                // Move the compiled.css file to a temp location
                Regex skinNameRegex = new Regex(@"Skins\\(?<skinName>\w+?)\\css\\compiled.css");
                string skinName = skinNameRegex.Match(OutputFileName).Groups["skinName"].Value;
                string tempPathDirectory = string.Format(@"C:\Windows\TEMP\Deploys\CompiledContent\{0}\{1}\", (DateTime.Now.ToShortDateString() + " " + DateTime.Now.ToShortTimeString()).Replace("\\", "").Replace("/", "").Replace("-", "").Replace(":", ""), skinName);
                if (!Directory.Exists(tempPathDirectory))
                {
                    Directory.CreateDirectory(tempPathDirectory);
                }

                string tempPath = tempPathDirectory + "compiled.css";
                File.Move(OutputFileName, tempPath);
            }

            return true;
        }

    }
}