﻿using System;
using System.IO;
using System.Linq;
using System.Security;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using Curse.MSBuild.Deployment;
using Curse.ServiceUpdate.WebClient;
using Curse.ServiceUpdate.WebClient.UpdateWebService;
using ICSharpCode.SharpZipLib.Zip;
using Microsoft.Build.Framework;

namespace Curse.ServiceUpdate.Deployment
{
    public class ServiceUpdateDeployTask : ITask
    {
        [Required]
        public string ApiKey { get; set; }

        [Required]
        public string ApplicationBinPath { get; set; }

        [Required]
        public string ApplicationName { get; set; }

        [Required]
        public string ApplicationVersion { get; set; }

        [Required]
        public string Environment { get; set; }

        [Required]
        public bool ServerFilteringPrompt { get; set; }

        public bool RequiresSpecificClientVersion { get; set; }

        public bool VerifyCommitPrompt { get; set; }

        private string _localZipPath;

        [Required]
        public string CodeSigningCertificateThumbprint { get; set; }

        public bool Execute()
        {
            bool disableRollout = false;
            int? hostID = null;
            int? regionID = null;
            if (ServerFilteringPrompt)
            {
                disableRollout = PromptForRolloutFilter(out regionID, out hostID);
            }

            var commitMode = VerifyCommitMode.First;
            if (VerifyCommitPrompt)
            {
                commitMode = PromptForVerifyCommitMode();
            }

            Version serviceVersion;
            if (!Version.TryParse(ApplicationVersion, out serviceVersion))
            {
                serviceVersion = new Version(1, 0, 0, 0);
            }

            var environment = (UpdateEnvironment) Enum.Parse(typeof (UpdateEnvironment), Environment);

            DeployStep.Initialize();

            DeployStep.RegisterStep("Zip application files", () => true, () =>
            {
                // Zip up everything
                _localZipPath = Path.GetTempFileName();
                var zip = new FastZip();
                zip.CreateZip(_localZipPath, ApplicationBinPath, true,
                    @"\.dll$;\.exe$;\.pdb$;\." + Environment + @".config$;\.exe.config$", "Configuration");
            });

            DeployStep.RegisterStep("Push Update", () => true, () =>
            {
                using (var stream = File.OpenRead(_localZipPath))
                {
                    using (var ms = new MemoryStream())
                    {
                        stream.CopyTo(ms);
                        var bytes = ms.ToArray();

                        X509Store store = new X509Store(StoreName.My, StoreLocation.LocalMachine);
                        store.Open(OpenFlags.ReadOnly);
                        var codeSigningCert = store.Certificates.Find(X509FindType.FindByThumbprint, CodeSigningCertificateThumbprint, true)[0];
                        store.Close();

                        var signature = ((RSACryptoServiceProvider) codeSigningCert.PrivateKey).SignData(bytes, new SHA1CryptoServiceProvider());

                        try
                        {
                            var response = UpdateDeployerHelper.DeployUpdate(environment, ApiKey, new UpdateCoordinatorCallback(), new DeployUpdateRequestBody
                            {
                                DisableRollout = disableRollout,
                                HostID = hostID,
                                RegionID = regionID,
                                ServiceName = ApplicationName,
                                ServiceVersion = serviceVersion.ToString(),
                                VerifyCommitMode = commitMode,
                                Payload = bytes,
                                PayloadCertificate = codeSigningCert.Export(X509ContentType.Cert),
                                PayloadSignature = signature
                            });
                            if (response.Status != DeploymentStatus.Success)
                            {
                                Console.WriteLine("Failed to push the update. {0}" + response.StatusMessage);
                            }
                        }
                        catch (Exception ex)
                        {
                            Console.WriteLine("Failed to push the update. An exception was thrown. {0}", ex.Message);
                        }
                    }
                }
            });

            DeployStep.RunAll();
            return true;
        }

        private bool PromptForRolloutFilter(out int? regionID, out int? hostID)
        {
            ConsoleKey key = ConsoleKey.Home;
            while (key != ConsoleKey.H && key != ConsoleKey.R && key != ConsoleKey.A)
            {
                Console.Write("Do you want to deploy this to a specific [H]ost, [R]egion, or to [A]ll services? (default=A): ");
                key = Console.ReadKey().Key;
            }
            Console.WriteLine();

            bool shouldFilter;
            switch (key)
            {
                case ConsoleKey.H:
                    hostID = PromptForHostID();
                    regionID = null;
                    shouldFilter = true;
                    break;
                case ConsoleKey.R:
                    hostID = null;
                    regionID = PromptForRegionID();
                    shouldFilter = true;
                    break;
                default:
                    hostID = null;
                    regionID = null;
                    shouldFilter = false;
                    break;
            }
            Console.WriteLine();

            return shouldFilter;
        }

        private VerifyCommitMode PromptForVerifyCommitMode()
        {
            Console.Write("Do you want to verify [A]ll, [N]one, or the [F]irst commit? (default=A): ");
            var key = Console.ReadKey().Key;
            var mode = key == ConsoleKey.N
                ? VerifyCommitMode.None
                : key == ConsoleKey.F
                    ? VerifyCommitMode.First
                    : VerifyCommitMode.All;

            Console.WriteLine();
            return mode;
        }

        private int PromptForHostID()
        {
            var host = -1;
            while (host < 0)
            {
                Console.Write("Enter the ID of the host to deploy to: ");
                if (!int.TryParse(Console.ReadLine(), out host))
                {
                    Console.Write("\rInvalid host ID. ");
                    host = -1;
                }
            }
            Console.WriteLine();
            return host;
        }

        private int PromptForRegionID()
        {
            var region = -1;
            while (region < 0)
            {
                Console.Write("Enter the ID of the region to deploy to: ");
                if (!int.TryParse(Console.ReadLine(), out region))
                {
                    Console.Write("\rInvalid region ID. ");
                    region = -1;
                }
            }
            return region;
        }

        public IBuildEngine BuildEngine { get; set; }
        public ITaskHost HostObject { get; set; }
    }
}
