﻿using System;
using System.Configuration;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.Serialization;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.ServiceProcess;
using System.Text;
using Curse.MSBuild.Deployment;
using Curse.ServiceUpdate.Deployment;
using Curse.ServiceUpdate.UpdateManagement;
using Curse.ServiceUpdate.UpdateManagement.Messages;
using Curse.ServiceUpdate.UpdateManagement.Security;
using Curse.Voice.UpdateManagement;
using ICSharpCode.SharpZipLib.Zip;

namespace Curse.ServiceUpdate.Tester
{
    class Program
    {

        private static void Main(string[] args)
        {
            if (Debugger.IsAttached)
            {
                new CurseUpdaterTestService().OnDebugStart();
#if OLD_UPDATER
                // Old updater API - hardcoded to work with CurseVoice service only for backwards compatibility
                // UpdateService needs to be running locally
                // Test will ensure the dummy CurseVoice service is installed
                RunTestWithOldUpdater();
#elif NEW_UPDATER
                // New updater API - works with any service
                // UpdateService needs to be running locally
                // Test will use InstallIfMissing to install the service during deploy
                RunTestWithNewUpdater();
#else
                // New central updater service - works with any installed service that does not need failover
                // UpdateService needs to be running locally
                // Test will ensure the test service is installed
                // UpdateCoordinator Web Service needs to be running on http://update-service.curse.dev/UpdateCoordinator.svc
                RunTestWithNewCentralUpdater();
#endif
            }
            else
            {
                ServiceBase.Run(new ServiceBase[] {new CurseUpdaterTestService()});
            }
        }


#if OLD_UPDATER
        const string ServiceName = "CurseVoice";
#else
        const string ServiceName = "CurseUpdaterTestService";
#endif

#if NEW_CENTRAL_UPDATER
        public static void RunTestWithNewCentralUpdater()
        {
            EnsureService(ServiceName);

            var task = new ServiceUpdateDeployTask
            {
                ApiKey = ConfigurationManager.AppSettings["ApiKey"],
                ApplicationBinPath = AppDomain.CurrentDomain.BaseDirectory,
                ApplicationName = ServiceName,
                ApplicationVersion = Assembly.GetExecutingAssembly().GetName().Version.ToString(),
                Environment = "Debug",
                RequiresSpecificClientVersion = false,
                ServerFilteringPrompt = true,
                VerifyCommitPrompt = true,
            };

            PrintResults(task.Execute());
            UninstallService(ServiceName);
        }
#endif

#if OLD_UPDATER
        public static void RunTestWithOldUpdater()
        {
            EnsureService(ServiceName);

            try
            {
                using (var updaterClient = new VoiceUpdaterClient("localhost", 8080))
                {
                    using (var stream = File.OpenRead(CreateZip()))
                    {
                        using (var memoryStream = new MemoryStream())
                        {
                            var version = Assembly.GetExecutingAssembly().GetName().Version;
                            stream.CopyTo(memoryStream);
                            var response = updaterClient.PushUpdate(ConfigurationManager.AppSettings["ApiKey"], version,
                                memoryStream.ToArray());
                            PrintResults(response);
                            UninstallService(ServiceName);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("An exception occurred while pushing the update: " + ex.Message);
                Debugger.Break();
            }
        }
#endif

#if NEW_UPDATER
        public static void RunTestWithNewUpdater()
        {
            // Don't ensure service so InstallIfMissing is used
            var store = new X509Store(StoreName.My, StoreLocation.LocalMachine);
            store.Open(OpenFlags.OpenExistingOnly);
            var cert = store.Certificates.Find(X509FindType.FindBySubjectDistinguishedName, "CN=tempCert", true)[0];

            byte[] zipBytes;
            using (var ms = new MemoryStream())
            {
                using (var fs = File.OpenRead(CreateZip()))
                {
                    fs.CopyTo(ms);
                }
                zipBytes = ms.ToArray();
            }

            using (var client = UpdateListenerClient.Create(Environment.MachineName, 8080, cert, ConfigurationManager.AppSettings["ApiKey"]))
            {
                var resp = client.PushUpdate(new PushRequestBody
                {
                    InstallIfMissing = true,
                    ServiceName = ServiceName,
                    ServiceVersion = Assembly.GetExecutingAssembly().GetName().Version,
                    ZipStream = zipBytes,
                });
                PrintResults(resp.Status == UpdateStatus.Success, resp.StatusMessage);
                UninstallService(ServiceName);
            }
        }
#endif

        private static void EnsureService(string serviceName)
        {
            // Ensure service is installed since the old updater behavior is only to update, never to install.
            if (!WindowsServiceHelper.HasService("localhost", serviceName))
            {
                Console.Write("Installing Service... ");
                if (WindowsServiceHelper.InstallService("localhost", serviceName,
                    Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Curse.WindowsServiceUpdater.Tester.exe"), true))
                {
                    Console.WriteLine("done!");
                }
                else
                {
                    Console.WriteLine("failed! Continuing anyway.");
                }
            }

            if (!WindowsServiceHelper.ToggleServiceState("localhost", serviceName, "start"))
            {
                Console.WriteLine("Failed to start {0}. Continuing anyway.", serviceName);
            }
        }

        private static void PrintResults(bool success, string message=null)
        {
            if (success)
            {
                Console.WriteLine("Push completed.");
            }
            else
            {
                Console.WriteLine("Push did not complete fully. {0}", message ?? "No reason specified.");
            }
            Console.WriteLine("Press Enter to exit.");
            Console.ReadLine();
        }

        private static string CreateZip()
        {
            var zipFilePath = Path.GetTempFileName();
            var zip = new FastZip();
            zip.CreateZip(zipFilePath, AppDomain.CurrentDomain.BaseDirectory, true, null, null);
            return zipFilePath;
        }

        private static void UninstallService(string name)
        {
            if (WindowsServiceHelper.HasService("localhost", name))
            {
                Console.Write("Uninstalling Service... ");
                if (WindowsServiceHelper.UninstallService("localhost", name,
                    Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Curse.WindowsServiceUpdater.Tester.exe"), true))
                {
                    Console.WriteLine("done!");
                }
                else
                {
                    Console.WriteLine("failed!");
                }
            }
        }
    }
}
