﻿using System;
using System.IO;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using Curse.ServiceUpdate.UpdateManagement.Messages;
using Curse.ServiceUpdate.UpdateManagement.Security;

namespace Curse.ServiceUpdate.UpdateManagement
{
    public abstract class TcpClientBase<TChannel> : ClientBase<TChannel> where TChannel:class,ISecureService
    {
        protected byte[] EncryptionKey;

        protected TcpClientBase(string address, int port, string path)
            :base(GetClientTcpBinding(),GetClientEndpoint(address,port,path))
        {
        }

        protected TcpClientBase(string address, int port, string path, object callback)
            : base(new InstanceContext(callback), GetClientTcpBinding(), GetClientEndpoint(address, port, path))
        {
        }

        protected void Handshake(X509Certificate2 certificate, string apiKey)
        {
            var handshake = Channel.Handshake(new ReverseHandshakeRequest
            {
                Certificate = certificate.Export(X509ContentType.Cert)
            });

            EncryptionKey = Encryption.DecryptKey(handshake.EncryptedKey, certificate);

            var pair = Encryption.EncryptData(apiKey, EncryptionKey);
            var response = Channel.Authenticate(new AuthenticateRequest
            {
                EncryptedApiKey = pair.EncryptedData,
                InitializationVector = pair.InitializationVector
            });

            if (!response.Success)
            {
                throw new InvalidOperationException("Authentication failed.");
            }
        }

        protected void Handshake(X509Certificate2 certificate, string apiKey, IUpdaterSeedCallback callback, byte[] callbackEncryptionKey)
        {
            var handshake = Channel.Handshake(new ReverseHandshakeRequest
            {
                Certificate = certificate.Export(X509ContentType.Cert)
            });

            var pair = Encryption.EncryptData(new GetKeyRequestBody {EncryptedKey = handshake.EncryptedKey}, callbackEncryptionKey);
            using (var stream = new MemoryStream(pair.EncryptedData))
            {
                var keyResponse = callback.GetKey(new GetKeyRequest
                {
                    EncryptedBody = stream,
                    InitializationVector = pair.InitializationVector
                });
                var keyResponseBody = Encryption.Decrypt<GetKeyResponseBody>(keyResponse.EncryptedBody, callbackEncryptionKey, keyResponse.InitializationVector);
                EncryptionKey = keyResponseBody.Key;
            }

            pair = Encryption.EncryptData(apiKey, EncryptionKey);
            var response = Channel.Authenticate(new AuthenticateRequest
            {
                EncryptedApiKey = pair.EncryptedData,
                InitializationVector = pair.InitializationVector
            });

            if (!response.Success)
            {
                throw new InvalidOperationException("Authentication failed.");
            }
        }

        private static NetTcpBinding GetClientTcpBinding()
        {
            NetTcpBinding binding = new NetTcpBinding(SecurityMode.None);
            binding.MaxReceivedMessageSize = Int32.MaxValue;
            binding.TransactionFlow = false;
            binding.OpenTimeout = TimeSpan.FromSeconds(5);
            binding.CloseTimeout = TimeSpan.FromSeconds(5);
            binding.ReceiveTimeout = TimeSpan.FromMinutes(5);
            binding.SendTimeout = TimeSpan.FromMinutes(1);
            binding.MaxBufferSize = Int32.MaxValue;
            binding.MaxReceivedMessageSize = Int32.MaxValue;
            binding.ReaderQuotas.MaxArrayLength = Int32.MaxValue;

            binding.Security.Mode = SecurityMode.None;
            binding.Security.Message.ClientCredentialType = MessageCredentialType.None;
            binding.Security.Transport.ClientCredentialType = TcpClientCredentialType.None;
            return binding;
        }

        private static EndpointAddress GetClientEndpoint(string address, int port, string path)
        {
            return new EndpointAddress(new Uri(string.Format("net.tcp://{0}:{1}/{2}", address, port, path)));
        }
    }
}
