﻿using System;
using System.Collections.Generic;
using System.Net.Security;
using System.ServiceModel;
using System.ServiceModel.Description;

namespace Curse.ServiceUpdate.UpdateService
{
    public class ServiceHostHelper
    {
        private static NetTcpBinding GetServerTcpBinding(bool portSharingEnabled = false)
        {
            NetTcpBinding binding = new NetTcpBinding(SecurityMode.None);
            binding.MaxConnections = Int32.MaxValue;
            binding.PortSharingEnabled = portSharingEnabled;
            binding.MaxReceivedMessageSize = Int32.MaxValue;
            binding.ReaderQuotas.MaxArrayLength = Int32.MaxValue;
            binding.TransactionFlow = false;
            binding.ListenBacklog = 1000;
            binding.Security.Message.ClientCredentialType = MessageCredentialType.None;
            binding.Security.Transport.ClientCredentialType = TcpClientCredentialType.None;
            binding.Security.Transport.ProtectionLevel = ProtectionLevel.None;

            binding.ReceiveTimeout = TimeSpan.FromDays(1);
            binding.SendTimeout = TimeSpan.FromDays(1);

            return binding;
        }

        private readonly Dictionary<string, NetTcpBinding> _bindingsMap = new Dictionary<string, NetTcpBinding>(); 

        public static ServiceHost CreateServiceHost<T>(object serviceInstance, string address,
            bool portSharingEnabled = false)
        {
            return new ServiceHostHelper(serviceInstance).AddTcpEndpoint<T>(address, portSharingEnabled).GetHost();
        }

        public static ServiceHost CreateServiceHost<T1, T2>(object serviceInstance, string address)
        {
            return new ServiceHostHelper(serviceInstance).AddTcpEndpoint<T1>(address, true)
                .AddTcpEndpoint<T2>(address, true).GetHost();
        }

        public ServiceHostHelper AddTcpEndpoint<T>(string address, bool portSharingEnabled = false)
        {
            if (_host.SingletonInstance is T)
            {
                NetTcpBinding binding;
                if (!_bindingsMap.TryGetValue(address, out binding))
                {
                    binding = GetServerTcpBinding(portSharingEnabled);
                    _bindingsMap[address] = binding;
                }
                _host.AddServiceEndpoint(typeof (T), binding, address);
            }
            return this;
        }

        private readonly ServiceHost _host;

        public ServiceHostHelper(object serviceInstance)
        {
            _host = new ServiceHost(serviceInstance);
        
            // Service Throttling
            {
                ServiceThrottlingBehavior stb = new ServiceThrottlingBehavior();
                _host.Description.Behaviors.Remove<ServiceThrottlingBehavior>();
                stb.MaxConcurrentCalls = Int32.MaxValue;
                stb.MaxConcurrentInstances = Int32.MaxValue;
                stb.MaxConcurrentSessions = Int32.MaxValue;
                _host.Description.Behaviors.Add(stb);
            }

            // Service Meta Data
            {
                ServiceMetadataBehavior serviceMetadataBehavior = new ServiceMetadataBehavior();
                serviceMetadataBehavior.HttpGetEnabled = false;
                _host.Description.Behaviors.Remove<ServiceMetadataBehavior>();
                _host.Description.Behaviors.Add(serviceMetadataBehavior);
            }

            {
                ServiceDebugBehavior serviceDebugBehavior = new ServiceDebugBehavior();
                serviceDebugBehavior.IncludeExceptionDetailInFaults = true;
                _host.Description.Behaviors.Remove<ServiceDebugBehavior>();
                _host.Description.Behaviors.Add(serviceDebugBehavior);
            }
        }

        public ServiceHost GetHost()
        {
            return _host;
        }
    }
}
