﻿using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Security;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceProcess;
using System.Threading;
using Curse.Logging;
using Curse.ServiceUpdate.UpdateManagement;
using Curse.ServiceUpdate.UpdateManagement.Messages;
using Curse.ServiceUpdate.UpdateManagement.Security;
using Curse.ServiceUpdate.UpdateService.Configuration;
using Curse.Voice.UpdateManagement;
using ICSharpCode.SharpZipLib.Zip;

namespace Curse.ServiceUpdate.UpdateService
{
    [ServiceBehavior(InstanceContextMode = InstanceContextMode.PerSession, ConcurrencyMode = ConcurrencyMode.Reentrant)]
    public class UpdaterService : IUpdateListener, IVoiceHostUpdateListener, IUpdaterSeed
    {
        private static bool _isUpdating;
        private bool _isSessionEstablished;
        private X509Certificate2 _certificate;
        private string _apiKey;

        public PushResponseMessage PushUpdate(PushRequestMessage message)
        {
            if (!_isSessionEstablished)
            {
                return new PushResponseMessage();
            }

            var body = Encryption.Decrypt<PushRequestBody>(message.EncryptedBody, _key, message.InitializationVector);
            var response = PushUpdate(body.ServiceName, body.ServiceVersion, body.ZipStream,
                body.BaseInstallLocation,
                body.InstallIfMissing);

            var pair = Encryption.EncryptData(response, _key);
            return new PushResponseMessage
            {
                InitializationVector = pair.InitializationVector,
                EncryptedBody = pair.EncryptedData
            };
        }

        private PushResponseBody PushUpdate(string serviceName, Version serviceVersion, byte[] payload, string baseInstallLocation, bool installIfMissing)
        {
            if (_isUpdating)
            {
                Logger.Warn("Received an update request while another update is already in progress.", new {serviceName, serviceVersion});
                return new PushResponseBody
                {
                    Status = UpdateStatus.Error,
                    StatusMessage = "An update is already in progress."
                };
            }

            Logger.Info("Received update for version " + serviceVersion);
            _isUpdating = true;

            try
            {

                var currentVersionFolder = GetCurrentInstallFolder(serviceName, baseInstallLocation);
                if (!currentVersionFolder.Exists)
                {
                    currentVersionFolder.Create();
                }
                var newVersionFolder = GetVersionedFolder(serviceName, serviceVersion, baseInstallLocation);

                // Write the payload to a temp file.
                var tempFilePath = Path.GetTempFileName();
                using (var fileStream = File.OpenWrite(tempFilePath))
                {
                    fileStream.Write(payload, 0, payload.Length);
                }

                // Extract the install to a versioned folder
                Logger.Debug("Extracting Payload to '" + newVersionFolder.FullName + "'");
                if (!Extract(newVersionFolder, tempFilePath))
                {
                    Logger.Error("Failed to extract update payload.");
                    return new PushResponseBody
                    {
                        Status = UpdateStatus.Error,
                        StatusMessage = "Failed to extract the zip stream."
                    };
                }

                bool isInstalled = ServiceInstalled(serviceName);

                Version lastVersion;
                if (isInstalled)
                {
                    // Next stop the current service
                    Logger.Info("Stopping Service " + serviceName);
                    if (!StopService(serviceName))
                    {
                        Logger.Error("Service failed to stop.");
                        return new PushResponseBody
                        {
                            Status = UpdateStatus.Error,
                            StatusMessage = "Failed to stop the service."
                        };
                    }

                    // Read in the version info from the current install
                    Logger.Debug("Getting Current Runtime Version");
                    if (!GetAssemblyVersion(GetCurrentExecutableFile(serviceName, baseInstallLocation), out lastVersion))
                    {
                        lastVersion = new Version("1.0.0.0");
                    }

                    // Copy current code into a versioned folder
                    var lastVersionFolder = GetVersionedFolder(serviceName, lastVersion, baseInstallLocation);

                    if (!lastVersionFolder.Exists)
                    {
                        lastVersionFolder.Create();
                    }

                    if (!serviceVersion.Equals(lastVersion))
                    {
                        Logger.Debug(string.Format("Copying '{0}' to '{1}'", currentVersionFolder.FullName, lastVersionFolder.FullName));
                        CopyFolderContents(currentVersionFolder.FullName, lastVersionFolder.FullName);
                    }
                }


                // Delete the contents of the Current folder
                Logger.Debug(string.Format("Deleting contents of '{0}'", currentVersionFolder.FullName));
                if (!DeleteFolderContents(currentVersionFolder.FullName, true))
                {
                    Logger.Error("Failed to delete the previous service files from the Current directory.");
                    return new PushResponseBody
                    {
                        Status = UpdateStatus.Error,
                        StatusMessage = "Failed to delete the old service files."
                    };
                }

                // Copy the new version folder contents into the Current directory
                Logger.Debug(string.Format("Copying '{0}' to '{1}'", newVersionFolder.FullName, currentVersionFolder.FullName));
                if (!CopyFolderContents(newVersionFolder.FullName, currentVersionFolder.FullName))
                {
                    Logger.Error("Failed to copy service files into the Current directory.");
                    return new PushResponseBody
                    {
                        Status = UpdateStatus.Error,
                        StatusMessage = "Failed to copy the service files."
                    };
                }

                if (!isInstalled && installIfMissing)
                {
                    Logger.Info("Installing service " + serviceName);
                    if (!InstallService(GetCurrentExecutableFile(serviceName, baseInstallLocation).FullName))
                    {
                        Logger.Error("Service failed to install.");
                        return new PushResponseBody
                        {
                            Status = UpdateStatus.Error,
                            StatusMessage = "Failed to install the service."
                        };
                    }
                }

                // Finally, start the service
                Logger.Info("Starting Service");
                if (!StartService(serviceName))
                {
                    Logger.Error("Service failed to start.");
                    return new PushResponseBody
                    {
                        Status = UpdateStatus.Error,
                        StatusMessage = "Failed to start the service."
                    };
                }

                Logger.Info("Update Successful");
                return new PushResponseBody { Status = UpdateStatus.Success };

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process update!");

                return new PushResponseBody
                {
                    Status = UpdateStatus.Error,
                    StatusMessage = "Unknown."
                };
            }
            finally
            {
                _isUpdating = false;
            }

        }

        private static bool DeleteFolderContents(string sourcePath, bool isRootFolder = false)
        {
            sourcePath = sourcePath.EndsWith(@"\") ? sourcePath : sourcePath + @"\";

            for (var i = 0; i < 3; i++)
            {

                try
                {
                    if (Directory.Exists(sourcePath))
                    {

                        foreach (var file in Directory.GetFiles(sourcePath, "*", SearchOption.AllDirectories))
                        {
                            File.Delete(file);
                        }

                        foreach (var drs in Directory.GetDirectories(sourcePath, "*", SearchOption.TopDirectoryOnly))
                        {
                            if (!DeleteFolderContents(drs))
                            {
                                Logger.Error("Failed to delete nested folder '" + drs + "'");
                                return false;
                            }
                        }

                        if (!isRootFolder)
                        {
                            Directory.Delete(sourcePath);
                        }
                    }
                    return true;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to delete folder contents of '" + sourcePath + "'");

                }

                Logger.Info("Failed to delete folder, trying again in 1 second...");

                Thread.Sleep(1000);
            }

            return false;
        }


        private static bool CopyFolderContents(string sourcePath, string destinationPath)
        {
            sourcePath = sourcePath.EndsWith(@"\") ? sourcePath : sourcePath + @"\";
            destinationPath = destinationPath.EndsWith(@"\") ? destinationPath : destinationPath + @"\";

            try
            {
                if (Directory.Exists(sourcePath))
                {
                    if (Directory.Exists(destinationPath) == false)
                    {
                        Directory.CreateDirectory(destinationPath);
                    }

                    foreach (string files in Directory.GetFiles(sourcePath))
                    {
                        var fileInfo = new FileInfo(files);
                        fileInfo.CopyTo(string.Format(@"{0}\{1}", destinationPath, fileInfo.Name), true);
                    }

                    foreach (string drs in Directory.GetDirectories(sourcePath))
                    {
                        var directoryInfo = new DirectoryInfo(drs);
                        if (CopyFolderContents(drs, destinationPath + directoryInfo.Name) == false)
                        {
                            return false;
                        }
                    }
                }
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to copy folder contents from '" + sourcePath + "' to '" + destinationPath + "'");
                return false;
            }
        }


        private static bool Extract(DirectoryInfo destination, string sourceFilePath)
        {
            try
            {
                // Ensure we can create a folder in the install location                
                if (!destination.Exists)
                {
                    destination.Create();
                }

                // Extract the uploaded zip to a local temp file
                var fastZip = new FastZip();
                fastZip.ExtractZip(sourceFilePath, destination.FullName, FastZip.Overwrite.Always, null, null, null, true);
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to extract payload");
                return false;
            }
        }

        private static DirectoryInfo GetVersionedFolder(string serviceName, Version version, string baseInstallLocation)
        {
            return new DirectoryInfo(
                Path.Combine(baseInstallLocation ?? UpdateServiceConfiguration.Instance.LocalInstallPath,
                    serviceName, version.ToString().Replace(".", "-")));
        }

        private static DirectoryInfo GetCurrentInstallFolder(string serviceName, string baseInstallLocation)
        {
            return new DirectoryInfo(
                Path.Combine(baseInstallLocation ?? UpdateServiceConfiguration.Instance.LocalInstallPath,
                    serviceName, "Current"));
        }

        private static FileInfo GetCurrentExecutableFile(string serviceName, string baseInstallLocation)
        {
            var directoryInfo = GetCurrentInstallFolder(serviceName, baseInstallLocation);
            var file = directoryInfo.EnumerateFiles("*.exe").FirstOrDefault();
            return file;
        }

        private static bool GetAssemblyVersion(FileInfo file, out Version version)
        {
            try
            {
                if (file != null)
                {
                    FileVersionInfo fileVersionInfo = FileVersionInfo.GetVersionInfo(file.FullName);
                    version = Version.Parse(fileVersionInfo.ProductVersion);
                    return true;
                }
                version = null;
                return false;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to determine the current version of the service");
                version = null;
                return false;
            }

        }

        private static bool StartService(string serviceName)
        {
            try
            {
                // Next stop the current service
                using (ServiceController sc = new ServiceController(serviceName))
                {
                    sc.Start();
                    sc.WaitForStatus(ServiceControllerStatus.Running, TimeSpan.FromSeconds(30));
                    return true;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to start CurseVoice service");
                return false;
            }

        }


        private static bool StopService(string serviceName)
        {
            try
            {
                // Next stop the current service
                using (var sc = new ServiceController(serviceName))
                {
                    if (sc.Status == ServiceControllerStatus.Stopped)
                    {
                        return true;
                    }

                    sc.Stop();

                    int i = 0;
                    while (sc.Status != ServiceControllerStatus.Stopped)
                    {
                        if (++i > 30)
                        {
                            Logger.Error("Failed to stop service, after 30 atempts!");
                            return false;
                        }
                        Thread.Sleep(1000);
                        sc.Refresh();
                    }
                }
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to stop CurseVoice service");
                return false;
            }

        }

        private static bool ServiceInstalled(string serviceName)
        {
            try
            {
                return ServiceController.GetServices().Any(s => s.ServiceName == serviceName);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to query services.");
                return false;
            }
        }

        private static bool InstallService(string binPath)
        {
            try
            {
                var installUtil = new Process();
                installUtil.StartInfo.FileName = Path.Combine(System.Runtime.InteropServices.RuntimeEnvironment.GetRuntimeDirectory(), "installutil.exe");
                installUtil.StartInfo.Arguments = string.Join(" ",
                    string.Format(@"""{0}""", binPath)
                    );
                installUtil.StartInfo.UseShellExecute = false;
                installUtil.Start();
                installUtil.WaitForExit();
                return installUtil.ExitCode == 0;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to install the service.");
                return false;
            }
        }

        public bool PushUpdate(string apiKey, Version newVersion, byte[] payload)
        {
            return PushUpdate("CurseVoice", newVersion, payload, @"C:\Program Files", false).Status ==
                   UpdateStatus.Success;
        }

        public UploadResponseMessage UploadUpdate(UploadRequestMessage request)
        {
            if (!_isSessionEstablished)
            {
                throw new SecurityException("Unauthorized access");
            }

            var body = Encryption.Decrypt<UploadRequestBody>(request.EncryptedBody, _key, request.InitializationVector);
            var response = UploadUpdate(body);
            var pair = Encryption.EncryptData(response, _key);
            return new UploadResponseMessage
            {
                InitializationVector = pair.InitializationVector,
                EncryptedBody = pair.EncryptedData
            };
        }

        public UploadResponseBody UploadUpdate(UploadRequestBody requestBody)
        {

            try
            {
                var directory = new DirectoryInfo(UpdateServiceConfiguration.Instance.UpdateStorePath);
                if (!directory.Exists)
                {
                    directory.Create();
                }

                var filePath = GetUpdateFilePath(requestBody.ServiceName, requestBody.ServiceVersion);
                using (var stream = File.Open(filePath, FileMode.OpenOrCreate))
                {
                    stream.Write(requestBody.ZipStream, 0, requestBody.ZipStream.Length);
                }

                return new UploadResponseBody { Status = UploadStatus.Success };
            }
            catch (Exception)
            {
                return new UploadResponseBody { Status = UploadStatus.Error, StatusMessage = "Unexpected Exception" };
            }
        }

        public UpdateResponseMessage UpdateService(UpdateRequestMessage request)
        {
            if (!_isSessionEstablished)
            {
                throw new SecurityException("Unauthorized access");
            }

            var body = Encryption.Decrypt<UpdateRequestBody>(request.EncryptedBody, _key, request.InitializationVector);
            var response = UpdateService(body);
            var pair = Encryption.EncryptData(response, _key);
            return new UpdateResponseMessage
            {
                InitializationVector = pair.InitializationVector,
                EncryptedBody = pair.EncryptedData
            };
        }

        private UpdateResponseBody UpdateService(UpdateRequestBody body)
        {

            try
            {
                Logger.Info("Received update request for " + body.TargetHost);
                byte[] zipBytes;
                using (var ms = new MemoryStream())
                {
                    using (var stream = File.OpenRead(GetUpdateFilePath(body.ServiceName, body.ServiceVersion)))
                    {
                        stream.CopyTo(ms);

                    }
                    zipBytes = ms.ToArray();
                }

                using (var client = UpdateListenerClient.Create(body.TargetHost, UpdateServiceConfiguration.Instance.ListenerPortNumber, _certificate, _apiKey,
                        OperationContext.Current.GetCallbackChannel<IUpdaterSeedCallback>(), _key))
                {
                    var response = client.PushUpdate(new PushRequestBody
                    {
                        BaseInstallLocation = body.BaseInstallLocation,
                        InstallIfMissing = body.InstallIfMissing,
                        ServiceName = body.ServiceName,
                        ServiceVersion = body.ServiceVersion,
                        ZipStream = zipBytes
                    });
                    return new UpdateResponseBody
                    {
                        Status =
                            response.Status == UpdateStatus.Success
                                ? UpdateStatus.Success
                                : UpdateStatus.Error,
                        StatusMessage =
                            response.Status == UpdateStatus.Success ? null : response.StatusMessage
                    };
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[UpdateService] Unexpected exception.", body);
                return new UpdateResponseBody {Status = UpdateStatus.Error, StatusMessage = "Unexpected Exception"};
            }
        }

        private static string GetUpdateFilePath(string serviceName, Version version)
        {
            var fileName = string.Format("{0}-{1}", serviceName, version.ToString().Replace(".", "-"));
            return Path.Combine(UpdateServiceConfiguration.Instance.UpdateStorePath, fileName);
        }

        private byte[] _key;

        public ReverseHandshakeResponse Handshake(ReverseHandshakeRequest request)
        {
            _certificate = new X509Certificate2(request.Certificate);
            if (!_certificate.Verify())
            {
                Logger.Error("Invalid Certificate", new { _certificate });
                throw new SecurityException("Invalid Certificate");
            }

            _key = Encryption.GenerateKey();
            return new ReverseHandshakeResponse
            {
                EncryptedKey = Encryption.EncryptKey(_key, _certificate)
            };
        }

        public AuthenticateResponse Authenticate(AuthenticateRequest request)
        {
            try
            {
                _apiKey = Encryption.Decrypt<string>(request.EncryptedApiKey, _key, request.InitializationVector);

                if (_apiKey != UpdateServiceConfiguration.Instance.ApiKey)
                {
                    Logger.Error("Invalid API key specified.");
                    return new AuthenticateResponse { Success = false };
                }

                _isSessionEstablished = true;
                return new AuthenticateResponse { Success = true };
            }
            catch (Exception)
            {
                return new AuthenticateResponse { Success = false };
            }
        }
    }
}
