﻿using System;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceModel.Channels;
using Curse.ServiceUpdate.UpdateManagement.Security;
using Curse.ServiceUpdate.WebClient.UpdateWebService;
using Newtonsoft.Json;

namespace Curse.ServiceUpdate.WebClient
{
    public static class UpdateCoordinatorHelper
    {
        private static Binding GetBinding(int sendTimeoutSeconds = 30)
        {
            var binding = new WSHttpBinding(SecurityMode.None, true)
            {
                OpenTimeout = TimeSpan.FromSeconds(5),
                SendTimeout = TimeSpan.FromSeconds(sendTimeoutSeconds),
                ReceiveTimeout = TimeSpan.FromSeconds(10),
                MaxReceivedMessageSize = Int32.MaxValue,
                ReaderQuotas = {MaxStringContentLength = Int32.MaxValue, MaxArrayLength = Int32.MaxValue, MaxBytesPerRead = Int32.MaxValue, MaxDepth = Int32.MaxValue}
            };

            return binding;
        }

        private static EndpointAddress GetRegistrationEndpoint(UpdateEnvironment environment)
        {
            string url;
            switch (environment)
            {
                case UpdateEnvironment.Debug:
                    url = "http://update-service.curse.dev/UpdateCoordinator.svc/registration";
                    break;
                case UpdateEnvironment.Staging:
                    url = "http://update-service.curse.stg/UpdateCoordinator.svc/registration";
                    break;
                case UpdateEnvironment.LoadTesting:
                    url = "http://update-service.curse.opt/UpdateCoordinator.svc/registration";
                    break;
                case UpdateEnvironment.Release:
                    url = "http://update-service.curse.us/UpdateCoordinator.svc/registration";
                    break;
                default:
                    throw new ArgumentException("Unknown environment: " + environment, "environment");
            }
            return new EndpointAddress(url);
        }

        public static RegisterHostResponse RegisterHost(UpdateEnvironment environment, string apiKey, RegisterHostBody body)
        {
            try
            {
                using (var client = new UpdateCoordinatorClient(GetBinding(), GetRegistrationEndpoint(environment)))
                {
                    var handshake = client.Handshake(new HandshakeRequest());
                    var key = Encryption.GenerateKey();
                    var pair = Encryption.EncryptData(apiKey, key);
                    var response = client.Authenticate(new AuthenticateRequest
                    {
                        EncryptedKey = Encryption.EncryptKey(key, new X509Certificate2(handshake.Certificate)),
                        EncryptedApiKey = pair.EncryptedData,
                        InitializationVector = pair.InitializationVector
                    });
                    if (!response.Success)
                    {
                        return new RegisterHostResponse(RegisterStatus.Error, "Failed Handshake");
                    }

                    pair = Encryption.EncryptData(JsonConvert.SerializeObject(body), key);
                    string statusMessage;
                    var status = client.RegisterHost(pair.InitializationVector, pair.EncryptedData, out statusMessage);
                    return new RegisterHostResponse(status, statusMessage);
                }
            }
            catch
            {
                return new RegisterHostResponse(RegisterStatus.Error, "Unexpected Exception");
            }
        }
    }
}
