﻿using System;
using System.IO;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using Curse.ServiceUpdate.UpdateManagement.Security;
using Curse.ServiceUpdate.WebClient.UpdateWebService;
using Newtonsoft.Json;

namespace Curse.ServiceUpdate.WebClient
{
    public static class UpdateDeployerHelper
    {

        private static WSDualHttpBinding GetBinding(int sendTimeoutSeconds = 30)
        {
            var binding = new WSDualHttpBinding()
            {
                OpenTimeout = TimeSpan.FromSeconds(5),
                SendTimeout = TimeSpan.FromSeconds(sendTimeoutSeconds),
                ReceiveTimeout = TimeSpan.FromSeconds(10),
            };

            binding.Security.Mode = WSDualHttpSecurityMode.None;

            return binding;
        }

        private static EndpointAddress GetDeploymentEndpoint(UpdateEnvironment environment)
        {
            string url;
            switch (environment)
            {
                case UpdateEnvironment.Debug:
                    url = "http://update-service.curse.dev/UpdateCoordinator.svc";
                    break;
                case UpdateEnvironment.Staging:
                    url = "http://update-service.curse.stg/UpdateCoordinator.svc";
                    break;
                case UpdateEnvironment.LoadTesting:
                    url = "http://update-service.curse.opt/UpdateCoordinator.svc";
                    break;
                case UpdateEnvironment.Release:
                    url = "http://update-service.curse.us/UpdateCoordinator.svc";
                    break;
                default:
                    throw new ArgumentException(string.Format("Unknown environment: {0}", environment), "environment");
            }

            return new EndpointAddress(url);
        }

        public static DeployUpdateResponse DeployUpdate(UpdateEnvironment environment, string apiKey, IUpdateDeployerCallback callback, DeployUpdateRequestBody body)
        {
            try
            {
                using (var client = new UpdateDeployerClient(new InstanceContext(callback), GetBinding(), GetDeploymentEndpoint(environment)))
                {
                    var handshake = client.Handshake(new HandshakeRequest());
                    var key = Encryption.GenerateKey();
                    var pair = Encryption.EncryptData(apiKey, key);
                    var response = client.Authenticate(new AuthenticateRequest
                    {
                        EncryptedKey = Encryption.EncryptKey(key, new X509Certificate2(handshake.Certificate)),
                        EncryptedApiKey = pair.EncryptedData,
                        InitializationVector = pair.InitializationVector
                    });

                    if (!response.Success)
                    {
                        return new DeployUpdateResponse(DeploymentStatus.Error, "Failed Handshake");
                    }

                    pair = Encryption.EncryptData(JsonConvert.SerializeObject(body), key);
                    using (var stream = new MemoryStream(pair.EncryptedData))
                    {
                        string statusMessage;
                        var status = client.DeployUpdate(pair.InitializationVector, stream, out statusMessage);
                        return new DeployUpdateResponse(status, statusMessage);
                    }
                }
            }
            catch
            {
                return new DeployUpdateResponse(DeploymentStatus.Error, "Unexpected error");
            }
        }
    }
}
