﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;

namespace Curse.ServiceUpdate.WebService.Data
{
    public class CurseServiceHost
    {
        public int ID { get; private set; }

        public int ServiceID { get; set; }

        public int RegionID { get; set; }

        public string Hostname { get; set; }
        
        public EnvironmentType Environment { get; set; }

        public DateTime DateUpdated { get; set; }

        public string LastRequestor { get; set; }

        public int? VersionID { get; set; }

        public CurseServiceHost()
        {
        }

        private CurseServiceHost(SqlDataReader reader)
        {
            ID = reader.GetInt32(0);
            ServiceID = reader.GetInt32(1);
            RegionID = reader.GetInt32(2);
            Hostname = reader.GetString(3);
            Environment = (EnvironmentType)reader.GetByte(4);
            DateUpdated = reader.GetDateTimeUtc(5);

            LastRequestor = reader.IsDBNull(6) ? null : reader.GetString(6);
            VersionID = reader.IsDBNull(7) ? (int?) null : reader.GetInt32(7);
        }

        public void SaveToDatabase()
        {
            using (SqlConnection connection = DatabaseHelper.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    if (ID == 0)
                    {
                        command.Parameters.AddWithValue("@RegionID", RegionID);
                        command.Parameters.AddWithValue("@Hostname", Hostname.ToLowerInvariant());
                        command.Parameters.AddWithValue("@Environment", (byte)Environment);
                        command.Parameters.AddWithValue("@ServiceID", ServiceID);
                        command.CommandText = "INSERT INTO [CurseServiceHost] (ServiceID, RegionID, Hostname, Environment) OUTPUT INSERTED.ID VALUES(@ServiceID, @RegionID, @Hostname, @Environment)";
                        ID = (int) command.ExecuteScalar();
                    }
                    else
                    {
                        command.Parameters.AddWithValue("@ID", ID);
                        command.Parameters.AddWithValue("@VersionID", VersionID.HasValue ? (object) VersionID.Value : DBNull.Value);
                        command.Parameters.AddWithValue("@LastRequestor", string.IsNullOrEmpty(LastRequestor) ? DBNull.Value : (object)LastRequestor);
                        command.CommandText = "UPDATE [CurseServiceHost] SET VersionID = @VersionID, LastRequestor = @LastRequestor, DateUpdated = GETUTCDATE(), Version = @Version WHERE ID = @ID";
                        command.ExecuteNonQuery();
                    }
                }
            }
        }

        public static CurseServiceHost[] GetAll(int serviceID, EnvironmentType environment)
        {
            var serviceHosts = new List<CurseServiceHost>();
            using (var connection = DatabaseHelper.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [CurseServiceHost] WHERE ServiceID = @ServiceID AND Environment = @Environment";
                    command.Parameters.AddWithValue("@ServiceID", serviceID);
                    command.Parameters.AddWithValue("@Environment", (byte) environment);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            serviceHosts.Add(new CurseServiceHost(reader));
                        }
                    }
                }
            }

            return serviceHosts.ToArray();
        }

        public static CurseServiceHost[] GetAllInRegion(int serviceID, int regionID, EnvironmentType environment)
        {
            var serviceHosts = new List<CurseServiceHost>();
            using (var connection = DatabaseHelper.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [CurseServiceHost] WHERE ServiceID = @ServiceID AND Environment = @Environment AND RegionID = @RegionID";
                    command.Parameters.AddWithValue("@ServiceID", serviceID);
                    command.Parameters.AddWithValue("@Environment", (byte) environment);
                    command.Parameters.AddWithValue("@RegionID", regionID);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            serviceHosts.Add(new CurseServiceHost(reader));
                        }
                    }
                }
            }

            return serviceHosts.ToArray();
        }

        public static CurseServiceHost GetByID(int id)
        {
            using (var connection = DatabaseHelper.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [CurseServiceHost] WHERE ID = @ID";
                    command.Parameters.AddWithValue("@ID", id);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        return reader.Read() ? new CurseServiceHost(reader) : null;
                    }
                }
            }
        }

        public static CurseServiceHost GetByServiceAndHostName(int serviceID, string hostname)
        {
            using (var connection = DatabaseHelper.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [CurseServiceHost] WHERE ServiceID = @ServiceID AND HostName = @HostName";
                    command.Parameters.AddWithValue("@ServiceID", serviceID);
                    command.Parameters.AddWithValue("@HostName", hostname.ToLowerInvariant());

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        return reader.Read() ? new CurseServiceHost(reader) : null;
                    }
                }
            }
        }
    }
}