﻿using System;
using System.Data;
using System.Data.SqlClient;
using Curse.ServiceUpdate.WebService.Messages;

namespace Curse.ServiceUpdate.WebService.Data
{
    public enum JobStatus
    {
        Pending,
        Running,
        Complete,
        Canceled
    }

    public class UpdateJob
    {
        public int ID { get; set; }
        public int ServiceID { get; set; }
        public EnvironmentType Environment { get; set; }
        public string Requestor { get; set; }
        public int VersionID { get; set; }
        public VerifyCommitMode VerifyCommitMode { get; set; }
        public DateTime DateCreated { get; set; }

        public JobStatus Status { get; set; }
        public DateTime DateModified { get; set; }
        public int TotalHosts { get; set; }
        public int CurrentProgress { get; set; }
        public int Successes { get; set; }
        public int Failures { get; set; }

        private UpdateJob()
        {

        }

        private UpdateJob(SqlDataReader reader)
        {
            ID = reader.GetInt32(0);
            ServiceID = reader.GetInt32(1);
            Environment = (EnvironmentType) reader.GetByte(2);
            Requestor = reader.GetString(3);
            VersionID = reader.GetInt32(4);
            VerifyCommitMode = (VerifyCommitMode) reader.GetByte(5);
            DateCreated = reader.GetDateTimeUtc(6);

            Status = (JobStatus) reader.GetByte(7);
            DateModified = reader.GetDateTimeUtc(8);
            TotalHosts = reader.GetInt32(9);
            CurrentProgress = reader.GetInt32(10);
            Successes = reader.GetInt32(11);
            Failures = reader.GetInt32(12);
        }

        public void SaveToDatabase(SqlConnection connection, SqlTransaction transaction)
        {
            using (var command = connection.CreateCommand())
            {
                command.Transaction = transaction;
                command.Parameters.AddWithValue("@DateModified", DateTime.UtcNow);
                command.Parameters.AddWithValue("@Status", (byte)Status);

                if (ID == 0)
                {
                    command.Parameters.AddWithValue("@VersionID", VersionID);
                    command.Parameters.AddWithValue("@TotalHosts", TotalHosts);
                    command.Parameters.AddWithValue("@ServiceID", ServiceID);
                    command.Parameters.AddWithValue("@Environment", (byte)Environment);
                    command.Parameters.AddWithValue("@Requestor", Requestor);
                    command.Parameters.AddWithValue("@VerifyCommitMode", (byte)VerifyCommitMode);
                    command.CommandText = "INSERT INTO [UpdateJob] (ServiceID, Environment, Requestor, VersionID, VerifyCommitMode, DateCreated, Status, DateModified, TotalHosts) OUTPUT INSERTED.ID VALUES(@ServiceID, @Environment, @Requestor, @VersionID, @VerifyCommitMode, GETUTCDATE(), @Status, GETUTCDATE(), @TotalHosts)";
                    ID = (int)command.ExecuteScalar();
                }
                else
                {
                    command.CommandText = "UPDATE [UpdateJob] SET DateModified = @DateModified, Status = @Status WHERE ID = @ID";
                    command.Parameters.AddWithValue("@ID", ID);
                    command.ExecuteNonQuery();
                }
            }
        }

        public void TrackSuccess(int targetID)
        {
            TrackChange(true, targetID, UpdateTargetStatus.Completed);
        }

        public void TrackFailure(int targetID, UpdateTargetStatus targetStatus = UpdateTargetStatus.Failed)
        {
            TrackChange(false, targetID, targetStatus);
        }

        private void TrackChange(bool success, int targetID, UpdateTargetStatus targetStatus = UpdateTargetStatus.Failed)
        {
            using (var connection = DatabaseHelper.GetConnection())
            {
                using (var transaction = connection.BeginTransaction())
                {
                    using (var command = connection.CreateCommand())
                    {
                        command.Transaction = transaction;
                        command.CommandText = "[TrackUpdateJobChange]";
                        command.CommandType = CommandType.StoredProcedure;

                        command.Parameters.AddWithValue("@Success", success);
                        command.Parameters.AddWithValue("@JobID", ID);
                        command.Parameters.AddWithValue("@TargetID", targetID);
                        command.Parameters.AddWithValue("@TargetStatus", (byte) targetStatus);

                        using (var reader = command.ExecuteReader())
                        {
                            if (reader.Read())
                            {
                                Successes = reader.GetInt32(0);
                                Failures = reader.GetInt32(1);
                                CurrentProgress = reader.GetInt32(2);
                            }
                        }
                    }

                    transaction.Commit();
                }
            }
        }

        public void ChangeStatus(JobStatus status)
        {
            if (Status == status)
            {
                return;
            }

            using (var connection = DatabaseHelper.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.Parameters.AddWithValue("@ID", ID);
                    command.Parameters.AddWithValue("@Status", status);
                    command.CommandText = "UPDATE [UpdateJob] SET Status = @Status, DateModified = GETUTCDATE() WHERE ID = @ID";
                    command.ExecuteNonQuery();
                }
            }
            Status = status;
        }

        public bool IsFinished()
        {
            return Status == JobStatus.Canceled || Status == JobStatus.Complete;
        }

        public static UpdateJob Create(CurseService service, CurseServiceHost[] hosts, UpdateVersion version, string requestor, VerifyCommitMode commitMode, EnvironmentType environment)
        {
            // Create a job, to roll this out
            using (SqlConnection conn = DatabaseHelper.GetConnection())
            {
                using (SqlTransaction transaction = conn.BeginTransaction())
                {
                    var job = new UpdateJob()
                    {
                        Status = JobStatus.Pending,
                        DateCreated = DateTime.UtcNow,
                        Environment = environment,
                        Requestor = requestor,
                        ServiceID = service.ID,
                        TotalHosts = hosts.Length,
                        VersionID = version.ID
                    };

                    job.SaveToDatabase(conn, transaction);

                    foreach (var host in hosts)
                    {
                        var target = new UpdateTarget()
                        {
                            ServiceHostID = host.ID,
                            RegionID = host.RegionID,
                            JobID = job.ID,
                            Status = UpdateTargetStatus.Pending,
                        };
                        target.SaveToDatabase(conn, transaction);
                    }

                    transaction.Commit();

                    return job;
                }
            }
        }

        public static UpdateJob GetByID(int id)
        {
            using (var conn = DatabaseHelper.GetConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [UpdateJob] WHERE ID = @ID";
                    command.Parameters.AddWithValue("@ID", id);

                    using (var reader = command.ExecuteReader())
                    {
                        return reader.Read() ? new UpdateJob(reader) : null;
                    }
                }
            }
        }
    }
}