﻿using System;
using System.Linq;
using System.Net;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceModel.Channels;
using Curse.Logging;
using Curse.ServiceUpdate.UpdateManagement;
using Curse.ServiceUpdate.UpdateManagement.Messages;
using Curse.ServiceUpdate.UpdateManagement.Security;
using Curse.ServiceUpdate.WebService.Configuration;
using Curse.ServiceUpdate.WebService.Data;
using Curse.ServiceUpdate.WebService.Messages;
using Newtonsoft.Json;
using AuthenticateRequest = Curse.ServiceUpdate.WebService.Messages.AuthenticateRequest;
using AuthenticateResponse = Curse.ServiceUpdate.WebService.Messages.AuthenticateResponse;

namespace Curse.ServiceUpdate.WebService
{
    [ServiceBehavior(InstanceContextMode = InstanceContextMode.PerSession, ConcurrencyMode = ConcurrencyMode.Reentrant)]
   public class UpdateCoordinator : IUpdateCoordinator, IUpdateCoordinatorRest, IUpdateDeployer
    {
        private byte[] _encryptionKey;
        private bool _sessionEstablished;

        public HandshakeResponse Handshake(HandshakeRequest request)
        {
            return new HandshakeResponse
            {
                Certificate = Global.WebServiceCertificate.Export(X509ContentType.Cert)
            };
        }

        public AuthenticateResponse Authenticate(AuthenticateRequest request)
        {
            try
            {
                _encryptionKey = Encryption.DecryptKey(request.EncryptedKey, Global.WebServiceCertificate);
                var apiKey = Encryption.Decrypt<string>(request.EncryptedApiKey, _encryptionKey, request.InitializationVector);
                _sessionEstablished = apiKey == CoordinatorConfiguration.Instance.ApiKey;
                return new AuthenticateResponse
                {
                    Success = _sessionEstablished
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[Authenticate] Unexpected exception");
                return new AuthenticateResponse {Success = false};
            }
        }

        public DeployUpdateResponse DeployUpdate(DeployUpdateRequest request)
        {
            if (!_sessionEstablished)
            {
                Logger.Error("Attempt to deploy an update without successfully handshaking first.");
                return new DeployUpdateResponse { Status = DeploymentStatus.Error, StatusMessage = "Unauthorized" };
            }

            try
            {
                var json = Encryption.Decrypt<string>(request.EncryptedBody, _encryptionKey, request.InitializationVector);
                return DeployUpdate(JsonConvert.DeserializeObject<DeployUpdateRequestBody>(json));
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[DeployUpdate] Unexpected exception.");
                return new DeployUpdateResponse {Status = DeploymentStatus.Error};
            }
        }

        public DeployUpdateResponse DeployUpdate(DeployUpdateRequestBody body)
        {
            if (!_sessionEstablished)
            {
                Logger.Error("Attempt to deploy an update without successfully handshaking first.");
                return new DeployUpdateResponse { Status = DeploymentStatus.Error, StatusMessage = "Unauthorized" };
            }

            try
            {
                Logger.Info("Deploying ");
                // Verify Payload Authenticity
                byte[] zipBytes = body.Payload;
                var codeCertificate = new X509Certificate2(body.PayloadCertificate);
                if (!codeCertificate.Verify() ||
                    !((RSACryptoServiceProvider)codeCertificate.PublicKey.Key).VerifyData(zipBytes, new SHA1CryptoServiceProvider(), body.PayloadSignature))
                {
                    return new DeployUpdateResponse { Status = DeploymentStatus.Error, StatusMessage = "Bad certificate or data" };
                }

                var requestor = ((RemoteEndpointMessageProperty) OperationContext.Current.IncomingMessageProperties[RemoteEndpointMessageProperty.Name]).Address;
                try
                {
                    var hostEntry = Dns.GetHostEntry(requestor);
                    requestor = hostEntry.HostName;
                }
                catch
                {
                    // ignore and just use IP
                }

                var environment = CoordinatorConfiguration.Mode;

                var service = CurseService.GetByName(body.ServiceName);

                // Determine targets
                CurseServiceHost[] hosts;
                if (body.HostID.HasValue)
                {
                    // only a specific host is to be updated
                    var host = CurseServiceHost.GetByID(body.HostID.Value);
                    hosts = host == null ? new CurseServiceHost[0] : new[] {host};
                }
                else if (body.RegionID.HasValue)
                {
                    // only a specific region is to be updated
                    hosts = CurseServiceHost.GetAllInRegion(service.ID, body.RegionID.Value, environment);
                }
                else
                {
                    // everything is to be updated
                    hosts = CurseServiceHost.GetAll(service.ID, environment);
                }

                // Create Job
                var versionValue = Version.Parse(body.ServiceVersion);
                var version = UpdateVersion.GetByServiceAndVersion(service.ID, versionValue);
                if (version == null)
                {
                    version = new UpdateVersion {ServiceID = service.ID, Version = versionValue};
                    version.SaveToDatabase();
                }
                var job = UpdateJob.Create(service, hosts, version, requestor, body.VerifyCommitMode, environment);

                var payload = UpdatePayload.GetByServiceAndVersion(job.ServiceID, job.VersionID);
                if (payload == null)
                {
                    payload = new UpdatePayload
                    {
                        ServiceID = job.ServiceID,
                        VersionID = job.VersionID,
                        Data = zipBytes
                    };
                }
                else
                {
                    payload.Data = zipBytes;
                }
                payload.SaveToDatabase();

                Rollout(job, service, version, payload);


                return new DeployUpdateResponse
                {
                    Status = DeploymentStatus.Success
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex,"[DeployUpdate] Unexpected exception.");
                return new DeployUpdateResponse
                {
                    Status = DeploymentStatus.Error,
                    StatusMessage = "Unknown"
                };
            }
        }

        private void Rollout(UpdateJob job, CurseService service, UpdateVersion version, UpdatePayload payload)
        {
            var targets = UpdateTarget.GetByJob(job.ID);

            var targetsByRegion = targets.GroupBy(t => t.RegionID).ToArray();

            var callback = OperationContext.Current.GetCallbackChannel<IUpdateDeployerCallback>();

            foreach (var regionGroup in targetsByRegion)
            {
                if (job.Status == JobStatus.Canceled)
                {
                    foreach (var target in targets.Where(t=>t.Status==UpdateTargetStatus.Pending))
                    {
                        job.TrackFailure(target.ID, UpdateTargetStatus.Canceled);
                    }
                    break;
                }
                DeployToRegion(job, regionGroup.ToArray(), service, version, payload, callback);
            }
        }

        private void DeployToRegion(UpdateJob job, UpdateTarget[] targets, CurseService service, UpdateVersion version, UpdatePayload payload, IUpdateDeployerCallback callback)
        {
            bool seedFound = false;
            int i = 0;
            for (;i<targets.Length;i++)
            {
                var potentialSeed = targets[i];
                if (seedFound || job.Status == JobStatus.Canceled)
                {
                    break;
                }

                var host = CurseServiceHost.GetByID(potentialSeed.ServiceHostID);
                if (host == null)
                {
                    continue;
                }

                try
                {
                    using (var seed = UpdaterSeedClient.Create(host.Hostname, 8080, Global.WebServiceCertificate, CoordinatorConfiguration.Instance.ApiKey))
                    {
                        var response = seed.UploadUpdate(new UploadRequestBody
                        {
                            ServiceName = service.ServiceName,
                            ServiceVersion = version.Version,
                            ZipStream = payload.Data
                        });


                        if (response.Status != UploadStatus.Success)
                        {
                            continue;
                        }

                        seedFound = true;
                        foreach (var target in targets)
                        {
                            if (job.Status == JobStatus.Canceled)
                            {
                                break;
                            }

                            var listener = CurseServiceHost.GetByID(target.ServiceHostID);
                            if (listener == null)
                            {
                                Logger.Warn("Update target has no registered listener.", target);
                                continue;
                            }

                            Logger.Debug("Deploying update", new {listener});
                            UpdateTargetHost(job, target, seed, new UpdateRequestBody
                            {
                                BaseInstallLocation = service.BaseInstallPathOverride,
                                InstallIfMissing = false,
                                ServiceName = service.ServiceName,
                                ServiceVersion = version.Version,
                                TargetHost = listener.Hostname,
                            }, job.VerifyCommitMode == VerifyCommitMode.All ||
                               (job.VerifyCommitMode == VerifyCommitMode.First && job.CurrentProgress == 0), callback);
                        }
                    }
                }
                catch
                {
                    // Could not use this target as a seed
                }
            }
            if (job.Status == JobStatus.Canceled)
            {
                Logger.Info("The update job has been canceled.", job);
                // Cancel all pending targets
                foreach (var target in targets.Where(t=>t.Status==UpdateTargetStatus.Pending))
                {
                    job.TrackFailure(target.ID, UpdateTargetStatus.Canceled);
                }
            }
            else if (!seedFound && i == targets.Length)
            {
                Logger.Warn("No seeds were found in the region.", new {job, targets});
                // Found no seeds in the region, fail all pending targets
                foreach (var target in targets)
                {
                    job.TrackFailure(target.ID);
                }
            }
            else
            {
                // fail anything still pending due to exceptions
                foreach (var target in targets.Where(t=>t.Status==UpdateTargetStatus.Pending))
                {
                    job.TrackFailure(target.ID);
                }
            }
        }

        private void UpdateTargetHost(UpdateJob job, UpdateTarget target, UpdaterSeedClient seed, UpdateRequestBody request, bool verifyCommit, IUpdateDeployerCallback callback)
        {
            if (target.Status != UpdateTargetStatus.Pending)
            {
                return;
            }

            target.Status = UpdateTargetStatus.Running;
            target.SaveToDatabase();

            var success = false;
            try
            {
                var updateResponse = seed.UpdateService(request);
                success = updateResponse.Status == UpdateStatus.Success;
            }
            catch(Exception ex)
            {
                // Unsuccessful
                Logger.Error(ex, "Failed to update a target.", target);
            }

            var notification = new UpdateCompleteNotification
            {
                Host = request.TargetHost,
                Success = success
            };

            if (notification.Success)
            {
                job.TrackSuccess(target.ID);
            }
            else
            {
                job.TrackFailure(target.ID);
            }

            if (!verifyCommit)
            {
                callback.NotifyUpdateComplete(notification);
                return;
            }

            try
            {
                var response = callback.VerifyCommit(notification);
                job.ChangeStatus(response.ContinueDeploy ? JobStatus.Running : JobStatus.Canceled);
            }
            catch (TimeoutException)
            {
                // Client isn't responding, so stop deploy
                job.ChangeStatus(JobStatus.Canceled);
            }
        }

        #region Session Creation

        public RegisterHostResponse RegisterHost(RegisterHostRequest request)
        {
            if (!_sessionEstablished)
            {
                Logger.Error("Attempt to register host without successfully handshaking first.");
                return new RegisterHostResponse { Status = RegisterStatus.Error, StatusMessage = "Unauthorized" };
            }
            try
            {
                var str = Encryption.Decrypt<string>(request.EncryptedBody, _encryptionKey, request.InitializationVector);
                return RegisterHost(JsonConvert.DeserializeObject<RegisterHostBody>(str));
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[RegisterHost] Unexpected exception.");
                return new RegisterHostResponse {Status = RegisterStatus.Error};
            }
        }

        public RegisterHostResponse RegisterHost(RegisterHostBody body)
        {
            if (!_sessionEstablished)
            {
                Logger.Error("Attempt to register host without successfully handshaking first.");
                return new RegisterHostResponse { Status = RegisterStatus.Error, StatusMessage = "Unauthorized" };
            }

            try
            {
                var service = CurseService.GetByName(body.ServiceName);
                if (service == null)
                {
                    service = new CurseService
                    {
                        ServiceName = body.ServiceName,
                        BaseInstallPathOverride = body.BaseInstallPathOverride
                    };
                    service.SaveToDatabase();
                }
                var host = CurseServiceHost.GetByServiceAndHostName(service.ID, body.Host);

                if (host == null)
                {
                    host = new CurseServiceHost
                    {
                        RegionID = GetRegionFromHost(body.Host),
                        Hostname = body.Host,
                        ServiceID = service.ID,
                        Environment = CoordinatorConfiguration.Mode
                    };
                    host.SaveToDatabase();
                }

                return new RegisterHostResponse {Status = RegisterStatus.Success};
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[RegisterHost] Unexpected exception.");
                return new RegisterHostResponse {Status = RegisterStatus.Error};
            }
        }

        private int GetRegionFromHost(string host)
        {
            var lowerhost = host.ToLowerInvariant();
            if (lowerhost.Contains("iad"))
            {
                return 1;
            }
            else if (lowerhost.Contains("dub"))
            {
                return 2;
            }
            else if (lowerhost.Contains("sin"))
            {
                return 3;
            }

            return 0;
        }

        #endregion

        #region REST

        public bool CancelUpdate(string jobID, string apiKey)
        {
            if (apiKey != CoordinatorConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt to cancel an update job without a valid API Key");
                return false;
            }

            try
            {
                var job = UpdateJob.GetByID(int.Parse(jobID));
                if (job == null || job.IsFinished())
                {
                    Logger.Warn("Attempted to cancel a non-existent or completed job", jobID);
                    return false;
                }
                job.ChangeStatus(JobStatus.Canceled);
                Logger.Info("Job was canceled.", job);
                return true;
            }
            catch
            {
                Logger.Error("Failed to cancel an update job.",jobID);
                return false;
            }
        }

        public UpdateStatusResponse Job(string jobID, string apiKey)
        {
            if (apiKey != CoordinatorConfiguration.Instance.ApiKey)
            {
                return new UpdateStatusResponse {Status = RequestStatus.Unauthorized};
            }
            try
            {
                var job = UpdateJob.GetByID(int.Parse(jobID));
                return new UpdateStatusResponse
                {
                    Status = job == null ? RequestStatus.NotFound : RequestStatus.Successful,
                    Job = job,
                    Targets = job == null ? null : UpdateTarget.GetByJob(job.ID)
                };
            }
            catch
            {
                return new UpdateStatusResponse {Status = RequestStatus.Error};
            }
        }


        public string Test()
        {
            return "Successful";
        }

        #endregion
    }
}
