﻿export abstract class BaseWebServiceClient {
    url: string;
    tokenJson: string;
        
    constructor(url: string, token?: string, userID?: number) {
        this.url = url;
        if (token != undefined) {
            this.tokenJson = JSON.stringify({ UserID: userID, Token: token });
        }
    }

    private getHeaders(): any {

        if (this.tokenJson != null) {
           return { AuthenticationToken: this.tokenJson };
        }

        return null;
    }
    
    public get<T>(action: string) : Promise<T> {           
        return this.request<T>("GET", action);
    }

    public post<T>(action: string, contract?: any) : Promise<T> {        
        return this.request<T>("POST", action, contract);
    }

    public patch<T>(action: string, contract?: any): Promise<T> {
        return this.request<T>("PATCH", action, contract);
    }

    public delete<T>(action: string, contract?: any): Promise<T> {        
        return this.request<T>("DELETE", action, contract);
    }

    public options<T>(action: string, contract?: any): Promise<T> {        
        return this.request<T>("OPTIONS", action, contract);
    }

    private request<T>(verb: string, action: string, contract?: any): Promise<T> {
        const fullUrl = action ? this.url + "/" + action : this.url;
        return this.makeRequest<T>(verb, fullUrl, this.getHeaders(), contract);
    }

    private dateRegex: RegExp = /^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2}(?:\.\d*)?)(?:Z|(\+|-)([\d|:]*))?$/;

    tryParseDate(value: any): any {
        const dateVal = this.dateRegex.exec(value);
        if (dateVal) {
            return new Date(value);
        }
        return value;
    }

    makeRequest<T>(verb: string, url: string, headers?: any, data?: any): Promise<T> {

        return new Promise<T>((resolve, reject) => {
            let req = new XMLHttpRequest();

            req.onload = () => {
                if (req.status >= 200 && req.status < 300) {
                    if (req.status === 204) {
                        resolve(true as any);
                    } else {
                        resolve(JSON.parse(req.responseText, (k, v) => {
                            return this.tryParseDate(v);
                        }));
                    }
                } 
                else {
                    reject(Error(req.statusText));
                }
            };


            req.onerror = error => {
                reject(error);
            };            

            req.open(verb, url);
            req.setRequestHeader("Content-type", "application/json; charset=utf-8");
            for (const header in headers.keys()) {
                req.setRequestHeader(header, headers[header]);
            }

            if (data) {
                req.send(JSON.stringify(data));
            } else {
                req.send();
            }
        });
    }
}