﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.Remoting;
using System.Text;
using System.Threading.Tasks;
using ProtoBuf;
using Curse.MurmurHash;

namespace ProtoGenerator
{
    class Program
    {
    

        static void Main(string[] args)
        {

            
            var path = args[0];
            var dest = args[1];
            var dir = new DirectoryInfo(path);
            var dlls = dir.GetFiles("*.dll");
            var dict = new Dictionary<string, string>();

            var processedMessageTypes = new HashSet<string>();

            var typeMap = new Dictionary<int, string>();

            foreach (var dll in dlls)
            {
                try
                {
                    var assembly = Assembly.LoadFrom(dll.FullName);
                    foreach (var type in assembly.GetTypes())
                    {
                        if (type == null ||type.Namespace == null)
                        {
                            continue;
                            
                        }

                        if (type.GetCustomAttribute<ProtoContractAttribute>() == null)
                        {
                            continue;
                        }

                        var proto = typeof(Serializer).GetMethod("GetProto").MakeGenericMethod(type).Invoke(null, null);
                        
                        if (!dict.ContainsKey(type.Namespace))
                        {
                            dict.Add(type.Namespace, "package " + type.Namespace + ";");
                            dict[type.Namespace] += Environment.NewLine + "import \"bcl.proto.txt\";" + Environment.NewLine;
                        }

                        var fileLines = proto.ToString().Split(new[] {Environment.NewLine}, StringSplitOptions.None);
                        var startingIndex = 0;
                        
                        // Skip the top section
                        while (fileLines[startingIndex].StartsWith("import") || fileLines[startingIndex].StartsWith("package"))
                        {
                            ++startingIndex;                            
                        }

                        var sanitized = new List<string>();
                        var isSkippingType = false;

                        foreach (var line in fileLines)
                        {
                            if (line.StartsWith("import") || line.StartsWith("package"))
                            {
                                continue;                                
                            }

                            if (isSkippingType)
                            {
                                if (line.StartsWith("}"))
                                {
                                    isSkippingType = false;
                                }
                                
                                continue;                                
                            }

                            if (line.StartsWith("message") || line.StartsWith("enum"))
                            {
                                var messageOrEnumName = line.Substring(line.IndexOf(" ") + 1, line.Length - line.IndexOf(" ") - 2);
                                if (processedMessageTypes.Contains(messageOrEnumName))
                                {
                                    isSkippingType = true;
                                    continue;
                                }

                                processedMessageTypes.Add(messageOrEnumName);

                            }

                            sanitized.Add(line);
                        }

                        
                        byte[] typeName = Encoding.ASCII.GetBytes(type.FullName);
                        uint code = MurmurHash2.GetHashCode(ref typeName, typeName.Length, 1);
                        var messageTypeID = (int)code;
                        typeMap[messageTypeID] = type.Namespace.Replace(".", "") + "." + type.Name;

                        var stripped = string.Join(Environment.NewLine, sanitized.ToArray());
                        
                        stripped = stripped.Replace(" [default = 00000000-0000-0000-0000-000000000000]", string.Empty);
                        dict[type.Namespace] += stripped;


                    }
                    
                }
                catch (Exception ex)
                {
                    Console.WriteLine("Failed to load " + dll.FullName + ": " + ex.Message);
                    continue;
                }
            }


            foreach (var kvp in dict)
            {
                File.WriteAllText(Path.Combine(dest, kvp.Key + ".proto.txt"), kvp.Value);
            }

            var sb = new StringBuilder();
            sb.AppendLine("Curse.MessageContractMap = {");
            sb.AppendLine("map: null,");
            sb.AppendLine("initialize: function() { Curse.MessageContractMap.map = new {");

            foreach (var kvp in typeMap)
            {
                sb.AppendFormat("\"{0}\"", kvp.Key);                
                sb.Append(": ");
                sb.AppendFormat("{0}", kvp.Value);                
                sb.AppendLine(",");

                sb.AppendFormat("\"{0}\"", kvp.Value);
                sb.Append(": ");
                sb.AppendFormat("\"{0}\"", kvp.Key);
                sb.AppendLine(",");

            }
            sb.AppendLine("}");
            sb.AppendLine("}");
            sb.Append("}");

            File.WriteAllText(Path.Combine(dest, "Curse.MessageContractMap.js"), sb.ToString());
        }
    }
}
