﻿using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.IO;
using System.Net;
using System.Text;
using Newtonsoft.Json;

namespace Curse.ServiceClients
{
    public abstract class BaseWebServiceClient
    {
        private readonly string _url;

        public string Url
        {
            get { return _url; }
        }

        protected BaseWebServiceClient(string url)
        {
            if (string.IsNullOrEmpty(url))
            {
                throw new ArgumentNullException("url", "Url must be supplied for service: " + GetType().Name);
            }

            _url = url;
        }

        public static ServicesConfiguration Configuration = new ServicesConfiguration();

        IDictionary<string, string> GetHeaders()
        {
            if (ServiceAuthentication.Token == null) return null;
            var headers = new Dictionary<string, string>
            {
                {"Authorization", $"OAuth {ServiceAuthentication.Token}"},
                {"X-Twitch-Id", ServiceAuthentication.UserId}
            };

            return headers;
        }

        protected virtual HttpWebRequest GetWebRequest(string method, string uri)
        {
            var request = (HttpWebRequest) WebRequest.Create(uri);
            request.Method = method;
            request.UserAgent = Configuration.UserAgent;
            request.Accept = Configuration.ContentType;
            request.Timeout = Configuration.TimeoutSeconds*1000;
            request.AutomaticDecompression = Configuration.Compression;

            if (Configuration.DisableProxyDetection)
            {
                request.Proxy = null;
            }

            var additionalHeaders = GetHeaders();
            if (additionalHeaders != null)
            {
                foreach (var kvp in additionalHeaders)
                {
                    request.Headers[kvp.Key] = kvp.Value;
                }
            }

            return request;
        }

        private string GetUrl(string action)
        {
            return _url + "/" + action;
        }

        protected virtual ServiceResponse Patch(string action, object body)
        {
            throw new NotImplementedException();
        }


        protected virtual ServiceResponse<TResponse> Post<TResponse>(string action)
        {
            return MakeRequest<TResponse>("POST", action);
        }

        protected virtual ServiceResponse<TResponse> Post<TResponse>(string action, object body)
        {
            return MakeRequest<TResponse>("POST", action, body);
        }

        protected virtual ServiceResponse Post(string action)
        {
            return MakeRequest("POST", action);
        }

        protected virtual ServiceResponse Post(string action, object body)
        {
            return MakeRequest("POST", action, body);
        }

        protected virtual ServiceResponse Delete(string action)
        {
            return MakeRequest("DELETE", action);
        }

        protected virtual ServiceResponse<TResponse> Delete<TResponse>(string action)
        {
            return MakeRequest<TResponse>("DELETE", action);
        }

        protected virtual ServiceResponse<TResponse> Get<TResponse>(string action)
        {
            return MakeRequest<TResponse>("GET", action);
        }

        protected virtual NameValueCollection GetCustomHeaders()
        {
            return null;
        }
        
        protected virtual string SerializeObject(object body)
        {
            return JsonConvert.SerializeObject(body);
        }

        protected ServiceResponse MakeRequest(string method, string action, object body = null)
        {
            var url = GetUrl(action);

            try
            {
                var request = GetWebRequest(method, url);

                HttpWebResponse response;
                using (new SimpleTimer(action, "Request"))
                {
                    response = SendRequest(request, body);
                }

                if (response == null)
                {
                    return new ServiceResponse(HttpStatusCode.InternalServerError);
                }

                return new ServiceResponse(response.StatusCode);
            }
            catch (Exception ex)
            {
                ServicesConfiguration.LogWarning(ex, string.Format("Service call failed to: {0} {1}", method, url));
                return new ServiceResponse(HttpStatusCode.InternalServerError);
            }
        }

        protected ServiceResponse<TResponse> MakeRequest<TResponse>(string method, string action, object body = null)
        {
            var url = GetUrl(action);

            try
            {
                var request = GetWebRequest(method, url);

                string responseContent = null;

                HttpWebResponse response;
                using (new SimpleTimer(action, "Request"))
                {
                    response = SendRequest(request, body);
                }
                if (response == null)
                {
                    return new ServiceResponse<TResponse>(HttpStatusCode.InternalServerError);
                }

                try
                {
                    if ((int)response.StatusCode < 200 || (int)response.StatusCode >= 300)
                    {
                        return new ServiceResponse<TResponse>(response.StatusCode);
                    }

                    var responseStream = response.GetResponseStream();
                    if (responseStream != null)
                    {
                        using (var streamReader = new StreamReader(responseStream))
                        {
                            responseContent = streamReader.ReadToEnd();
                        }
                    }

                    TResponse value;
                    using (new SimpleTimer(action, "Deserialization"))
                    {
                        value = JsonConvert.DeserializeObject<TResponse>(responseContent);
                    }

                    return new ServiceResponse<TResponse>(response.StatusCode, value);
                }
                finally
                {
                    response.Close();
                }
            }
            catch (Exception ex)
            {
                ServicesConfiguration.LogWarning(ex, string.Format("Service call failed to: {0} {1}", method, url));
                return new ServiceResponse<TResponse>(HttpStatusCode.InternalServerError);
            }
        }

        protected HttpWebResponse SendRequest(HttpWebRequest request, object body = null)
        {
            try
            {
                request.ContentType = "application/json";
                if (body == null)
                {
                    request.ContentLength = 0L;
                }
                else
                {
                    var serialized = this.SerializeObject(body);
                    var content = Encoding.UTF8.GetBytes(serialized);
                    request.ContentLength = content.Length;
                    using (var requestStream = request.GetRequestStream())
                    {
                        requestStream.Write(content, 0, content.Length);
                    }
                }

                return (HttpWebResponse)request.GetResponse();
            }
            catch (WebException ex)
            {
                return ex.Response as HttpWebResponse;
            }
            catch (Exception ex)
            {
                ServicesConfiguration.LogWarning(ex, "Service call failed to");
                return null;
            }
        }
    }
}