﻿using System;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using Curse.ServiceClients.Contracts;
using System.Threading;
using Newtonsoft.Json;
using Newtonsoft.Json.Converters;

namespace Curse.ServiceClients.Tester
{
    class Program
    {

        private static readonly Guid MachineKey = new Guid("936DA01F-9ABD-4d9d-80C7-02AF85C822A8");
        private static readonly string DeviceID = "TEST";

        private static string GetPassword()
        {
            var pwd = string.Empty;
            while (true)
            {
                var i = Console.ReadKey(true);
                if (i.Key == ConsoleKey.Enter)
                {
                    break;
                }
                else if (i.Key == ConsoleKey.Backspace)
                {
                    if (pwd.Length > 0)
                    {
                        pwd = pwd.Substring(pwd.Length - 1);
                        Console.Write("\b \b");
                    }
                }
                else
                {
                    pwd = pwd + (i.KeyChar);
                    Console.Write("*");
                }
            }
            return pwd;
        }

        static void Main(string[] args)
        {           
            ServicesConfiguration.SetDefaultUrls();
            ServicesConfiguration.DisableCertificateValidation();
            
            Console.WriteLine("Username: ");
            var username = Console.ReadLine();

            Console.WriteLine("Password: ");
            var password = GetPassword();

            Console.WriteLine();
            Console.Write("Logging in...");

            // Login
            var resp = Services.Instance.Login.Login(new LoginRequest { Username = username, Password = password });
            

            if (!resp.Success)
            {
                Console.WriteLine("Login failed: " + resp.StatusCode);
                Console.ReadKey(true);
                return;
            }

            ServiceAuthentication.Token = resp.Response.Session.Token;

            TestGroupNickname();

            Console.ReadKey(true);
        }

        static void TestAdHocCall()
        {
            Console.Write("Creating an ad hoc call...");

            var resp = Services.Instance.Calls.AdHoc(new AdHocCallRequest {ClientVersion = "7.0.0.0", UserDisplayName = "Adamar" });
            if (!resp.Success)
            {
                Console.WriteLine("Failed: " + resp.StatusCode);
            }

            Console.WriteLine("Success!");
            Console.WriteLine(resp.Response.InviteUrl);
            
        }

        static void TestAggregateStats()
        {
            var resp = Services.Instance.Stats.GetAggretateStats(0);
            Console.WriteLine(resp.Success);
        }

        static void TestClaimingAccount()
        {
            var claimResp = Services.Instance.Register.ClaimTempAccount(new ClaimTempRequest
            {
                Email = "mcomperda@somethign.com",
                Password = "password1234!",
                TempAccountToken = "6506528d-bae5-4fc8-ae4d-6a94f1d17f92"
            });

            if (!claimResp.Success)
            {
                Console.WriteLine("Failed: " + claimResp.StatusCode);
            }
        }

        static void TestMassTempAccounts()
        {
            for (var i = 0; i < 20; i++)
            {
                var testUsername = Guid.NewGuid().ToString().Replace("-", "").Substring(0, 32);
                var getResp = Services.Instance.Register.RegisterTempAccount(new RegisterTempRequest { Username = testUsername });

                if (!getResp.Success)
                {
                    Console.WriteLine("Failed to get settings: " + getResp.StatusCode);                    
                }

                Thread.Sleep(1000);
            }
            
        }

        static void TestTempAccounts()
        {            
            var testUsername = Guid.NewGuid().ToString().Replace("-", "").Substring(0, 32);
            var getResp = Services.Instance.Register.RegisterTempAccount(new RegisterTempRequest { Username = testUsername });

            if (!getResp.Success)
            {
                Console.WriteLine("Failed to get settings: " + getResp.StatusCode);
                return;
            }

            Console.WriteLine("Enter an email address");
            var testEmail = Console.ReadLine();

            var claimResp = Services.Instance.Register.ClaimTempAccount(new ClaimTempRequest
                {
                    Email = testEmail,
                    Password = "password1234!",
                    TempAccountToken = getResp.Response.TempAccountToken
                });

            if (!claimResp.Success)
            {
                Console.WriteLine("Failed: " + claimResp.StatusCode);
            }
        }


        static void TestServerSettings()
        {

            var getResp = Services.Instance.Servers.GetGeneralSettings(new Guid("ff22ca3d-ccbd-42a2-a5d1-77b86cdfc0a0"));
            if (!getResp.Success)
            {
                Console.WriteLine("Failed to get settings: " + getResp.StatusCode);
                return;;
            }

            var settings = getResp.Response;
            Console.WriteLine(settings.Title);

            var counter = 0;
            while (true)
            {
                var newTitle = ++counter % 2 == 0 ? "Noah's Lair" : "Noah's Dungeon";
                var resp = Services.Instance.Servers.SetGeneralSettings(new Guid("ff22ca3d-ccbd-42a2-a5d1-77b86cdfc0a0"), new ServerGeneralSettingsContract { Title = newTitle, IsPublic = false, VoiceRegion = 0, AfkChannel = null, AfkTimeMinutes = 10 });
                Console.WriteLine(resp.Success);
                Thread.Sleep(1000);
    
            }            
        }

        static void TestServerSearchSettings()
        {
            var getResp = Services.Instance.Servers.GetSearchSettings(new Guid("ff22ca3d-ccbd-42a2-a5d1-77b86cdfc0a0"));
            if (!getResp.Success)
            {
                Console.WriteLine("Failed to get search settings: " + getResp.StatusCode);
                return; ;
            }

            var settings = getResp.Response;
            Console.WriteLine(settings.Description);

            while (true)
            {
                var resp = Services.Instance.Servers.SetSearchSettings(new Guid("ff22ca3d-ccbd-42a2-a5d1-77b86cdfc0a0"), new ServerSearchSettingsContract { Description = "This is my awesome lair " + DateTime.UtcNow, Games = new[] { 1 }, IsSearchable = true, MatchAllGames = false, SearchTags = new[] { GroupSearchTag.Hardcore } });
                if (!resp.Success)
                {
                    Console.WriteLine("Failed to update search settings: " + getResp.StatusCode);
                }

                getResp = Services.Instance.Servers.GetSearchSettings(new Guid("ff22ca3d-ccbd-42a2-a5d1-77b86cdfc0a0"));
                settings = getResp.Response;
                Console.Write(settings.Description);
                Console.ReadKey(true);
            }            
        }

        static void TestCreatingBugReport()
        {
            var bugReport = new BugReport
            {
                Title = "Test Bug Report",
                ClientVersion = "7.0.0.0",
                Platform = DevicePlatform.Windows,
                Type = BugReportType.BugOrDefect,
                Message = "Hello world!",
                Attachments = new[] { new BugReportAttachment { FileContents = Encoding.Default.GetBytes("This is a test, please work!"), FileName = "Test.txt" } }
            };

            Console.Write("Submitting bug report... ");
            var bugResp = Services.Instance.Bugs.Report(bugReport);
            Console.WriteLine("Done: " + bugResp.StatusCode);
        }

        static void TestCreatingFeedback()
        {
            var feedback = new Feedback
            {
                Platform = DevicePlatform.Windows,
                ClientVersion = "7.0.0.0",
                FeedbackType = FeedbackType.Positive,
                Message = "Your app rocks!"
            };

            Console.Write("Submitting feedback... ");
            var bugResp = Services.Instance.Feedback.Report(feedback);
            Console.WriteLine("Done: " + bugResp.StatusCode);
        }

        static void TestCreatingServer()
        {
            Console.Write("Creating a new server...");

            var resp = Services.Instance.Servers.Create(new CreateServerRequest
            {
                GuestRoleName = "Guests",
                OwnerRoleName = "Owners",
                ModeratorRoleName = "Moderators", 
                TextChannelTitle = "Lobby",
                Title = "Testing Server",
                VoiceChannelTitle = "Calls", 
                IsPublic = false
            });

            if (!resp.Success)
            {
                Console.WriteLine("Failed: " + resp.StatusCode);
                Console.ReadKey(true);
                return;
            }
            
            Console.WriteLine("Success!");

            var channels = Services.Instance.Groups.Details(resp.Response.GroupID, false);

            foreach (var channel in channels.Response.Channels)
            {
                Console.WriteLine(channel.GroupTitle);
            }

            
            Console.Write("Creating a channel... ");

            var channelResp = Services.Instance.Servers.CreateChannel(resp.Response.RootGroupID,
                new CreateChannelRequest { Mode = GroupMode.TextOnly, Title = "Owners Only!", IsPublic = false, AccessRoles = new [] { 1 } });

            Console.WriteLine("Done: " + channelResp.StatusCode);
            
            if (!channelResp.Success)
            {
                Console.WriteLine("Failed to create channel: " + channelResp.StatusCode);
                Console.ReadKey(true);
                return;
            }

            Console.WriteLine("Sleeping 1 seconds...");
            Thread.Sleep(1000);

            channels = Services.Instance.Groups.Details(resp.Response.GroupID, false);

            foreach (var channel in channels.Response.Channels)
            {
                Console.WriteLine(channel.GroupTitle);
            }

            Console.Write("Deleting channel...");

            var deleteResp = Services.Instance.Servers.DeleteChannel(resp.Response.GroupID, channelResp.Response.GroupID);

            if (!deleteResp.Success)
            {
                Console.WriteLine("Failed to delete channel: " + channelResp.StatusCode);
                Console.ReadKey(true);
                return;
            }

            Console.WriteLine("Done: " + deleteResp.StatusCode);
            Console.WriteLine("Sleeping 1 seconds...");
            Thread.Sleep(1000);

            channels = Services.Instance.Groups.Details(resp.Response.GroupID, false);

            foreach (var channel in channels.Response.Channels)
            {
                Console.WriteLine(channel.GroupTitle);
            }
        }

        static void TestRoleRanks()
        {
            var groups = Services.Instance.Contacts.GetAll().Response.Groups.Where(p => p.GroupType == GroupType.Large && p.GroupID == p.RootGroupID).ToArray();
            foreach (var group in groups)
            {
                if (group.Membership.BestRole == 1)
                {
                    Console.WriteLine(group.GroupTitle + " (" + group.GroupID + ")");
                }
            }

            Console.WriteLine("Enter a group ID:");

            var choice = Console.ReadLine();
            if (string.IsNullOrEmpty(choice))
            {
                return;
            }
            
            var groupID = Guid.Parse(choice);
            var getRolesResp = Services.Instance.Servers.GetRoles(groupID);
            if (!getRolesResp.Success)
            {
                Console.WriteLine("Failed to get roles: " + getRolesResp.StatusCode);
                Console.ReadKey(true);
                TestRoleRanks();
                return;
            }

            var roles = getRolesResp.Response;
            foreach (var role in roles.OrderBy(p => p.Role.Rank))
            {
                Console.WriteLine(role.Role.Name + " (ID: " + role.Role.RoleID + ", Rank: " + role.Role.Rank + ")");
            }

            var roleRanks = roles.ToDictionary(p => p.Role.RoleID, p => p.Role.Rank);

            try
            {
                Console.WriteLine("Enter a role ID:");
                var roleID = int.Parse(Console.ReadLine());

                Console.WriteLine("Enter a new rank:");
                var rank = int.Parse(Console.ReadLine());

                roleRanks[roleID] = rank;

                var resp = Services.Instance.Servers.ModifyRoleRanks(groupID, roleRanks);
                if (resp.Success)
                {
                    foreach (var role in resp.Response.OrderBy(p => p.Role.Rank))
                    {
                        Console.WriteLine(role.Role.Name + " (ID: " + role.Role.RoleID + ", Rank: " + role.Role.Rank + ")");
                    }
                }
                else
                {
                    Console.WriteLine("Failed: " + resp.StatusCode);
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("Failed: " + ex.Message);
            }
            

            Console.WriteLine("Press any key to continue...");
            Console.ReadKey(true);
            TestRoleRanks();
        }

        static void TestIdleness()
        {
            var resp = Services.Instance.Account.SetIdle(new SetIdleRequest { MachineKey = MachineKey});
            Console.WriteLine("Set Idle Resp: " + resp.Success);
            Console.WriteLine("Waiting 5 seconds...");

            resp = Services.Instance.Account.SetActive(new SetActiveRequest { MachineKey = MachineKey });
            Console.WriteLine("Set Active Resp: " + resp.Success);
            Console.WriteLine("Done!");
            Console.ReadLine();
        }

        static void TestGameStatus()
        {
            var resp = Services.Instance.Account.ChangeGameStatus(new ChangeGameStatusRequest { GameID = 1, GameState = 1, GameStatusMessage = null, IsRunning = true, MachineKey = MachineKey.ToString()});
            Console.WriteLine("Change game status resp: " + resp.Success);
            Console.WriteLine("Waiting 5 seconds...");

            resp = Services.Instance.Account.ChangeGameStatus(new ChangeGameStatusRequest { GameID = 0, GameState = 1, GameStatusMessage = null, IsRunning = false, MachineKey = MachineKey.ToString() });
            Console.WriteLine("Change game status resp: " + resp.Success);
            Console.WriteLine("Done!");
            Console.ReadLine();
        }

        static void TestFriendSync()
        {            
            var resp = Services.Instance.FriendSync.Get();
            Console.WriteLine("Get friend sync hints..." + resp.Success);
            Console.WriteLine("Waiting 5 seconds...");

            // Change the status of one
            resp.Response.FirstOrDefault().Status = FriendHintStatus.Normal;            

            var uploadResp = Services.Instance.FriendSync.Upload(resp.Response);
            Console.WriteLine("Updload friend hints..." + resp.Success);

            var multiSearchReq = Services.Instance.FriendSync.MultiSearch(new FriendListMultiSearchRequest { FriendLists = new [] { new FriendListSearchRequest { FriendsList = new FriendHintContract[0], Identity = null} }});
        }

        static void TestUsageReporting()
        {
            var resp = Services.Instance.Reporting.Usage(new UsageProfile
            {
                ClientVersion = "7.0.0.0", 
                DotNetVersion = 4.5f,
                MainWindowHeight = 800,
                MainWindowWidth = 600,
                OperatingSystemPlatform = "Windows",
                OperatingSystemVersion = "Windows 10",
                Platform = DevicePlatform.Windows, 
                SoftwareRendering = false,                 
            });
        }

        static void TestAddingUserToServer()
        {
            var resp = Services.Instance.Groups.RemoveMember(Guid.Parse("88c9c6bd-00a7-4378-8715-3348a7e752f2"),817369);
            var aresp = Services.Instance.Groups.AddMembers(Guid.Parse("88c9c6bd-00a7-4378-8715-3348a7e752f2"), new[] { 817369 });

        }

        static void TestGroupNickname()
        {
            Console.WriteLine("Enter a server id: ");
            var guid = Guid.Parse(Console.ReadLine());


            while (true)
            {
                Console.WriteLine("Enter a nickname: ");
                var name = Console.ReadLine();

                if (string.IsNullOrWhiteSpace(name))
                {
                    break;
                }

                var resp = Services.Instance.Groups.Nickname(guid, new GroupMemberNicknameRequest {Nickname = name});
                Console.WriteLine(resp.Success);
            }

        }
    }
}
