﻿using System;
using System.Web;
using System.Threading;
using System.Data.SqlClient;
using System.Collections.Generic;
using AddOnService.Models;
using Curse;
using System.Data;
using System.Configuration;

namespace AddOnService
{
    public static class AddOnReporting
    {
        private const int WRITE_INTERVAL = 5000;

        public static List<InstalledAddon> sInstalledAddons = new List<InstalledAddon>();
        public static List<UserLog> sLoggedUsers = new List<UserLog>();

        public static int sUpdateThreadInterval = 10000; // 10 Second Thread
        private static DataTable sInstalledAddonsTable = null;
        private static string sConnectionString = null;
        private static Thread sWriteToDatabaseThread = null;

        public static void Initialize()
        {
            sInstalledAddonsTable = new DataTable();
            sConnectionString = ConfigurationManager.ConnectionStrings["InstallsDB"].ConnectionString;
            using (SqlConnection conn = new SqlConnection(sConnectionString))
            {
                conn.Open();
                DBUtility.LoadSchema(ref sInstalledAddonsTable, "installed_addon", conn);
            }
            sWriteToDatabaseThread = new Thread(WriteToDatabaseThread);
            sWriteToDatabaseThread.Start();
        }

        public static void ReportUserAddons(int userID, List<InstalledAddon> userAddons)
        {
            lock (sInstalledAddons)
            {
                sInstalledAddons.RemoveAll(p => p.UserID == userID);
                sInstalledAddons.AddRange(userAddons);
            }
        }

        public static void LogUser(int userID, string userAgent)
        {
            UserLog userLog = new UserLog();
            userLog.Date = DateTime.UtcNow;
            userLog.UserID = userID;
            userLog.UserAgent = userAgent;

            lock (sLoggedUsers)
            {
                sLoggedUsers.RemoveAll(p => p.UserID == userID);
                sLoggedUsers.Add(userLog);
            }
        }

        public static void WriteToDatabaseThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                //Every 5 seconds
                Thread.Sleep(WRITE_INTERVAL);
                try
                {
                    WriteToDatabase();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                }
                catch (Exception ex)
                {
                    Logger.Log(ELogLevel.Error,
                           null,
                           "AddOnReporting Error: " + ex.Message + Environment.NewLine + "Stack:" + ex.StackTrace);      
                }
            }

        }

        private static void WriteToDatabase()
        {
            List<InstalledAddon> installedAddons = null;

            lock (sInstalledAddons)
            {
                installedAddons = new List<InstalledAddon>(sInstalledAddons);
                sInstalledAddons.Clear();
            }

            List<UserLog> loggedUsers = null;

            lock (sLoggedUsers)
            {
                loggedUsers = new List<UserLog>(sLoggedUsers);
                sLoggedUsers.Clear();
            }
            SaveLoggedUsers(loggedUsers);
            SaveInstalledAddons(installedAddons);
            

        }

        private static void SaveLoggedUsers(List<UserLog> loggedUsers)
        {

            if (loggedUsers.Count == 0)
            {
                return;
            }

            List<string> userAgents = new List<string>();                                
            DataTable userLogTable = new DataTable();
            userLogTable.Columns.Add("user_id", typeof(int));
            userLogTable.Columns.Add("first_seen", typeof(DateTime));
            userLogTable.Columns.Add("last_seen", typeof(DateTime));
            userLogTable.Columns.Add("user_agent", typeof(string));

            foreach (UserLog log in loggedUsers)
            {               
                bool contains = null != userAgents.Find(delegate(string str)
                {
                    return str.ToLower().Equals(log.UserAgent.ToLower());
                });

                if (!contains)
                {
                    userAgents.Add(log.UserAgent);
                }

                DataRow row = userLogTable.NewRow();
                row["user_id"] = log.UserID;
                row["first_seen"] = log.Date;
                row["last_seen"] = log.Date;
                row["user_agent"] = log.UserAgent;
                userLogTable.Rows.Add(row);
            }
            

            using (SqlConnection conn = new SqlConnection(sConnectionString))
            {
                conn.Open();
                SqlCommand cmd = conn.CreateCommand();

                // Create any missing user agents:
                cmd.Parameters.Add("@version_name", SqlDbType.VarChar, 64);
                cmd.CommandText = "insert into client_version(version_name) select @version_name where not exists(select top 1 1 from client_version as b where lower(b.version_name) = lower(@version_name));";                    
                foreach (string userAgent in userAgents)
                {                    
                    cmd.Parameters["@version_name"].Value = userAgent;
                    cmd.ExecuteNonQuery();
                }

                // Save out the log:
                cmd = conn.CreateCommand();
                cmd.CommandText = "create table #logged_user (user_id int, first_seen datetime, last_seen datetime, user_agent varchar(64) );";
                cmd.ExecuteNonQuery();
                using (SqlBulkCopy bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, null))
                {
                    bulk.BulkCopyTimeout = 300;
                    bulk.DestinationTableName = "#logged_user";
                    bulk.WriteToServer(userLogTable);
                }

                // Update existing
                cmd.CommandText = "update logged_user" +
                    " set last_seen = tmp.last_seen, " +
                    " version_id = (select version_id from client_version with(nolock) where lower(client_version.version_name) = lower(tmp.user_agent))" +
                    "from logged_user, #logged_user as tmp where logged_user.user_id=tmp.user_id;";

                cmd.CommandTimeout = 300;
                cmd.ExecuteNonQuery();

                //Insert new
                cmd.CommandText = "insert into logged_user select tmp.user_id, tmp.first_seen, tmp.last_seen, (select version_id from client_version with(nolock) where lower(client_version.version_name) = lower(tmp.user_agent)) from #logged_user as tmp where not exists(select top 1 1 from logged_user where logged_user.user_id = tmp.user_id)";
                cmd.CommandTimeout = 300;
                cmd.ExecuteNonQuery();
               
            }
        }

        private static void SaveInstalledAddons(List<InstalledAddon> installedAddons)
        {

            if (installedAddons.Count == 0)
            {
                return;
            }

            DataTable installedAddonsTable = sInstalledAddonsTable.Clone();
            installedAddonsTable.BeginLoadData();

            foreach (InstalledAddon install in installedAddons)
            {
                DataRow row = installedAddonsTable.NewRow();
                row["user_id"] = install.UserID;
                row["addon_id"] = install.AddOnID;
                row["posted"] = install.Posted;
                installedAddonsTable.Rows.Add(row);
            }

            using (SqlConnection conn = new SqlConnection(sConnectionString))
            {
                conn.Open();
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "create table #installed_addon (user_id int, addon_id int, posted datetime);";                
                cmd.ExecuteNonQuery();
                using (SqlBulkCopy bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, null))
                {
                    bulk.BulkCopyTimeout = 300;
                    bulk.DestinationTableName = "#installed_addon";
                    bulk.WriteToServer(installedAddonsTable);
                }
                cmd.CommandText = "delete from installed_addon where exists(select top 1 1 user_id from #installed_addon as a where a.user_id = installed_addon.user_id);" +
                    "insert into installed_addon select * from #installed_addon;";
                cmd.CommandTimeout = 300;

                cmd.ExecuteNonQuery();
            }
        }
    }
}
