﻿using Curse;
using Curse.Auth;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.IO;
using System.Net;
using System.Security.Cryptography;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using System.Web;
using System.Web.Services;
using System.Web.Services.Protocols;
using System.Web.Caching;
using System.Threading;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters;
using System.Runtime.Serialization.Formatters.Binary;
using AddOnService.Models;

namespace AddOnService
{
    /**
     * Roaming Profile Front-End Web Service Class
     * Inherits AuthenticatableWebService
     *
     * @author Michael Comperda
     */
    [WebService(Namespace = "http://addonservice.curse.com/")]
    [WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
    [ToolboxItem(false)]
    public class AddOnService
        : AuthenticatableWebService
    {


        private static bool sIsStarted = false;
        //private static bool sIsFirstRun = true;
        private static string sRoamingDB = null;                
        private static string[] sLocaleKeys = null;
        private static readonly string sWebRootPath = null;
        private static Dictionary<int, AddOn> sAddOnCache = new Dictionary<int,AddOn>();
        
        private static Dictionary<string, byte[]> sFileKeyCache = new Dictionary<string, byte[]>();        
        private static byte[] sUpdateFeedBytes = {};
        private static byte[] sCompleteFeedBytes = {};
        private static byte[] sExceptionFeedBytes = {};
        private static Thread sUpdateThread = null;
        private static Thread sNewsFeedThread = null;
        private static int sUpdateThreadInterval;
        private static DateTime sLastQueryTime = DateTime.Parse("1/1/1977 12:00:00 AM");
        private static Regex sValidVersion = new Regex("[0-9\\.]", RegexOptions.Compiled);
                
        /**
         * Static constructor
         * Does one-time initialization of the profile service
         */
        static AddOnService()
        {
            Logger.SetAutoGrowthMegabytes = 100; // 100 Megabytes            
            Logger.SetLogLevel = (ELogLevel)Array.IndexOf<String>(Enum.GetNames(typeof(ELogLevel)), ConfigurationManager.AppSettings["LogLevel"]);
            Logger.SetLogPath = ConfigurationManager.AppSettings["LogPath"];
            

            sClientCipher = new StringCipher(ConfigurationManager.AppSettings["ClientKey"]);
            sClientKey = Utility.HexStringToByteArray(ConfigurationManager.AppSettings["ClientKey"]);
            String authServiceUrl = ConfigurationManager.AppSettings["AuthenticationService"];
            Int32 authId = Convert.ToInt32(ConfigurationManager.AppSettings["AuthenticationId"]);
            String authKey = ConfigurationManager.AppSettings["AuthenticationKey"];
            Int32 sessionExpiration = Convert.ToInt32(ConfigurationManager.AppSettings["SessionExpiration"]);
                        
            sWebRootPath = HttpContext.Current.Request.PhysicalApplicationPath;

            sAuthentication = new Authentication(authServiceUrl,
                                                 authId,
                                                 authKey,
                                                 sessionExpiration);

            sUpdateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            sLocaleKeys = ConfigurationManager.AppSettings["LocaleKeys"].Split(',');
            sRoamingDB = ConfigurationManager.ConnectionStrings["RoamingDB"].ConnectionString;

            try
            {
               
                Logger.Log(ELogLevel.Info,
                           null,
                           "Starting Service...");

                Thread startupThread = new Thread(StartService);
                startupThread.Start();
            }
            catch (Exception ex)
            {
                Logger.Log(ELogLevel.Warning, null, "Startup Failure:{0}, Stack:{1}", ex.Message, ex.StackTrace);
            }
        }

        private static void StartService()
        {

            // Establish connection to the installs DB.
            AddOnReporting.Initialize();
            AddOnDownloadTracker.Initialize();

            // Update the addon cache and news feed.
            Logger.Log(ELogLevel.Info,
                         null,
                         "Updating News...");
            
            // Update the news feed:
            UpdateNewsFeed();
            
            Logger.Log(ELogLevel.Info,
                         null,
                         "News Updated");
            Logger.Log(ELogLevel.Info,
                         null,
                         "Updating Addons...");

            // Update in-memory add on cache:
            UpdateAddOnCache();

            Logger.Log(ELogLevel.Info,
                         null,
                         "Addons Updated");
            Logger.Log(ELogLevel.Info,
                          null,
                          "Service Started");
            sIsStarted = true;

            // Thread for updating the addon cache. It's priority is set to below normal so that it does not contend with servicing requests.
            sUpdateThread = new Thread(UpdateAddOnCacheThread);
            sUpdateThread.Priority = ThreadPriority.BelowNormal;
            sUpdateThread.Start();

            // Thread for updating the news cache. It's priority is set to below normal so that it does not contend with servicing requests.
            sNewsFeedThread = new Thread(UpdateNewsFeedThread);
            sUpdateThread.Priority = ThreadPriority.BelowNormal;
            sNewsFeedThread.Start();


        }
        
        private static void UpdateNewsFeedThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(sUpdateThreadInterval);

                try
                {
                    UpdateNewsFeed();
                }                
                catch (ThreadAbortException tae)
                {
                    aborted = true;
                    sUpdateThread.Join(100);
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "ProcessUpdateThread Aborted: {0}", tae.Message + "\n" + tae.StackTrace);
                }
                catch (Exception ex)
                {
                    Logger.Log(ELogLevel.Warning,
                               null,
                               "Update News Feed failed: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }
        private static void UpdateNewsFeed()
        {

            using (SqlConnection conn = new SqlConnection(sRoamingDB))
            {

                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }
                
                foreach (string localeKey in sLocaleKeys)
                {
                    SqlCommand command = new SqlCommand("curseService_GetNewsFeed", conn);
                    command.CommandType = CommandType.StoredProcedure;
                    command.Parameters.Add("@strLocaleKey", SqlDbType.VarChar, 5);
                    command.Parameters["@strLocaleKey"].Value = localeKey;

                    StringBuilder newsFeed = new StringBuilder(5000);
                    RssDocument.WriteHeader(newsFeed,
                                            "Breaking News from Curse", 
                                            "http://www.curse.com",
                                            "The Latest News and Articles from Curse.com",
                                            localeKey,
                                            DateTime.UtcNow);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {

                        while (reader.Read())
                        {
                            NewsHeadline headline = new NewsHeadline(reader);
                            headline.WriteItem(newsFeed);

                        }
                    }

                    RssDocument.WriteFooter(newsFeed);

                    string newsPath = sWebRootPath + "news-" + localeKey + ".xml";
                    using (StreamWriter writer = new StreamWriter(newsPath, false, Encoding.UTF8))
                    {
                        writer.Write(newsFeed.ToString());
                    }
                }

            }
        }

        private static void UpdateAddOnCacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(sUpdateThreadInterval);
                GC.Collect();
                try
                {
                    UpdateAddOnCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    sUpdateThread.Join(100);
                    Logger.Log(ELogLevel.Info,
                               null,
                               "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {
                 
                    Logger.Log(ELogLevel.Info,
                               null,
                               "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }                               
            }
        }

        private static int SortAddOnsByKeyCount(AddOn obj1, AddOn obj2)
        {
            if (obj2.FileKeyCount > obj1.FileKeyCount)
            {
                return -1;
            }
            return 0;
        }

        private static void UpdateExceptionCache()
        {
            List<AddOnFingerprintException> exceptions = new List<AddOnFingerprintException>();

            using (SqlConnection conn = new SqlConnection(sRoamingDB))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "select ProjectId, FilePath from curse_ProjectFingerprintException";

                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        exceptions.Add(new AddOnFingerprintException(reader));
                    }
                }

            }


            lock (sExceptionFeedBytes)
            {
                sExceptionFeedBytes = AddOnFingerprintException.GetSerializedList(exceptions);
            }
        }

        private static void UpdateFileKeyCache(Dictionary<int, AddOn> pAddOnCache)
        {
            
            Dictionary<string, byte[]> fileKeyCache = new Dictionary<string, byte[]>();            
            using (SqlConnection conn = new SqlConnection(sRoamingDB))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }
                
                SqlCommand command = new SqlCommand("curseService_GetFileKeyList", conn);
                command.CommandType = CommandType.StoredProcedure;
                
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    string lastFileKey = String.Empty;
                    string currentFileKey = String.Empty;
                    List<AddOn> associatedAddOns = new List<AddOn>();
                    StringWriter sw = null;
                    XmlTextWriter writer = null;
                    int currentProjectId = 0;
                    reader.Read();
                    do
                    {
                        currentFileKey = reader.GetString(0).ToLower();
                        currentProjectId = reader.GetInt32(1);

                        if (lastFileKey != String.Empty && currentFileKey != lastFileKey)
                        {

                            sw = new StringWriter();
                            writer = new XmlTextWriter(sw);
                            writer.WriteStartElement("projectHints");
                            //Create the XML output for this key:
                            associatedAddOns.Sort(new Comparison<AddOn>(SortAddOnsByKeyCount));
                            
                            foreach (AddOn addOn in associatedAddOns)
                            {                                
                                writer.WriteStartElement("projectHint");
                                writer.WriteAttributeString("name", addOn.Name);
                                writer.WriteAttributeString("curseID", addOn.Id.ToString());
                                writer.WriteAttributeString("url", addOn.Url);
                                writer.WriteAttributeString("feed", addOn.FeedUrl);
                                writer.WriteAttributeString("author", addOn.PrimaryAuthor);                                
                                writer.WriteAttributeString("popularity", addOn.Downloads);
                                //Summary
                                addOn.WriteSummary(writer);
                                //Keys
                                addOn.WriteKeys(writer);
                                writer.WriteEndElement();
                            }
                            writer.WriteEndElement();                            
                            fileKeyCache.Add(lastFileKey, System.Text.Encoding.UTF8.GetBytes(sw.ToString()));
                            associatedAddOns.Clear();

                        }
                        lastFileKey = currentFileKey;
                        if (pAddOnCache.ContainsKey(currentProjectId))
                        {
                            associatedAddOns.Add(pAddOnCache[currentProjectId]);
                        }
                        else
                        {
                            //Logger.Log(ELogLevel.Debug, null, "Missing project: {0}", currentProjectId.ToString());
                        }
                    }
                    while (reader.Read());
                }
            }
            lock (sFileKeyCache)
            {
                sFileKeyCache = fileKeyCache;
            }
        }

        private static void UpdateAddOnCache()
        {
                       
            using (SqlConnection conn = new SqlConnection(sRoamingDB))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                Dictionary<int, AddOn> addOnCache = new Dictionary<int, AddOn>(sAddOnCache);             
                
                
                DateTime changeDate = sLastQueryTime.AddMinutes(-5);
                SqlCommand command = new SqlCommand("curseService_GetAddOnList", conn);
                command.CommandType = CommandType.StoredProcedure;
                command.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                command.Parameters["@LastUpdated"].Value = changeDate;
                
                sLastQueryTime = DateTime.UtcNow;

                // Populate the cache with recently modified addons:
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    Dictionary<int, List<IndividualFileFingerprint>> individualFingerprintCache = new Dictionary<int, List<IndividualFileFingerprint>>();
                    Dictionary<int, List<AddOnFile>> fileArchiveCache = new Dictionary<int, List<AddOnFile>>();
                    Dictionary<int, List<AddOnFileDependency>> fileDependencyCache = new Dictionary<int,List<AddOnFileDependency>>();

                    // Get a cache of the individual fingerprints:
                    using (SqlCommand fingerprintCommand = new SqlCommand("curseService_GetAllIndividualFingerprints", conn))
                    {
                        fingerprintCommand.CommandType = CommandType.StoredProcedure;
                        fingerprintCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                        fingerprintCommand.Parameters["@LastUpdated"].Value = changeDate;
                        fingerprintCommand.CommandTimeout = 300;
                        using (SqlDataReader fingerprintReader = fingerprintCommand.ExecuteReader())
                        {
                            while (fingerprintReader.Read())
                            {
                                int id = fingerprintReader.GetInt32(0);
                                int fileId = fingerprintReader.GetInt32(1);
                                long fingerprint = fingerprintReader.GetInt64(2);
                                string folder = fingerprintReader.GetString(3);
                                if (!individualFingerprintCache.ContainsKey(id))
                                {
                                    individualFingerprintCache.Add(id, new List<IndividualFileFingerprint>());
                                }
                                individualFingerprintCache[id].Add(new IndividualFileFingerprint(id, fileId, fingerprint, folder));
                            }
                        }
                    }

                    // Get a cache of the file dependenecies:
                    using (SqlCommand dependencyCommand = new SqlCommand("curseService_GetAllFileDependencies", conn))
                    {
                        dependencyCommand.CommandType = CommandType.StoredProcedure;
                        dependencyCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                        dependencyCommand.Parameters["@LastUpdated"].Value = changeDate;
                        dependencyCommand.CommandTimeout = 300;
                        using (SqlDataReader dependencyReader = dependencyCommand.ExecuteReader())
                        {
                            while (dependencyReader.Read())
                            {                                
                                int fileId = dependencyReader.GetInt32(0);
                                int dependencyId = dependencyReader.GetInt32(1);
                                byte dependencyType = dependencyReader.GetByte(2);

                                if (!fileDependencyCache.ContainsKey(fileId))
                                {
                                    fileDependencyCache.Add(fileId, new List<AddOnFileDependency>());
                                }
                                fileDependencyCache[fileId].Add(new AddOnFileDependency(dependencyId, dependencyType));
                            }
                        }
                    }

                    // Get a cache of the file archive:
                    using (SqlCommand archiveCommand = new SqlCommand("curseService_GetAllArchivedFiles", conn))
                    {
                        archiveCommand.CommandType = CommandType.StoredProcedure;
                        archiveCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                        archiveCommand.Parameters["@LastUpdated"].Value = changeDate;

                        using (SqlDataReader archiveReader = archiveCommand.ExecuteReader())
                        {
                            while (archiveReader.Read())
                            {
                                int id = archiveReader.GetInt32(0);                                
                                if (!fileArchiveCache.ContainsKey(id))
                                {
                                    fileArchiveCache.Add(id, new List<AddOnFile>());
                                }
                                fileArchiveCache[id].Add(new AddOnFile(archiveReader, true));
                            }
                        }
                    }

                    string addOnName = null;
                    Int32 addOnId = 0;
                    while (reader.Read())
                    {
                        addOnId = reader.GetInt32(reader.GetOrdinal("addon_id"));
                        addOnName = reader.GetString(reader.GetOrdinal("addon_name"));

                        if (addOnCache.ContainsKey(addOnId))
                        {
                            addOnCache[addOnId] = new AddOn(conn, reader, individualFingerprintCache, fileArchiveCache, fileDependencyCache);
                        }
                        else
                        {
                            addOnCache.Add(addOnId, new AddOn(conn, reader, individualFingerprintCache, fileArchiveCache, fileDependencyCache));
                        }                                                                        
                    }
                }
                

                // Disable any deleted addons:
                command = new SqlCommand("curseService_GetDeletedAddOnList", conn);
                command.CommandType = CommandType.StoredProcedure;
                command.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                command.Parameters["@LastUpdated"].Value = changeDate;
            
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    Int32 addOnId = 0;
                    while (reader.Read())
                    {
                        addOnId = reader.GetInt32(0);                        
                        if (addOnCache.ContainsKey(addOnId))
                        {
                            addOnCache[addOnId].Available = false;                 
                        }                        
                    }
                }                

                DateTime currentTime = DateTime.UtcNow;

                // Create the update feed:
                StringBuilder updateXmlBuilder = new StringBuilder();
                updateXmlBuilder.Append("<psyn version=\"1.0\">");
                foreach (KeyValuePair<int, AddOn> kvp in addOnCache)
                {
                    TimeSpan elapsed = currentTime.Subtract(kvp.Value.LastUpdated);
                    if (elapsed.TotalHours <= 3)
                    {
                        updateXmlBuilder.Append(kvp.Value.ProjectElementXml);
                    }
                }
                updateXmlBuilder.Append("</psyn>");

                //Create the complete list, grouped by game ID:
                MemoryStream listStream = new MemoryStream();                
                byte[] serviceVersion = BitConverter.GetBytes((UInt32)0);
                listStream.Write(serviceVersion, 0, serviceVersion.Length);

                Dictionary<Int32, Int32> distinctGames = new Dictionary<int, int>();
                //Get a distinct list of games:
                foreach (KeyValuePair<int, AddOn> kvp in addOnCache)
                {
                    if (!distinctGames.ContainsKey(kvp.Value.GameId))
                    {
                        distinctGames.Add(kvp.Value.GameId, 0);
                    }
                    // Increment the count:
                    if (kvp.Value.Available)
                    {
                        distinctGames[kvp.Value.GameId] += 1;
                    }
                }

                // Write out the number of games included in the feed:
                byte[] gameCount = BitConverter.GetBytes((UInt16)distinctGames.Count);
                listStream.Write(gameCount, 0, gameCount.Length);

                // Write out the game ID, count, then the list of addons
                foreach (KeyValuePair<Int32, Int32> game in distinctGames)
                {
                    byte[] gameIdBytes = BitConverter.GetBytes((UInt32)game.Key);
                    listStream.Write(gameIdBytes, 0, gameIdBytes.Length);

                    byte[] gameCountBytes = BitConverter.GetBytes((UInt32)game.Value);
                    listStream.Write(gameCountBytes, 0, gameCountBytes.Length);

                    foreach (KeyValuePair<int, AddOn> kvp in addOnCache)
                    {
                        if (kvp.Value.GameId == game.Key && kvp.Value.Available) 
                        {
                            kvp.Value.WriteListElement(listStream);
                        }
                    }
                }
                
                

                //Update the file key cache, using the local version:
                // We don't use file keys any longer!
                //UpdateFileKeyCache(addOnCache);

                AddOnFingerprints.Rebuild(addOnCache);

                lock (sAddOnCache)
                {
                    sAddOnCache = addOnCache;
                }

                lock (sUpdateFeedBytes)
                {
                    sUpdateFeedBytes = System.Text.Encoding.UTF8.GetBytes(updateXmlBuilder.ToString());
                }

                byte[] compressedBytes = Utility.GetCompressedBytes(listStream.ToArray());
                lock (sCompleteFeedBytes)
                {                    
                    sCompleteFeedBytes = compressedBytes;
                }

                listStream.Close();

            }
            UpdateExceptionCache();
            GC.Collect();
            
        }

        private bool IsAdmin(string pAdminKey)
        {
            if (pAdminKey != ConfigurationManager.AppSettings["AdminKey"])
            {
                
                Logger.Log(ELogLevel.Error,
                           Context.Request.GetClientIPAddress().ToString(),
                           "SetClientVersion attempted with invalid admin key");
                Context.Response.Write("Invalid admin key, attempt has been logged");
                return false;
            }
            else
            {
                return true;
            }

        }

        private bool IsAutheticated(string pSession)
        {
            Int32 userId;
            StatusCode status = sAuthentication.ValidateSessionKeyToUserId(pSession, out userId);
            if (status != StatusCode.Ok)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.GetClientIPAddress().ToString(),
                           "ValidateSessionKeyToUserId failed with status {0}",
                           status);
                Context.Response.OutputStream.WriteByte((Byte)status);
                return false;
            }
            return true;
        }


        private void SaveInstalledAddons(Int32 pUserId, int[] pAddOnList)
        {
            if(pAddOnList.Length == 0)
            {
                return;
            }

            List<InstalledAddon> userAddons = new List<InstalledAddon>();
            
            foreach (int id in pAddOnList)
            {
                InstalledAddon install = new InstalledAddon();                
                install.UserID = pUserId;
                install.AddOnID = id;
                install.Posted = DateTime.UtcNow;
                userAddons.Add(install);
            }
            AddOnReporting.ReportUserAddons(pUserId, userAddons);            
        }

        

        [WebMethod]
        public void GetInstallCounts(string pApiKey)
        {
            if (pApiKey != ConfigurationManager.AppSettings["ApiKey"])
            {
                Context.Response.Write("Invalid Api Key");
                return;
            }

            if (Context.Cache["InstallCounts"] != null)
            {
                Context.Response.Write(Context.Cache["InstallCounts"]);
                return;
            }

            List<string> installCounts = new List<string>();
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["InstallsDB.Production"].ConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    return;
                }
                SqlCommand command = conn.CreateCommand();
                command.CommandText = "select addon_id, count(0) as install_count from installed_addon group by addon_id;";
                command.CommandTimeout = 300;                
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while(reader.Read())
                    {
                        installCounts.Add(reader.GetInt32(0) + "," + reader.GetInt32(1));
                    }
                }                
            }

            Context.Cache.Add("InstallCounts", string.Join(Environment.NewLine, installCounts.ToArray()), null, Cache.NoAbsoluteExpiration, TimeSpan.FromMinutes(5), CacheItemPriority.High, null);
            Context.Response.Write(Context.Cache["InstallCounts"]);
        }

        [WebMethod]
        public void GetFingerprintExceptions(string pSession)
        {
            if (!sIsStarted)
            {
                return;
            }

            if (!IsAutheticated(pSession))
            {
                return;
            }

            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
            Context.Response.OutputStream.Write(sExceptionFeedBytes, 0, sExceptionFeedBytes.Length);


        }

        [WebMethod]
        public void GetAddOn(string pSession, int pAddOnId)
        {
            if (!sIsStarted)
            {
                return;
            }

            if (!IsAutheticated(pSession))
            {
                return;
            }

            if (!sAddOnCache.ContainsKey(pAddOnId))
            {
                Context.Response.OutputStream.WriteByte((Byte)StatusCode.AddOnNotFound);
                return;
            }

            // Log this as a download
            AddOnDownloadTracker.LogFileDownload(pAddOnId, sAddOnCache[pAddOnId].DefaultFileId);


            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);

            // User data, for use in hash code generation:
            WriteUserData(Context, Context.Response.OutputStream);

            PsynDocument.WriteHeader(Context.Response.OutputStream);
            sAddOnCache[pAddOnId].WriteElement(Context.Response.OutputStream);
            PsynDocument.WriteFooter(Context.Response.OutputStream);

        }

        [WebMethod]
        public void ReportInstalledAddons(string pSession, int[] pAddOnId)
        {          
            if (!IsAutheticated(pSession))
            {
                return;
            }
            int userId = sAuthentication.GetSession(pSession).UserId;
            AddOnReporting.LogUser(userId, Context.Request.UserAgent); 
            SaveInstalledAddons(userId, pAddOnId);            
            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
        }
              
        [WebMethod]
        public void GetCustomList(string pSession, int[] pAddOnId)
        {
            if (!sIsStarted)
            {
                return;
            }
            if (!IsAutheticated(pSession))
            {
                return;
            }
           
            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
            
            // User data, for use in hash code generation:
            WriteUserData(Context, Context.Response.OutputStream);

            PsynDocument.WriteHeader(Context.Response.OutputStream);

            foreach (int id in pAddOnId)
            {
                if (!sAddOnCache.ContainsKey(id))
                {
                    continue;
                }
             
                sAddOnCache[id].WriteElement(Context.Response.OutputStream);
            }         
            
            PsynDocument.WriteFooter(Context.Response.OutputStream);                        
        }

        [WebMethod]
        public void GetUpdateList(string pSession)
        {
            if (!sIsStarted)
            {
                return;
            }

            if (!IsAutheticated(pSession))
            {
                return;
            }
            // Status Code:
            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
            // User data, for use in hash code generation:
            WriteUserData(Context, Context.Response.OutputStream);
            // Actual Addon Data:
            Context.Response.OutputStream.Write(sUpdateFeedBytes, 0, sUpdateFeedBytes.Length);
        }
               

#if DEBUG

        [WebMethod]
        public void TestGetFingerprintMatches(long[] pFingerprints)
        {

            if (!sIsStarted)
            {
                return;
            }
            AddOnFingerprintMatchResult result = AddOnFingerprints.GetMatches(pFingerprints);

            if (result.ExactMatches.Count == 0 && result.PartialMatches.Count == 0) // No matches
            {
                return;
            }


            StringWriter sw = new StringWriter();
            XmlTextWriter writer = new XmlTextWriter(sw);
            writer.Formatting = Formatting.Indented;
            writer.WriteStartDocument();
            writer.WriteStartElement("matches");
            // Write out each match:
            if (result.ExactMatches != null)
            {
                foreach (AddOnFingerprintMatch match in result.ExactMatches)
                {
                    writer.WriteStartElement("match");
                    writer.WriteAttributeString("type", "exact");
                    writer.WriteAttributeString("name", sAddOnCache[match.Id].Name);
                    writer.WriteStartElement("fingerprints");
                    foreach (long fingerprint in match.Fingerprints)
                    {
                        writer.WriteStartElement("fingerprint");
                        writer.WriteString(fingerprint.ToString());
                        writer.WriteEndElement();
                    }
                    writer.WriteEndElement();
                    writer.WriteEndElement();
                }
            }
            if (result.PartialMatches != null)
            {
                foreach (AddOnFingerprintMatch match in result.PartialMatches)
                {
                    writer.WriteStartElement("match");
                    writer.WriteAttributeString("type", "partial");
                    writer.WriteAttributeString("name", sAddOnCache[match.Id].Name);
                    writer.WriteStartElement("fingerprints");

                    foreach (long fingerprint in result.PartialMatchFingerprints[match.Id])
                    {
                        writer.WriteStartElement("fingerprint");
                        writer.WriteString(fingerprint.ToString());
                        writer.WriteEndElement();
                    }
                    writer.WriteEndElement();
                    writer.WriteEndElement();
                }
            }
            writer.WriteEndElement();

            writer.WriteEndDocument();

            Context.Response.Write(sw.ToString());
        }

        [WebMethod]
        public void TestGetFuzzyMatches(string pFoldername, long[] pFingerprints)
        {
            if (!sIsStarted)
            {
                return;
            }
            AddOnFuzzyMatchResult result = AddOnFingerprints.GetFuzzyMatches(pFoldername, pFingerprints);

            if (result.Match == null) // No matches
            {
                return;
            }

            StringWriter sw = new StringWriter();
            XmlTextWriter writer = new XmlTextWriter(sw);
            writer.Formatting = Formatting.Indented;
            writer.WriteStartDocument();
            writer.WriteStartElement("matches");
            
            writer.WriteStartElement("match");
            writer.WriteAttributeString("type", "fuzzy");
            writer.WriteAttributeString("name", sAddOnCache[result.Match.Id].Name);                
            writer.WriteStartElement("fingerprints");
            foreach (long fingerprint in result.Match.Fingerprints)
            {
                writer.WriteStartElement("fingerprint");
                writer.WriteString(fingerprint.ToString());
                writer.WriteEndElement();
            }
            writer.WriteEndElement();
            writer.WriteEndElement();
                        
            writer.WriteEndElement();

            writer.WriteEndDocument();

            Context.Response.Write(sw.ToString());
        }
#endif
        [WebMethod]
        public void GetFingerprintMatches(string pSession, long[] pFingerprints)
        {
            if (!sIsStarted)
            {
                return;
            }

            if (!IsAutheticated(pSession))
            {
                return;
            }

            CPUMonitor monitorTotal = new CPUMonitor("GetFingerprintMatches for {0} fingerprints.", pFingerprints.Length);

            AddOnFingerprintMatchResult result = AddOnFingerprints.GetMatches(pFingerprints);

            if (result.ExactMatches.Count == 0 && result.PartialMatches.Count == 0)
            {
                Context.Response.OutputStream.WriteByte((Byte)StatusCode.AddOnNoHints);
                return;
            }
            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);

            // Write out the number of exact matches:            
            byte[] exactMatchCount = BitConverter.GetBytes(result.ExactMatches.Count);
            Context.Response.OutputStream.Write(exactMatchCount, 0, exactMatchCount.Length);            
            foreach (AddOnFingerprintMatch match in result.ExactMatches)
            {
                match.WriteExactMatch(Context.Response.OutputStream);
            }

            // Write out the number of partial matches:            
            byte[] partialMatchCount = BitConverter.GetBytes(result.PartialMatches.Count);
            Context.Response.OutputStream.Write(partialMatchCount, 0, partialMatchCount.Length);
            foreach (AddOnFingerprintMatch match in result.PartialMatches)
            {
                match.WritePartialMatch(Context.Response.OutputStream);
            }

            Logger.Log(ELogLevel.Debug,
                      null,
                      monitorTotal.ToString());

            
        }

        [WebMethod]
        public void GetFuzzyMatches(string pSession, string pFoldername, long[] pFingerprints)
        {
            if (!sIsStarted)
            {
                return;
            }
            if (!IsAutheticated(pSession))
            {
                return;
            }
            AddOnFuzzyMatchResult result = AddOnFingerprints.GetFuzzyMatches(pFoldername, pFingerprints);

            if (result.Match == null) // No matches
            {
                Context.Response.OutputStream.WriteByte((Byte)StatusCode.AddOnNoHints);
                return;                
            }
            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
            
            result.Match.Write(Context.Response.OutputStream);            
        }
        
        [WebMethod]
        public void GetHints(string pSession, string pFileKey)
        {
            if (!sIsStarted)
            {
                return;
            }

            if (!IsAutheticated(pSession))
            {
                return;
            }

            if (!sFileKeyCache.ContainsKey(pFileKey))
            {
                Context.Response.OutputStream.WriteByte((Byte)StatusCode.AddOnNoHints);
                return;
            }
            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
            Context.Response.OutputStream.Write(sFileKeyCache[pFileKey], 0, sFileKeyCache[pFileKey].Length);
        }

        
        public static void WriteUserData(HttpContext pContext, Stream pStream)
        {
            string ipString = pContext.Request.GetClientIPAddress().ToString();
            byte[] ip = System.Text.Encoding.UTF8.GetBytes(ipString);
            byte[] time = BitConverter.GetBytes((UInt64)Utility.GetEpochDate(System.DateTime.UtcNow));
            pStream.WriteByte((byte)ipString.Length);
            pStream.Write(ip, 0, ip.Length);
            pStream.Write(time, 0, time.Length);
        }

        [WebMethod]
        public void GetChangeLog(string pSession, int pAddOnId, int pFileId)
        {
            if (!sIsStarted)
            {
                return;
            }

            if (!IsAutheticated(pSession))
            {
                return;
            }

            if (!sAddOnCache.ContainsKey(pAddOnId))
            {
                Context.Response.OutputStream.WriteByte((Byte)StatusCode.AddOnNotFound);
                return;
            }

            if(pFileId == 0)
            {
                pFileId = sAddOnCache[pAddOnId].DefaultFileId;
            }

            if (!sAddOnCache[pAddOnId].Files.ContainsKey(pFileId))
            {
                Context.Response.OutputStream.WriteByte((Byte)StatusCode.AddOnNotFound);
                return;
            }

            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
            sAddOnCache[pAddOnId].Files[pFileId].WriteChangeLogBytes(Context.Response.OutputStream);            
        }

        [WebMethod]
        public void GetCompleteList(string pSession)
        {
            if (!sIsStarted)
            {
                return;
            }

            if (!IsAutheticated(pSession))
            {
                return;
            }            
            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
            Context.Response.OutputStream.Write(sCompleteFeedBytes, 0, sCompleteFeedBytes.Length);
        }



        [WebMethod]
        public void GetStats(string pAdminKey)
        {
            if (!IsAdmin(pAdminKey))
            {
                return;
            }

            int sessionCount = sAuthentication.GetSessionCount();
            long memoryUsed = System.Diagnostics.Process.GetCurrentProcess().WorkingSet64 / (1024 * 1024);            
            Context.Response.Write(String.Format("Session Count: {0}\n\nMemory Usage: {1}",
                sessionCount.ToString(),
                memoryUsed.ToString()));
        }      

        [WebMethod]
        public void RecycleCache(string pAdminKey)
        {
            if (!IsAdmin(pAdminKey))
            {
                return;
            }            
            sLastQueryTime = DateTime.Parse("1/1/1977 12:00:00 AM");            
        }
        
        [WebMethod]
        public void SetClientVersion(string pAdminKey, string pVersion, string pUrl, string pFullSetupUrl)
        {         
            // Make sure this person is an admin:
            if (!IsAdmin(pAdminKey))
            {
                return;
            }
           
            if(!sValidVersion.Match(pVersion).Success)
            {
                Context.Response.Write("Invalid version.");
                return;
            }
            //StringWriter sw = new StringWriter();
            
            string XmlPath = Context.Server.MapPath("client.xml");
            using (XmlTextWriter writer = new XmlTextWriter(XmlPath, Encoding.UTF8))
            {
                writer.Formatting = Formatting.Indented;
                writer.WriteStartDocument();
                writer.WriteStartElement("clientInfo");
                writer.WriteStartElement("version");
                writer.WriteAttributeString("latest", pVersion);
                writer.WriteAttributeString("update", pUrl);
                writer.WriteAttributeString("setup", pFullSetupUrl);
                writer.WriteEndElement();
                writer.WriteEndElement();
            }

        }

        [WebMethod]
        public void SetClientMessage(string pAdminKey, string pTitle, string pBody)
        {            
            // Make sure this person is an admin:
            if (pAdminKey != ConfigurationManager.AppSettings["AdminKey"])
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.GetClientIPAddress().ToString(),
                           "SetClientMessage attempted with invalid admin key");
                Context.Response.Write("Invalid admin key, attempt has been logged");
                return;
            }
            
            string XmlPath = Context.Server.MapPath("message.xml");
            using (XmlTextWriter writer = new XmlTextWriter(XmlPath, Encoding.UTF8))
            {
                writer.Formatting = Formatting.Indented;
                writer.WriteStartDocument();
                
                writer.WriteStartElement("clientMessage");

                // Title
                writer.WriteStartElement("title");
                writer.WriteString(pTitle);
                writer.WriteEndElement();

                // Body
                writer.WriteStartElement("body");
                writer.WriteString(pBody);
                writer.WriteEndElement();

                writer.WriteEndElement();

                writer.WriteEndDocument();
            }

        }
    }
}
