﻿using System;
using System.Collections.Generic;
using System.Collections.Concurrent;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Web;
using Curse.ServiceModels.Caching;
using Curse.DownloadStatisticService.Models;
using System.Data.SqlClient;
using Curse.Extensions;
using Curse.ServiceModels.Configuration;
using Curse.ServiceModels.Models;
using System.Net;
using Curse.ServiceModels.Exceptions;
using System.ServiceModel.Web;
using System.Data;
using System.Threading;
using System.Threading.Tasks;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using System.Configuration;

namespace Curse.DownloadStatisticService.Caching
{
    public class DownloadLogEntryCache
    {
        #region Processing Objects
        List<AddDownloadLogEntryMessage> _logMessages;
        private ConcurrentQueue<StrongBox<AddDownloadLogEntryMessage>> _messageQueue;
        object _queueLock = new object();
        string _staticFileDestination;
        #endregion

        string _dbConnectionString;

        public static DownloadLogEntryCache _instance = new DownloadLogEntryCache();
        public static DownloadLogEntryCache Instance { get { return _instance; } }

        public DownloadLogEntryCache()
        {

            var config = ServiceConfiguration.Instance;

            SynchronizationContext.SetSynchronizationContext(new SynchronizationContext());
            TaskScheduler t = TaskScheduler.FromCurrentSynchronizationContext();
            Task.Factory.StartNew(() => ProcessingLoop(t));

            _logMessages = new List<AddDownloadLogEntryMessage>();
            _messageQueue = new ConcurrentQueue<StrongBox<AddDownloadLogEntryMessage>>();

            _staticFileDestination = ConfigurationManager.AppSettings["staticFileLocation"];
            if (!Directory.Exists(_staticFileDestination))
            {
                Directory.CreateDirectory(_staticFileDestination);
            }

            _dbConnectionString = DatabaseConfiguration.Instance["DownloadStatistics"].ConnectionString;

        }

        public void Initialize() { }

        public void ProcessingLoop(TaskScheduler t)
        {
            while (true)
            {

                List<Task> tasks = new List<Task>();

                for (int i = 0; i < 10; i++) //TODO: make this number configurable
                {
                    tasks.Add(Task.Factory.StartNew(var => ProcessLogs(), t));
                }

                Task[] tasksToProcess = tasks.ToArray();
                Task.WaitAll(tasksToProcess);

                var config = ServiceConfiguration.Instance;
                Thread.Sleep(TimeSpan.FromMilliseconds(config.Processing.ProcessingInterval));
            }
        }

        #region Process Queues

        public bool ProcessLogs()
        {
            if (!_messageQueue.Any()) return true;

            var logMessages = new List<AddDownloadLogEntryMessage>();

            for (int i = 0; i < 10000; i++)
            {
                StrongBox<AddDownloadLogEntryMessage> message;
                bool result = _messageQueue.TryDequeue(out message);
                if (result)
                {
                    logMessages.Add(message.Value);
                }
            }

            DataTable _downloadEntries = new DataTable();
            _downloadEntries.Columns.Add("ID", typeof(long)).AutoIncrement = true;
            _downloadEntries.Columns.Add("SourceID", typeof(long));
            _downloadEntries.Columns.Add("FileID", typeof(int));
            _downloadEntries.Columns.Add("ProjectID", typeof(int));
            _downloadEntries.Columns.Add("GameID", typeof(int));
            _downloadEntries.Columns.Add("ReferrerID", typeof(int));
            _downloadEntries.Columns.Add("UserAgentID", typeof(int));
            _downloadEntries.Columns.Add("[Date]", typeof(DateTime));
            _downloadEntries.Columns.Add("IpAddress", typeof(string));

            try
            {
                foreach (var message in logMessages)
                {
                    DateTime date;
                    DateTime.TryParse(message.Date, out date);

                    DataRow newRow = _downloadEntries.NewRow();

                    if (message.ID != null)
                    {
                        newRow["SourceID"] = message.ID;
                    }

                    newRow["FileID"] = message.FileID;
                    newRow["ProjectID"] = message.ProjectID;
                    newRow["GameID"] = message.GameID;
                    if (String.IsNullOrEmpty(message.Referrer))
                    {
                        newRow["ReferrerID"] = DBNull.Value;
                    }
                    else
                    {
                        newRow["ReferrerID"] = ReferrerCache.Instance.GetIDByReferrer(message.Referrer);
                    }

                    if (String.IsNullOrEmpty(message.UserAgent))
                    {
                        newRow["UserAgentID"] = DBNull.Value;
                    }
                    else
                    {
                        newRow["UserAgentID"] = UserAgentCache.Instance.GetIDByUserAgent(message.UserAgent);
                    }
                    newRow["[Date]"] = date;
                    newRow["IpAddress"] = message.IpAddress;

                    _downloadEntries.Rows.Add(newRow);
                }

                using (var conn = new SqlConnection(_dbConnectionString))
                {
                    conn.Open();
                    using (SqlTransaction trans = conn.BeginTransaction())
                    {
                        var bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.Default, trans);
                        bulk.DestinationTableName = "DownloadLogEntry";
                        bulk.BulkCopyTimeout = 360;
                        bulk.BatchSize = 10000;

                        try
                        {
                            bulk.WriteToServer(_downloadEntries);
                            trans.Commit();
                        }
                        catch
                        {
                            try
                            {
                                trans.Rollback();
                            }
                            catch (Exception trnEx)
                            {
                                Logger.Log("Failed to rollback transaction. Details: {0}", ELogLevel.Error, trnEx.GetExceptionDetails());
                            }

                            throw;
                        }
                    }
                    _downloadEntries.Clear();
                }
            }
            catch (Exception exc)
            {
                Logger.Log("Failed to add download entry log. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
                LogMessageError(logMessages, exc);
            }

            return true;
        }

        public bool AddDownloadLogEntry(AddDownloadLogEntryMessage message)
        {
            ErrorData error = null;
            try
            {

                IPAddress address;
                if (!IPAddress.TryParse(message.IpAddress, out address))
                {
                    error = new ErrorData("Could not parse the IPAddress supplied", 1001);
                    throw new WebFaultException<ErrorData>(error, HttpStatusCode.BadRequest);
                }

                DateTime date;
                if (!DateTime.TryParse(message.Date, out date))
                {
                    error = new ErrorData("Could not parse the date supplied", 1002);
                    throw new WebFaultException<ErrorData>(error, HttpStatusCode.BadRequest);
                }

                var fileCheckMessage = string.Empty;

                if (!FileCache.Instance.CheckFileData(message, out fileCheckMessage))
                {
                    error = new ErrorData(fileCheckMessage, 1002);
                    throw new WebFaultException<ErrorData>(error, HttpStatusCode.BadRequest);
                }


                var projectCheckMessage = string.Empty;
                if (!ProjectCache.Instance.CheckProjectData(message, out projectCheckMessage))
                {
                    error = new ErrorData(projectCheckMessage, 1002);
                    throw new WebFaultException<ErrorData>(error, HttpStatusCode.BadRequest);
                }

                if (!String.IsNullOrEmpty(message.Referrer))
                {
                    var referrerCheckMessage = string.Empty;
                    if (!ReferrerCache.Instance.CheckReferrerData(message, out referrerCheckMessage))
                    {
                        error = new ErrorData(referrerCheckMessage, 1002);
                        throw new WebFaultException<ErrorData>(error, HttpStatusCode.BadRequest);
                    }
                }

                if (!String.IsNullOrEmpty(message.UserAgent))
                {
                    var userAgentCheckMessage = string.Empty;
                    if (!UserAgentCache.Instance.CheckUserAgentData(message, out userAgentCheckMessage))
                    {
                        error = new ErrorData(userAgentCheckMessage, 1002);
                        throw new WebFaultException<ErrorData>(error, HttpStatusCode.BadRequest);
                    }
                }

                StrongBox<AddDownloadLogEntryMessage> box = new StrongBox<AddDownloadLogEntryMessage> { Value = message };
                _messageQueue.Enqueue(box);
            }
            catch (Exception exc)
            {
                Logger.Log("Failed to add download log entry record from. Details: {0}. Exc message is: {1}", ELogLevel.Error, error.ErrorMessage, exc.Message);
                LogMessageError(message, error.ErrorMessage);
            }

            return true;
        }

        public void LogMessageError(List<AddDownloadLogEntryMessage> messages, Exception exception)
        {
            var errorTable = new DataTable();
            errorTable.Columns.Add("ID", typeof(long)).AutoIncrement = true;
            errorTable.Columns.Add("SourceID", typeof(long));
            errorTable.Columns.Add("FileID", typeof(int));
            errorTable.Columns.Add("ProjectID", typeof(int));
            errorTable.Columns.Add("GameID", typeof(int));
            errorTable.Columns.Add("Referrer", typeof(string));
            errorTable.Columns.Add("UserAgent", typeof(string));
            errorTable.Columns.Add("[Date]", typeof(DateTime));
            errorTable.Columns.Add("IpAddress", typeof(string));
            errorTable.Columns.Add("ErrorMessage", typeof(string));

            foreach (var message in messages)
            {
                var newRow = errorTable.NewRow();
                newRow["SourceID"] = message.ID;
                newRow["FileID"] = message.FileID;
                newRow["ProjectID"] = message.ProjectID;
                newRow["GameID"] = message.GameID;
                newRow["Referrer"] = message.Referrer;
                newRow["UserAgent"] = message.UserAgent;
                newRow["[Date]"] = message.Date;
                newRow["IpAddress"] = message.IpAddress;
                newRow["ErrorMessage"] = exception.Message;

                errorTable.Rows.Add(newRow);
            }

            using (var conn = new SqlConnection(_dbConnectionString))
            {
                conn.Open();

                using (SqlTransaction trans = conn.BeginTransaction())
                {

                    SqlBulkCopy bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.Default, trans);
                    bulk.DestinationTableName = "DownloadLogError";
                    bulk.BulkCopyTimeout = 360;
                    bulk.BatchSize = 10000;

                    try
                    {
                        bulk.WriteToServer(errorTable);
                        trans.Commit();
                    }
                    catch (Exception exc)
                    {
                        Logger.Log("Failed to Save Erroneous Message. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
                        try
                        {
                            trans.Rollback();
                        }
                        catch (Exception trnEx)
                        {
                            Logger.Log("Failed to rollback transaction. Details: {0}", ELogLevel.Error, trnEx.GetExceptionDetails());
                        }

                        SaveMessagesToDisk(messages);
                    }
                }
            }
        }

        public void LogMessageError(AddDownloadLogEntryMessage message, string exception)
        {
            using (var conn = DatabaseConfiguration.Instance["DownloadStatistics"].GetDatabaseConnection())
            {
                var cmd = new SqlCommand("spAddDownloadLogError", conn);
                cmd.CommandType = System.Data.CommandType.StoredProcedure;
                cmd.Parameters.Add("SourceID", System.Data.SqlDbType.BigInt).Value = message.ID;
                cmd.Parameters.Add("FileID", System.Data.SqlDbType.Int).Value = message.FileID;
                cmd.Parameters.Add("ProjectID", System.Data.SqlDbType.Int).Value = message.ProjectID;
                cmd.Parameters.Add("GameID", System.Data.SqlDbType.Int).Value = message.GameID;
                cmd.Parameters.Add("Referrer", System.Data.SqlDbType.NVarChar).Value = message.Referrer;
                cmd.Parameters.Add("UserAgent", System.Data.SqlDbType.NVarChar).Value = message.UserAgent;
                cmd.Parameters.Add("Date", System.Data.SqlDbType.VarChar).Value = message.Date;
                cmd.Parameters.Add("ipAddress", System.Data.SqlDbType.VarChar).Value = message.IpAddress;
                cmd.Parameters.Add("ErrorMessage", System.Data.SqlDbType.NVarChar).Value = exception;

                try
                {
                    cmd.ExecuteNonQuery();
                }
                catch (Exception exc)
                {
                    Logger.Log("Failed to Save Erroneous Message. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
                    SaveMessageToDisk(message);
                }
            }
        }

        #endregion

        public void FlushQueueToDisk()
        {
            List<AddDownloadLogEntryMessage> messages = new List<AddDownloadLogEntryMessage>();

            for (int i = 0; i <= _messageQueue.Count; i++)
            {
                StrongBox<AddDownloadLogEntryMessage> message;
                bool result = _messageQueue.TryDequeue(out message);
                if (result)
                {
                    messages.Add(message.Value);
                }
            }

            SaveMessagesToDisk(messages);
        }


        #region Re-process ErrorLog Table and Files
        public void ProcessErrorLogs()
        {
            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["DownloadStatistics"].ConnectionString))
            {
                conn.Open();

                var processedIds = new List<int>();

                var cmd = new SqlCommand("SELECT * FROM DownloadLogError", conn);
                using (var reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {

                        var message = new AddDownloadLogEntryMessage
                        {
                            ID = reader.GetInt64(reader.GetOrdinal("SourceID")),
                            FileID = reader.GetInt32(reader.GetOrdinal("FileID")),
                            ProjectID = reader.GetInt32(reader.GetOrdinal("ProjectID")),
                            GameID = reader.GetInt32(reader.GetOrdinal("GameID")),
                            Date = reader.GetString(reader.GetOrdinal("Date")),
                            IpAddress = reader.GetString(reader.GetOrdinal("IpAddress"))
                        };

                        var referrer = reader.GetNullableValue<string>("Referrer");
                        if (!string.IsNullOrEmpty(referrer))
                        {
                            message.Referrer = referrer;
                        }

                        var userAgent = reader.GetNullableValue<string>("UserAgent");
                        if (!string.IsNullOrEmpty(userAgent))
                        {
                            message.UserAgent = userAgent;
                        }

                        if (AddDownloadLogEntry(message))
                        {
                            processedIds.Add(reader.GetInt32(reader.GetOrdinal("ID")));
                        }
                    }
                }

                if (processedIds.Count > 0)
                {
                    cmd.CommandText = string.Format("DELETE FROM DownloadLogError WHERE ID IN ({0})", string.Join(", ", processedIds.ToArray()));
                    try
                    {
                        cmd.ExecuteNonQuery();
                    }
                    catch (Exception exc)
                    {
                        Logger.Log("Failed to delete processed records from the error table. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
                        return;
                    }
                }
            }
        }

        //Double redundancy
        public void ProcessErrorFiles()
        {
            string[] fileNames = Directory.GetFiles(_staticFileDestination);
            foreach (string fileName in fileNames)
            {
                AddDownloadLogEntryMessage message = LoadMessageFromDisk(fileName);
                if (message != null)
                {
                    try
                    {
                        AddDownloadLogEntry(message);
                        System.IO.File.Delete(fileName);
                    }
                    catch (Exception exc)
                    {
                        Logger.Log("Unable to process message file ({0}). Details: {1}", ELogLevel.Error, fileName, exc.GetExceptionDetails());
                    }
                }
            }
        }

        private void SaveMessagesToDisk(List<AddDownloadLogEntryMessage> messages)
        {
            foreach (var message in messages)
            {
                SaveMessageToDisk(message);
            }
        }

        private void SaveMessageToDisk(AddDownloadLogEntryMessage message)
        {
            using (MemoryStream ms = new MemoryStream())
            {
                BinaryFormatter bf = new BinaryFormatter();
                bf.Serialize(ms, message);
                ms.Position = 0;

                byte[] uncompressed = ms.ToArray();
                byte[] compressed = Utility.GetCompressedBytes(uncompressed);

                string tempfile = Path.Combine(_staticFileDestination, Utility.UniqueNumber + ".txt");
                using (BinaryWriter binWriter = new BinaryWriter(System.IO.File.Open(tempfile, FileMode.Create)))
                {
                    binWriter.Write(compressed);
                }
            }
        }

        private AddDownloadLogEntryMessage LoadMessageFromDisk(string fileName)
        {
            AddDownloadLogEntryMessage message;

            // Read the compressed bytes out of the file
            byte[] compressed, uncompressed;
            try
            {
                using (var binReader = new BinaryReader(System.IO.File.Open(fileName, FileMode.Open)))
                {
                    compressed = binReader.ReadBytes((int)binReader.BaseStream.Length);
                }
            }
            catch (Exception ex)
            {
                Logger.Log("Unable to load message from feed file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                return null;
            }

            // Deserialize the stream
            try
            {
                uncompressed = Utility.GetDecompressedBytes(compressed);
                using (var ms = new MemoryStream())
                {
                    ms.Write(uncompressed, 0, uncompressed.Length);
                    ms.Position = 0;

                    var bf = new BinaryFormatter();
                    message = (AddDownloadLogEntryMessage)bf.Deserialize(ms);
                }
            }
            catch (Exception ex)
            {
                Logger.Log("Unable to deserialize the message! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                return null;
            }

            return message;
        }
        #endregion

        public int GetQueueCount()
        {
            return _messageQueue.Count;
        }


    }
}