function Parse(code, build)
	debug.sethook();
	debug.sethook(function() error("Infinite Loop Detected"); end, "", 10000000);
    local env = {}
    local loadData = loadstring(code)

    setfenv(loadData, env)
    local success, error = pcall(loadData)

    if not success then
	error('Failed to load data: ' .. error);
	return;
    end
    
    local data = env.WoWDBProfilerData;
    
    if not data or type(data.build_num) ~= 'number' then
	error('Invalid lua file -- not an addon SavedVariables file')
    end
    
--    if data.build_num ~= build then
--	error('Invalid build #' .. data.build_num .. ' -- Accepting only build #' .. build);
--    end

	if not data.version or data.version < 2 then
		error('Invalid version -- Only acception version 2+.');
	end


    LoadNpcData(data.global.npcs);
	LoadItems(data.global.items);
	LoadQuests(data.global.quests);
	LoadZones(data.global.zones);

	for objectKey, objectData in pairs(data.global.objects or {}) do

		if not tonumber(objectKey) then
			local actionName, objectName = split(objectKey, ':');
			LoadObject(objectName, objectData, actionName);
		end
	end
end

function LoadZones(data)
	for key, fishingData in pairs(data or {}) do
		local fishing, _ = split(key, ':');

		if fishing == 'FISHING' then
			for difficulty, dataByDifficulty in pairs(fishingData) do
				for wmaData, locationList in pairs(dataByDifficulty.locations or {}) do
					local wmaId = splitGetLast(wmaData, ':');
					local zoneId = DBC_GetZoneIdFromWma(tonumber(wmaId));

					for coords, dropInfo in pairs(locationList) do
						local floor, x, y = split(coords, ':');

						if type(dropInfo) == 'table' then
							AddDropData(difficulty, dropInfo.drops, DROP_FISHING, zoneId, TYPE_ZONE);

							RegisterDropCount(difficulty, DROP_FISHING, zoneId, TYPE_ZONE, tonumber(dropInfo.drops_count));
						end
					end
				end
			end
		end
	end
end

function LoadObject(name, data, action)
	local objectId = DBC_GetObjectId(name);

	if objectId == 0 then
		return;
	end

	for difficulty, dataByDifficulty in pairs(data) do
		for wmaData, locationList in pairs(dataByDifficulty.locations or {}) do
			local wmaId = splitGetLast(wmaData, ':');

			for coords, _ in pairs(locationList) do
				local floor, x, y = split(coords, ':');

				RegisterLocation(TYPE_OBJECT, objectId, difficulty, tonumber(wmaId), tonumber(floor), tonumber(x), tonumber(y));
			end
		end

		AddDropData(difficulty, dataByDifficulty.drops, DropTypes[action], objectId, TYPE_OBJECT);
		RegisterDropCount(difficulty, DropTypes[action], objectId, TYPE_OBJECT, tonumber(dataByDifficulty.drops_count));

	end
end

function LoadQuests(data)
	if type(data) ~= 'table' then
		return
	end

	for questId, questData in pairs(data) do
		questId = tonumber(questId) or 0;
		local timer = tonumber(questData.timer) or 0;

		LoadQuestBeginEndData(questId, timer, questData.begin, true);
		LoadQuestBeginEndData(questId, timer, questData["end"], false);

		for class, _ in pairs(questData.classes or {}) do
			RegisterQuestClass(questId, DBC_GetClassId(class));
		end

		for race, _ in pairs(questData.races or {}) do
			RegisterQuestRace(questId, DBC_GetRaceId(race));
		end
	end
end

function LoadQuestBeginEndData(questId, timer, data, isStart)
	if type(data) ~= 'table' then
		return;
	end

	for beginData, _ in pairs(data) do
		local type, typeId = split(beginData, ':');

		if type ~= 'PLAYER' and type ~= 'VEHICLE' then
			if type == 'NPC' then
				type = TYPE_NPC;
			elseif type == 'OBJECT' then
				type = TYPE_OBJECT;
			else
				error('Unknown Type: ' .. type);
			end

			typeId = tonumber(typeId);

			RegisterQuest(questId, typeId, type, isStart, timer);
		end
	end
end

function LoadItems(data)
	if type(data) ~= 'table' then
		return
	end

	for itemId, itemData in pairs(data) do
		local itemId = tonumber(itemId) or 0;

		if itemId <= 0 then
			error('Invalid item id: ' .. itemId);
		end

		local durability = tonumber(itemData.durability) or 0;

		if durability ~= 0  then
			RegisterItemDurability(itemId, durability);
		end

		if itemData.black_market then
			RegisterItemBlackMarket(itemId, itemData.black_market);
		end



		for _i, e in pairs(DropTypes) do
			local i = string.lower(_i);
			if(itemData[i] ~= nil) then
				AddDropData('', itemData[i], e, itemId, TYPE_ITEM);
				if(itemData[i .. "_count"] ~= nil) then
					RegisterDropCount('', e, itemId, TYPE_ITEM, tonumber(itemData[i .. "_count"]));
				end
			end
		end

	end
end




function AddDropData(difficulty, dataTable, dropSource, sourceId, sourceType)
	for _, dropData in pairs(dataTable or {}) do
		local itemId, amount, currencyIcon = split(dropData, ':');

		amount = tonumber(amount);

		if not amount then
			error('Invalid Amount: ' .. amount .. ' - ' .. dropData);
		end

		if itemId == 'money' then
			RegisterDrop(difficulty, dropSource, 0, TYPE_MONEY, sourceId, sourceType, tonumber(amount));
		elseif itemId == 'currency' then
			if not currencyIcon then
				error('Invalid Currency Drop String: ' .. dropData .. ' - ID: ' .. sourceId .. ' - type: ' .. sourceType);
			end

			local currencyId = DBC_GetCurrencyId(currencyIcon);
			if currencyId ~= 0 then
				RegisterDrop(difficulty, dropSource, tonumber(currencyId), TYPE_CURRENCY, sourceId, sourceType, tonumber(amount));
			end

		elseif not tonumber(itemId) then
			error('Invalid Drop Item ID: ' .. itemId .. ' - ' .. dropData .. ' - ID: ' .. sourceId .. ' - type: ' .. sourceType);
		else
			RegisterDrop(difficulty, dropSource, tonumber(itemId), TYPE_ITEM, sourceId, sourceType, tonumber(amount));
		end
	end
end






function LoadNpc(npcId, npcData)
	local repairs = npcData.can_repair or false;
	local factionId = npcData.faction and DBC_GetFactionId(npcData.faction) or 0;
	local pvpFlagged = npcData.is_pvp or false;
		
	for className, data2 in pairs(npcData.teaches or {}) do
		for professionName, data3 in pairs(data2) do
			for spellId, requirements in pairs(data3) do
				local requiredLevel, requiredSkill = split(requirements, ':');

				spellId = tonumber(spellId);					
				requiredSkill = tonumber(requiredSkill);

				if spellId > 0 and requiredSkill > 0 then
					RegisterTradeskillRequirement(spellId, requiredSkill);
				end
			end
		end
	end

	for factionData, reputationAmount in pairs(npcData.reputations or {}) do
		local faction, standing = split(factionData, ':');

		faction = DBC_GetFactionId(faction);
		standing = DBC_GetFactionStanding(standing);
		reputationAmount = tonumber(reputationAmount);

		if reputationAmount > 0 and faction > 0 then
			RegisterNpcReputation(npcId, faction, standing, reputationAmount);
		end
	end

	for difficulty, dataByDifficulty in pairs(npcData.encounter_data or {}) do
		
		for levelString, statData in pairs(dataByDifficulty.stats or {}) do
			local level = tonumber(string.match(levelString, 'level_(%d+)')) or 0
			local maxHealth = statData.max_health or 0
			local powerName, powerAmount = split(statData.power, ':');
				
			RegisterNpcData(npcId, difficulty, repairs, factionId, maxHealth, DBC_GetPowerId(powerName or "NONE"), tonumber(powerAmount) or 0, level, pvpFlagged);

			for wmaData, locationList in pairs(statData.locations or {}) do
				local wmaId = splitGetLast(wmaData, ':');

				for coords, _ in pairs(locationList) do
					local floor, x, y = split(coords, ':');

					if not tonumber(wmaId) or not tonumber(floor) or not tonumber(x) or not tonumber(y) then
						error('Invalid Location Data - Npc: ' .. npcId .. ' WNA: ' .. wmaData .. ' - ' .. wmaId .. ' Coords: ' .. coords);
					end

					RegisterLocation(TYPE_NPC, npcId, difficulty, tonumber(wmaId), tonumber(floor), tonumber(x), tonumber(y));
				end
			end
		end

		for spellId, timesCasted in pairs(dataByDifficulty.spells or {}) do
			RegisterNpcSpell(difficulty, npcId, spellId);
		end


		for _i, e in pairs(DropTypes) do
			local i = string.lower(_i);
			if(dataByDifficulty[i] ~= nil) then
				AddDropData(difficulty, dataByDifficulty[i], e, npcId, TYPE_NPC);
			end
		end


		for dropType, dropCount in pairs(dataByDifficulty.loot_counts or {}) do
			RegisterDropCount(difficulty, GetDropType(dropType), npcId, TYPE_NPC, tonumber(dropCount));
		end
	end


		-- note: the following block seems like unused code(?) :

	for dropType, dropCount in pairs(npcData.loot_counts or {}) do
		if dropType == 'drops' then
			RegisterDropCount('', DROP_KILLING, npcId, TYPE_NPC, tonumber(dropCount));
		end
	end
		
	LoadVendorItems(npcId, npcData.sells);

	if type(npcData.reaction) == 'string' then
		local _, faction, reaction = split(npcData.reaction, ':');

		if faction == 'Horde' then
			faction = 1;
		elseif faction == 'Alliance' then
			faction = 2;
		elseif faction == 'Neutral' then
			faction = 0;
		else
			error('Unknown Faction: ' .. faction .. ' NPC: ' .. npcId);
		end

		if reaction == 'FRIENDLY' then
			reaction = 1;
		elseif reaction == 'HOSTILE' then
			reaction = 2;
		elseif reaction == 'NEUTRAL' then
			reaction = 3;
		else
			error('Unknown Reaction: ' .. reaction);
		end

		if faction ~= 0 then
			RegisterReaction(npcId, faction, reaction);
		end
	end
end

function LoadNpcData(data)
	if type(data) ~= 'table' then
		return
	end

	for npcId, npcData in pairs(data) do
		if tonumber(npcId) then
			LoadNpc(tonumber(npcId), npcData);
		end
	end
end

function LoadVendorItems(npcId, data)
	if type(data) ~= 'table' then
		return
	end
	
	for itemId, keyString in pairs(data) do
		itemId = tonumber(itemId) or -1
		
		if itemId == -1 then
			return
		end

		local availability, stack, costData = keyString:match("(-?%d+):(%d+):%[([^]]*)%]");
		
		RegisterVendorItem(tonumber(npcId), tonumber(itemId), tonumber(stack), tonumber(availability), costData);
	end
end

