﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data.SqlClient;
using System.Threading;
using Curse.Logging;

namespace Curse.ClientService
{
    public abstract class BaseLookupCache
    {
        private readonly int _reloadLookupInterval;
        protected string DatabaseConnectionString;
        protected Dictionary<CustomKey, Int32> LookupCache = new Dictionary<CustomKey, Int32>(new CustomKey.CustomKeyComparer());
        protected bool LookupsLoaded = false;
        private readonly string[] _lookupTables;
        private readonly Thread _reloadLookupThread = null;
        private readonly bool _useTimestamps;

        protected BaseLookupCache(string connectionString, string[] lookupTables, int lookupReloadIntervalSeconds = 5, bool useTimestamps = false)
        {
            _reloadLookupInterval = lookupReloadIntervalSeconds * 1000;
            _useTimestamps = useTimestamps;
            DatabaseConnectionString = connectionString;
            _lookupTables = lookupTables;
            LookupsLoaded = ReloadLookupCache();

            if (bool.Parse(ConfigurationManager.AppSettings["LookupCacheEnabled"]))
            {
                Logger.Info("Lookup cache is enabled.");
                _reloadLookupThread = new Thread(ReloadLookupCacheThread) { IsBackground = true };
                _reloadLookupThread.Start();
            }
            else
            {
                Logger.Warn("Lookup cache is not enabled.");
            }
        }


        private void ReloadLookupCacheThread()
        {

            while (true)
            {
                try
                {
                    Thread.Sleep(_reloadLookupInterval);
                    ReloadLookupCache();
                }
                catch (ThreadAbortException)
                {

                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to load lookup cache!");
                }
            }
        }

        private DateTime _lastLoadedLookups = new DateTime(2000, 1, 1);

        private bool ReloadLookupCache()
        {
            try
            {
                var now = DateTime.UtcNow;
                using (var conn = new SqlConnection(DatabaseConnectionString))
                {
                    conn.Open();

                    foreach (var tableName in _lookupTables)
                    {
                        using (var command = conn.CreateCommand())
                        {
                            command.CommandText = "select * from " + tableName;

                            if (_useTimestamps)
                            {
                                command.CommandText += " where DateCreated >= @DateCreated";
                                command.Parameters.AddWithValue("@DateCreated", _lastLoadedLookups.AddMinutes(-1));
                            }

                            using (var reader = command.ExecuteReader())
                            {
                                while (reader.Read())
                                {
                                    LookupCache[GetLookupKey(tableName, (string)reader["LookupLabel"])] = (int)reader["LookupID"];
                                }
                            }
                        }
                    }
                }

                _lastLoadedLookups = now;
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to load lookup cache");
                return false;
            }
        }

        protected CustomKey GetLookupKey(string tableName, string nameValue)
        {
            return new CustomKey(tableName.ToLowerInvariant(), nameValue.ToLowerInvariant());
        }

        protected int GetLookupID(SqlConnection conn, string tableName, string labelValue)
        {
            if (string.IsNullOrEmpty(labelValue))
            {
                labelValue = "Unknown";
            }

            labelValue = labelValue.Trim();
            var lookupKey = GetLookupKey(tableName, labelValue);
            if (LookupCache.ContainsKey(lookupKey))
            {
                return LookupCache[lookupKey];
            }

            using (var command = conn.CreateCommand())
            {
                command.CommandText = "insert into " + tableName + "(LookupLabel) output inserted.LookupID values(@LookupLabel)";
                var param = command.Parameters.Add("@LookupLabel", System.Data.SqlDbType.NVarChar, 256);
                param.Value = labelValue;
                var id = (int)command.ExecuteScalar();
                LookupCache[lookupKey] = id;
                return id;
            }
        }
    }
}
