﻿using System;
using System.Collections.Generic;
using Curse.ClientService.Models;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using System.Threading;
using Curse.Logging;
using Newtonsoft.Json;


namespace Curse.ClientService
{
    public class CHardwareReportCache : BaseLookupCache
    {
        private static readonly CHardwareReportCache _instance = new CHardwareReportCache(); 

        private const int WRITE_INTERVAL = 5000;        
        private readonly Thread _writeToDatabaseThread = null;        
        private List<HardwareProfile> _hardwareProfiles;                        

        public static CHardwareReportCache Instance
        {
            get
            {
                return _instance;
            }
        }

        private CHardwareReportCache()
            : base(ConfigurationManager.ConnectionStrings["ClientHardwareReporting"].ConnectionString, new[]
        {
            "CpuManufacturer",
            "CpuModel",
            "DirectXVersion",
            "GpuDriverName",
            "GpuDriverVersion",
            "GpuManufacturer",
            "GpuModel",
            "OperatingSystemVersion",
            "Language",
            "MonitorModel",
            "RamManufacturer",
            "RamType",
            "ServicePackVersion",
            "SoundModel"
        }, 30, true)
        {
            
            
            _hardwareProfiles = new List<HardwareProfile>();

            _writeToDatabaseThread = new Thread(WriteToDatabaseThread) { IsBackground = true };
            _writeToDatabaseThread.Start();
        }              

        public void Initialize() {}
        
        public bool SaveReport(int pUserID, HardwareProfile pHardwareProfile)
        {
            if (!LookupsLoaded)
            {
                return false;
            }

            pHardwareProfile.UserID = pUserID;
            lock (_hardwareProfiles)
            {
                _hardwareProfiles.Add(pHardwareProfile);
            }
            return true;
        }
               
        private void WriteToDatabase()
        {
            List<HardwareProfile> profiles = null;

            lock (_hardwareProfiles)
            {
                profiles = new List<HardwareProfile>(_hardwareProfiles);
                _hardwareProfiles.Clear();
            }
            
            if (profiles.Count == 0)
            {
                return;
            }

            using (var conn = new SqlConnection(DatabaseConnectionString))
            {
                conn.Open();
                foreach (var profile in profiles)
                {
                    try
                    {
                        SaveProfileToDatabase(conn, profile);
                    }
                    catch (Exception ex)
                    {
                        var objectDump = JsonConvert.ToString(profile);                        
                        Logger.Error(ex, "CHardwareProfile - Failed to save hardware profile.", new { ObjectDump = objectDump });
                    }
                }
            }
        }

        private void WriteToDatabaseThread()
        {            
            while (true)
            {
                Thread.Sleep(WRITE_INTERVAL);

                try
                {
                    WriteToDatabase();
                }
                catch (ThreadAbortException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "CHardwareProfile SaveThread Thread Exception!");
                }
            }
        }

        private void SaveProfileToDatabase(SqlConnection conn, HardwareProfile hardwareProfile)
        {
            using (var command = new SqlCommand("spUpdateHardwareProfile", conn))
            {
                command.CommandType = CommandType.StoredProcedure;

                var param = command.Parameters.Add("@UserID", SqlDbType.Int);
                param.Value = hardwareProfile.UserID;

                param = command.Parameters.Add("@ComputerName", SqlDbType.NVarChar, 256);
                param.Value = hardwareProfile.ComputerName;

                param = command.Parameters.Add("@LanguageID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "Language", hardwareProfile.LanguageName);

                param = command.Parameters.Add("@IsLaptop", SqlDbType.Bit);
                param.Value = hardwareProfile.IsLaptop;

                param = command.Parameters.Add("@DirectXVersionID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "DirectXVersion", hardwareProfile.DirectXVersionName);

                // OS
                param = command.Parameters.Add("@OSVersionID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "OperatingSystemVersion",
                    GetNormalizedOSName(hardwareProfile.OSVersionName));

                param = command.Parameters.Add("@OsIs64Bit", SqlDbType.Bit);
                param.Value = hardwareProfile.OSIs64Bit;

                param = command.Parameters.Add("@OSServicePackID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "ServicePackVersion", hardwareProfile.OSServicePackName);

                param = command.Parameters.Add("@OSMajorVersion", SqlDbType.SmallInt);
                param.Value = hardwareProfile.OSVersionMajor;

                param = command.Parameters.Add("@OSMajorRevision", SqlDbType.SmallInt);
                param.Value = hardwareProfile.OSVersionMajorRevision;

                param = command.Parameters.Add("@OSMinorVersion", SqlDbType.SmallInt);
                param.Value = hardwareProfile.OSVersionMinor;

                param = command.Parameters.Add("@OSMinorRevision", SqlDbType.SmallInt);
                param.Value = hardwareProfile.OSVersionMinorRevision;

                //RAM
                param = command.Parameters.Add("@RamManufacturerID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "RamManufacturer", hardwareProfile.RamManufacturerName);

                param = command.Parameters.Add("@RamTypeID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "RamType", hardwareProfile.RamTypeName);

                param = command.Parameters.Add("@RamSize", SqlDbType.SmallInt);
                if (hardwareProfile.RamSize > Int16.MaxValue)
                {
                    param.Value = Int16.MaxValue;
                }
                else
                {
                    param.Value = hardwareProfile.RamSize;
                }

                param = command.Parameters.Add("@RamFrequency", SqlDbType.SmallInt);
                param.Value = hardwareProfile.RamFrequency;

                // CPU
                param = command.Parameters.Add("@CpuManufacturerID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "CpuManufacturer", hardwareProfile.CpuManufacturerName);

                param = command.Parameters.Add("@CpuFrequency", SqlDbType.SmallInt);
                param.Value = hardwareProfile.CpuFrequency;

                param = command.Parameters.Add("@CpuModelID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "CpuModel", hardwareProfile.CpuDescription);

                param = command.Parameters.Add("@CpuCount", SqlDbType.TinyInt);
                param.Value = hardwareProfile.CpuCount;

                // GPU
                param = command.Parameters.Add("@GpuManufacturerID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "GpuManufacturer", hardwareProfile.GpuManufacturerName);

                param = command.Parameters.Add("@GpuModelID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "GpuModel", hardwareProfile.GpuDescription);

                param = command.Parameters.Add("@GpuMemorySize", SqlDbType.SmallInt);
                param.Value = hardwareProfile.GpuMemorySize;

                param = command.Parameters.Add("@GpuCount", SqlDbType.SmallInt);
                param.Value = hardwareProfile.GpuCount;

                param = command.Parameters.Add("@GpuDriverNameID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "GpuDriverName", hardwareProfile.GpuDriverName);

                param = command.Parameters.Add("@GpuDriverVersionID", SqlDbType.Int);
                param.Value = GetLookupID_DriverVersion(conn, "GpuDriverVersion", hardwareProfile.GpuDriverVersionName,
                    hardwareProfile.GpuDriverDate);

                // Display                
                param = command.Parameters.Add("@MonitorModelID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "MonitorModel", hardwareProfile.MonitorDescription);

                param = command.Parameters.Add("@MonitorResolutionWidth", SqlDbType.SmallInt);
                param.Value = hardwareProfile.MonitorResolutionWidth;

                param = command.Parameters.Add("@MonitorResolutionHeight", SqlDbType.SmallInt);
                param.Value = hardwareProfile.MonitorResolutionHeight;

                param = command.Parameters.Add("@MonitorCount", SqlDbType.TinyInt);
                param.Value = hardwareProfile.MonitorCount;

                // Sound
                param = command.Parameters.Add("@SoundIsPresent", SqlDbType.Bit);
                param.Value = hardwareProfile.SoundIsPresent;

                param = command.Parameters.Add("@SoundIsPci", SqlDbType.Bit);
                param.Value = hardwareProfile.SoundIsPci;

                param = command.Parameters.Add("@SoundIsHwAccel", SqlDbType.Bit);
                param.Value = hardwareProfile.SoundIsHwAccel;

                param = command.Parameters.Add("@SoundModelID", SqlDbType.Int);
                param.Value = GetLookupID(conn, "SoundModel", hardwareProfile.SoundDescription);

                // HDD
                param = command.Parameters.Add("@HDDCount", SqlDbType.TinyInt);
                param.Value = hardwareProfile.HDDCount;

                param = command.Parameters.Add("@HDDTotalSpace", SqlDbType.Int);
                param.Value = hardwareProfile.HDDTotalSpace;

                param = command.Parameters.Add("@HDDFreeSpace", SqlDbType.Int);
                param.Value = hardwareProfile.HDDFreeSpace;

                command.ExecuteNonQuery();
            }
        }

        private string GetNormalizedOSName(string name)
        {
            string loweredName = name.ToLowerInvariant();
            if (loweredName.Contains(" 7"))
            {
                return "Windows 7";
            }
            else if (loweredName.Contains(" vista"))
            {
                return "Windows Vista";
            }
            else if (loweredName.Contains(" server") && loweredName.Contains("2008"))
            {
                return "Windows Server 2008";
            }
            else if (loweredName.Contains(" server") && loweredName.Contains("2003"))
            {
                return "Windows Server 2003";
            }
            else if (loweredName.Contains(" xp"))
            {
                return "Windows XP";
            }            
            else if (loweredName.Contains(" 2000"))
            {
                return "Windows 2000";
            }
            else if (loweredName.Contains(" me"))
            {
                return "Windows ME";
            }
            else if (loweredName.Contains(" 98"))
            {
                return "Windows 98";
            }
            else if (loweredName.Contains(" 95"))
            {
                return "Windows 95";
            }
            
            return "Unknown";
        }
              
        private int GetLookupID_DriverVersion(SqlConnection conn, string tableName, string labelValue, DateTime driverDate)
        {
            DateTime date = driverDate;
            if (string.IsNullOrEmpty(labelValue) || date.Year == DateTime.MinValue.Year)
            {
                labelValue = "Unknown";
                date = new DateTime(2000, 1, 1);
            }

            labelValue = labelValue.Trim();

            var lookupKey = GetLookupKey(tableName, labelValue);
            if (LookupCache.ContainsKey(lookupKey))
            {
                return LookupCache[lookupKey];
            }
            using (var command = conn.CreateCommand())
            {
                command.CommandText = "insert into " + tableName +
                                      "(LookupLabel, DriverDate) output inserted.LookupID values(@LookupLabel, @DriverDate)";
                var param = command.Parameters.Add("@LookupLabel", System.Data.SqlDbType.NVarChar, 256);
                param.Value = labelValue;
                param = command.Parameters.Add("@DriverDate", System.Data.SqlDbType.DateTime);
                param.Value = date;
                var id = (int) command.ExecuteScalar();
                LookupCache[lookupKey] = id;
                return id;
            }
        }
    }
}
