﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using Curse.ClientService.Models;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using System.Threading;
using Curse.Logging;

namespace Curse.ClientService
{
    public static class UsageReportingCache
    {
        private static readonly ConcurrentDictionary<string, int> Lookups = new ConcurrentDictionary<string, int>();
        private const int WriteIntervalMilliseconds = 500;
        private static readonly ConcurrentQueue<UsageProfile> _userProfiles = new ConcurrentQueue<UsageProfile>();
        private static readonly string DatabaseConnectionString;

        private static SqlConnection GetDatabaseConnection()
        {
            try
            {
                var conn = new SqlConnection(DatabaseConnectionString);
                conn.Open();
                return conn;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to open database connection.");
                throw;
            }
        }

        
        public static void Initialize() { }


        static UsageReportingCache()
        {
            DatabaseConnectionString = ConfigurationManager.ConnectionStrings["ClientUsageReporting"].ConnectionString;
            var writeToDatabaseThread = new Thread(WriteToDatabaseThread) { IsBackground = true };
            writeToDatabaseThread.Start();
        }

        public static bool SaveReport(int userID, UsageProfile usageProfile)
        {
            if (usageProfile == null)
            {
                return false;
            }

            usageProfile.UserID = userID;
            usageProfile.Timestamp = DateTime.UtcNow;
            _userProfiles.Enqueue(usageProfile);
            return true;
        }

        private const int ProfilesPerIteration = 1000;

        private static void WriteToDatabase()
        {           
            if (_userProfiles.Count == 0)
            {
                return;
            }

            if (_userProfiles.Count > ProfilesPerIteration)
            {
                Logger.Warn("The usage profile queue is larger than the iteration process size: " + _userProfiles.Count);
            }

            var profiles = new List<UsageProfile>(_userProfiles.Count);

            UsageProfile profile = null;
            var added = 0;

            while (_userProfiles.TryDequeue(out profile) && added <= ProfilesPerIteration) // Only save out 500 records at a time
            {
                profiles.Add(profile);
                ++added;
            }


            try
            {
                SaveVoiceUsers(profiles);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Exception occurred while saving voice users.");
            }

            try
            {
                SaveVoiceUserPlatforms(profiles);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Exception occurred while saving voice user platforms.");
            }
        }

        private static void WriteToDatabaseThread()
        {
            while (true)
            {
                try
                {
                    Thread.Sleep(WriteIntervalMilliseconds);
                    WriteToDatabase();
                }
                catch (ThreadAbortException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "CHardwareProfile SaveThread Exception");
                }
            }
        }

        private static void SaveVoiceUsers(List<UsageProfile> usageProfiles)
        {
            if (usageProfiles.Count == 0)
            {
                return;
            }

            using (var conn = new SqlConnection(DatabaseConnectionString))
            {
                conn.Open();
                var cmd = conn.CreateCommand();

                var userTable = new DataTable();
                userTable.Columns.Add("UserID", typeof(int));
                userTable.Columns.Add("DateFirstSeen", typeof(DateTime));
                userTable.Columns.Add("DateLastSeen", typeof(DateTime));
                userTable.Columns.Add("ClientVersionID", typeof(int));
                userTable.Columns.Add("OperatingSystemVersionID", typeof(int));
                userTable.Columns.Add("OperatingSystemPlatformID", typeof(int));
                userTable.Columns.Add("MainWindowHeight", typeof(float));
                userTable.Columns.Add("MainWindowWidth", typeof(float));
                userTable.Columns.Add("SoftwareRendering", typeof(bool));
                userTable.Columns.Add("DotNetVersion", typeof(float));
                userTable.Columns.Add("CountryCode", typeof(string));
                userTable.Columns.Add("ReferralCode", typeof(string));

                userTable.BeginLoadData();

                var processedUserIDs = new HashSet<int>();

                foreach (var usageProfile in usageProfiles)
                {
                    if (processedUserIDs.Contains(usageProfile.UserID))
                    {
                        continue;
                    }

                    var row = userTable.NewRow();
                    row["UserID"] = usageProfile.UserID;
                    row["DateFirstSeen"] = usageProfile.Timestamp;
                    row["DateLastSeen"] = usageProfile.Timestamp;
                    row["ClientVersionID"] = GetLookupID("ClientVersion", usageProfile.ClientVersion);
                    row["OperatingSystemVersionID"] = GetLookupID("OperatingSystemVersion", usageProfile.OperatingSystemVersion);
                    row["OperatingSystemPlatformID"] = GetLookupID("OperatingSystemPlatform", usageProfile.OperatingSystemPlatform);
                    row["MainWindowHeight"] = (int)usageProfile.MainWindowHeight;
                    row["MainWindowWidth"] = (int)usageProfile.MainWindowWidth;
                    row["SoftwareRendering"] = usageProfile.SoftwareRendering;
                    row["DotNetVersion"] = usageProfile.DotNetVersion;

                    if (!string.IsNullOrEmpty(usageProfile.CountryCode))
                    {
                        row["CountryCode"] = usageProfile.CountryCode;
                    }
                    else
                    {
                        row["CountryCode"] = DBNull.Value;
                    }

                    if (!string.IsNullOrEmpty(usageProfile.ReferralCode) && usageProfile.ReferralCode.Length <= 6)
                    {
                        row["ReferralCode"] = usageProfile.ReferralCode;
                    }
                    else
                    {
                        row["ReferralCode"] = DBNull.Value;
                    }

                    userTable.Rows.Add(row);
                    processedUserIDs.Add(usageProfile.UserID);
                }

                // Save out the users:
                cmd.CommandText = "create table #VoiceUser (UserID int, DateFirstSeen datetime, DateLastSeen datetime, ClientVersionID int, OperatingSystemVersionID int, OperatingSystemPlatformID int,"
                + " MainWindowHeight float, MainWindowWidth float, SoftwareRendering bit, DotNetVersion float, CountryCode char(2) NULL, ReferralCode varchar(6) null);";
                cmd.ExecuteNonQuery();
                using (var bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, null))
                {
                    bulk.BulkCopyTimeout = 300;
                    bulk.DestinationTableName = "#VoiceUser";
                    bulk.WriteToServer(userTable);
                }

                // Update existing
                cmd.CommandText = "UpdateVoiceUsers";
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.CommandTimeout = 300;
                cmd.ExecuteNonQuery();
            }
        }

        private static void SaveVoiceUserPlatforms(List<UsageProfile> usageProfiles)
        {
            if (usageProfiles.Count == 0)
            {
                return;
            }

            using (var conn = new SqlConnection(DatabaseConnectionString))
            {
                conn.Open();
                var cmd = conn.CreateCommand();

                var userTable = new DataTable();
                userTable.Columns.Add("UserID", typeof(int));
                userTable.Columns.Add("Platform", typeof(byte));
                userTable.Columns.Add("DateFirstSeen", typeof(DateTime));
                userTable.Columns.Add("DateLastSeen", typeof(DateTime));
                userTable.Columns.Add("ClientVersionID", typeof(int));
                userTable.Columns.Add("OperatingSystemVersionID", typeof(int));
                userTable.BeginLoadData();

                var processed = new HashSet<string>();

                foreach (var usageProfile in usageProfiles)
                {
                    var row = userTable.NewRow();
                    var key = usageProfile.UserID + "-" + usageProfile.Platform;

                    if (processed.Contains(key))
                    {
                        continue;
                    }

                    processed.Add(key);
                    row["UserID"] = usageProfile.UserID;
                    row["Platform"] = (byte)usageProfile.Platform;
                    row["DateFirstSeen"] = usageProfile.Timestamp;
                    row["DateLastSeen"] = usageProfile.Timestamp;
                    row["ClientVersionID"] = GetLookupID("ClientVersion", usageProfile.ClientVersion);
                    row["OperatingSystemVersionID"] = GetLookupID("OperatingSystemVersion", usageProfile.OperatingSystemVersion);
                    userTable.Rows.Add(row);
                }

                // Save out the users:
                cmd.CommandText = "create table #VoiceUserPlatform (UserID int, Platform tinyint, DateFirstSeen datetime, DateLastSeen datetime, ClientVersionID int, OperatingSystemVersionID int);";
                cmd.ExecuteNonQuery();
                using (var bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, null))
                {
                    bulk.BulkCopyTimeout = 300;
                    bulk.DestinationTableName = "#VoiceUserPlatform";
                    bulk.WriteToServer(userTable);
                }


                cmd.CommandText = "UpdateVoiceUserPlatforms";
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.CommandTimeout = 300;
                cmd.ExecuteNonQuery();
            }
        }

        private static string GetLookupKey(string tableName, string value)
        {
            return tableName.ToLowerInvariant() + "-" + value.ToLowerInvariant();
        }

        private static int? GetLookupID(string tableName, string labelValue)
        {
            if (string.IsNullOrEmpty(labelValue))
            {
                labelValue = "Unknown";
            }

            int lookupID;
            labelValue = labelValue.Trim();
            var lookupKey = GetLookupKey(tableName, labelValue);

            if (Lookups.TryGetValue(lookupKey, out lookupID))
            {
                return lookupID;
            }

            var databaseID = GetLookupIDFromDatabase(tableName, labelValue);

            if (databaseID.HasValue)
            {
                return databaseID.Value;
            }

            return InsertLookup(lookupKey, tableName, labelValue);

        }

        private static int? GetLookupIDFromDatabase(string tableName, string labelValue)
        {
            using (var conn = GetDatabaseConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "select [LookupID] from [" + tableName + "] where [LookupLabel] = @LookupLabel";
                    var param = command.Parameters.Add("@LookupLabel", System.Data.SqlDbType.NVarChar, 256);
                    param.Value = labelValue;
                    var id = command.ExecuteScalar();

                    if (id == null || DBNull.Value.Equals(id))
                    {
                        return null;
                    }

                    var lookupKey = GetLookupKey(tableName, labelValue);
                    Lookups.TryAdd(lookupKey, (int)id);
                    return (int)id;
                }
            }
        }

        private static int? InsertLookup(string lookupKey, string tableName, string labelValue)
        {

            try
            {
                using (var conn = GetDatabaseConnection())
                {
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandText = "insert into [" + tableName + "] ([LookupLabel]) output inserted.[LookupID] values(@LookupLabel)";
                        var param = command.Parameters.Add("@LookupLabel", System.Data.SqlDbType.NVarChar, 256);
                        param.Value = labelValue;
                        var id = (int)command.ExecuteScalar();
                        Lookups.TryAdd(lookupKey, id);
                        return id;
                    }
                }
            }
            catch (Exception ex)
            {
                int lookupID;
                if (Lookups.TryGetValue(lookupKey, out lookupID))
                {
                    return lookupID;
                }

                Logger.Error(ex, "Failed to get lookup id: " + tableName + " - " + labelValue);
            }

            return null;
        }
    }
}
