﻿using System;
using System.Data;
using Curse.ClientService.Models;
using System.Data.SqlClient;
using Curse.Extensions;
using System.Configuration;
using System.Collections.Generic;
using System.Linq;

namespace Curse.ClientService.Extensions
{
    public static class SavedGameExtension
    {
        private static string _databaseConnectionString;
        
        static SavedGameExtension()
        {
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
        }

        public static void SetFromDataReader(this SavedGame savedGame, SqlDataReader reader, Dictionary<int, List<SavedGameRevision>> revisions)
        {
            savedGame.ID = reader.GetInt32(reader.GetOrdinal("ID"));
            savedGame.UserID = reader.GetInt32(reader.GetOrdinal("UserID"));
            savedGame.GameID = reader.GetInt32(reader.GetOrdinal("GameID"));
            savedGame.Name = reader.GetString(reader.GetOrdinal("Name"));
            savedGame.Description = reader.GetNullableValue<string>("Description");
            savedGame.FileName = reader.GetString(reader.GetOrdinal("FileName"));
            savedGame.DateCreated = reader.GetDateTime(reader.GetOrdinal("DateCreated"));
            savedGame.DateModified = reader.GetNullableValue<DateTime>("DateModified");
            savedGame.Status = (ESavedGameStatus)reader.GetByte(reader.GetOrdinal("Status"));
            savedGame.DefaultRevisionID = reader.GetNullableValue<int>("DefaultRevisionID");

            if (revisions.ContainsKey(savedGame.ID))
            {
                savedGame.SavedGameRevisions = revisions[savedGame.ID];
            }
        }

        public static void Save(this SavedGame savedGame)
        {
            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                var cmd = conn.CreateCommand();
                cmd.CommandText = "spUpdateSavedGame";
                cmd.CommandType = CommandType.StoredProcedure;

                cmd.Parameters.Add("ID", SqlDbType.Int).Value = savedGame.ID;
                cmd.Parameters["ID"].Direction = ParameterDirection.InputOutput;
                if (savedGame.ID == 0)
                {
                    cmd.Parameters["ID"].Value = DBNull.Value;
                }

                cmd.Parameters.Add("UserID", SqlDbType.Int).Value = savedGame.UserID;
                cmd.Parameters.Add("GameID", SqlDbType.Int).Value = savedGame.GameID;
                cmd.Parameters.Add("Name", SqlDbType.NVarChar).Value = savedGame.Name;
                cmd.Parameters.Add("FileName", SqlDbType.NVarChar).Value = savedGame.FileName;
                cmd.Parameters.Add("Description", SqlDbType.NVarChar).Value = savedGame.Description ?? string.Empty;
                cmd.Parameters.Add("Status", SqlDbType.TinyInt).Value = savedGame.Status;
                cmd.Parameters.Add("MaxSavedGames", SqlDbType.Int).Value = 0;
                cmd.Parameters.Add("DateModified", SqlDbType.DateTime).Value = savedGame.DateModified;

                conn.Open();
                cmd.ExecuteNonQuery();

                savedGame.ID = (int)cmd.Parameters["ID"].Value;
                savedGame.SavedGameRevisions.ForEach(p => p.SavedGameID = savedGame.ID);
            }

            savedGame.SavedGameRevisions.ForEach(p => p.Save());
        }

        public static void SetDefaultRevision(this SavedGame savedGame, int revisionId)
        {
            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                var cmd = new SqlCommand("UPDATE SavedGame SET DefaultRevisionID = @RevisionID WHERE ID = @ID", conn);
                cmd.Parameters.Add("ID", SqlDbType.Int).Value = savedGame.ID;
                cmd.Parameters.Add("RevisionID", SqlDbType.Int).Value = revisionId;

                conn.Open();
                cmd.ExecuteNonQuery();
            }
        }

        public static void Delete(this SavedGame savedGame)
        {
            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                var cmd = conn.CreateCommand();
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.CommandText = "spDeleteSavedGame";
                cmd.Parameters.Add("ID", SqlDbType.Int).Value = savedGame.ID;

                conn.Open();
                cmd.ExecuteNonQuery();
            }
        }
    }
}