﻿using System;
using System.Linq;
using System.Configuration;
using System.Threading;
using System.Net;
using System.IO;
using Curse.Logging;
using ICSharpCode.SharpZipLib.Zip;
using System.Text.RegularExpressions;
using System.Data;
using System.Data.SqlClient;

namespace Curse.ClientService.GeoCoding
{
    public static class IPDatabaseUpdater
    {

        private static readonly Regex CsvSplit = new Regex(",(?!(?:[^\",]|[^\"],[^\"])+\")", RegexOptions.Compiled);        
        private const string DownloadUrl = "http://www.maxmind.com/app/download_new?edition_id=108&suffix=zip&license_key=2bPFVu3togLd";
        private const string BackupLocation = @"\\smb02d-atl.curse.us\GeoLocationService\Geo.zip";
        private const string Username = "accounting@curse.com";
        private const string Password = "7u1y3e8u";
        private static readonly string DatabaseConnectionString = null;
        private static readonly string JobMachineName = null;
        private static readonly LogCategory Logger = new LogCategory("IPDatabaseUpdater");


        static IPDatabaseUpdater()
        {            
            DatabaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
            JobMachineName = ConfigurationManager.AppSettings["JobMachineName"].ToLower();
        }

        internal static void Initialize()
        {
            new Thread(UpdateIPDatabaseThread) { IsBackground = true }.Start();
        }

        private static void UpdateIPDatabaseThread()
        {
            while (true)
            {
                if (string.IsNullOrEmpty(JobMachineName) || Environment.MachineName.ToLower().Equals(JobMachineName))
                {
                    try
                    {
                        UpdateIPDatabase();
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Unable to update IP database");
                    }
                }

                try
                {
                    GeoIPHelper.Instance.PopulateGeoData();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to build geo IP cache!");
                }
                
                Thread.Sleep(TimeSpan.FromHours(12));
            }
        }

        private static void UpdateIPDatabase()
        {
            Logger.Info("Started");
            
            // First, download the CSV
            var credentialCache = new CredentialCache();
            credentialCache.Add(new Uri(DownloadUrl), "Basic", new NetworkCredential(Username, Password));

            var tempfile = Path.Combine(Path.GetTempPath(), Utility.UniqueNumber + ".zip"); ;
            using (var client = new WebClient())
            {
                client.Proxy = null;
                client.Credentials = credentialCache;

                try
                {
                    client.DownloadFile(DownloadUrl, tempfile);
                }
                catch (Exception)
                {
                    Logger.Info("Could not download geo IP CSV file from provider.");

                    try
                    {
                        client.DownloadFile(BackupLocation, tempfile);
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Could not find the backup geo IP file on share.");
                    }
                    return;                    
                }
                
            }

            var unzipPath = Path.Combine(Path.GetTempPath(), "IPDatabase");

            var unzipper = new FastZip();            
            unzipper.ExtractZip(tempfile, unzipPath, FastZip.Overwrite.Always, null, null, null, false);
            
            // Find any CSV file in the folder
            var filePath = Directory.GetFiles(unzipPath, "*.csv", SearchOption.AllDirectories).FirstOrDefault();

            if (filePath == null)
            {
                Logger.Warn("IP Database Update Failed! Could not find CSV file in '" + unzipPath + "'");
                return;
            }

            // Get a data table for bulk copying our data
            DataTable dt = GenerateIPTable();
            dt.BeginLoadData();

            // File format is: "begin_ip","end_ip","begin_num","end_num","countryCode","countryName"

            using (var reader = new StreamReader(filePath))
            {
                while (true)
                {
                    var line = reader.ReadLine();
                    if (line == null)
                    {
                        break;
                    }

                    var fields = CsvSplit.Split(line);

                    if (fields.Length != 6)
                    {
                        continue;
                    }

                    // Remove the quotes
                    for (int j = 0; j < fields.Length; j++)
                    {
                        fields[j] = fields[j].Replace("\"", string.Empty);
                    }
                    
                    string beginIPString = fields[2];
                    Int64 beginIP;
                    string endIpString = fields[3];
                    Int64 endIP;
                    string country = fields[4];

                    DataRow row = dt.NewRow();
                    row[0] = country;

                    if (!Int64.TryParse(beginIPString, out beginIP))
                    {
                        continue;
                    }

                    if (!Int64.TryParse(endIpString, out endIP))
                    {
                        continue;
                    }

                    row[1] = beginIP;
                    row[2] = endIP;

                    dt.Rows.Add(row);
                }
            }

            try
            {
                using (var conn = new SqlConnection(DatabaseConnectionString))
                {
                    conn.Open();

                    using (var txn = conn.BeginTransaction())
                    {

                        using (var cmd = conn.CreateCommand())
                        {
                            cmd.Transaction = txn;
                            cmd.CommandText = "truncate table CountryIPRange";
                            cmd.ExecuteNonQuery();

                            using (var bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, txn))
                            {
                                bulk.DestinationTableName = "CountryIPRange";
                                bulk.WriteToServer(dt);
                            }

                            txn.Commit();
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to commit changed to database!");
                return;
            }


            Logger.Info("Finished");
           
        }

        private static DataTable GenerateIPTable()
        {
            var dt = new DataTable("CountryIPRange");
            dt.Columns.Add("CountryCode", typeof(string));
            dt.Columns.Add("IPRangeStart", typeof(Int64));
            dt.Columns.Add("IPRangeEnd", typeof(Int64));
            return dt;
        }
    }
}
