﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Configuration;
using System.Data.SqlClient;
using System.IO;
using System.Linq;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.Threading;
using System.Web;
using Curse.ClientService.Extensions;
using Curse.ClientService.Models;
using Curse.Logging;
using Curse.ServiceAuthentication;
using Curse.Extensions;
using Newtonsoft.Json;

namespace Curse.ClientService.Managers
{
    public class BugReportingManager
    {
         private static readonly ConcurrentDictionary<Int64, Int32> StackTraceToReportID =
            new ConcurrentDictionary<Int64, Int32>();

        private static readonly ConcurrentDictionary<string, Int32> VersionStringToID =
            new ConcurrentDictionary<string, Int32>();
        
        private const int MaxQueueSize = 500;
        private static readonly ConcurrentQueue<BugReport> CrashReportQueue = new ConcurrentQueue<BugReport>();
        private static readonly ConcurrentQueue<BugReport> BugReportQueue = new ConcurrentQueue<BugReport>();
        private static readonly string CrashReportFolderPath;
        private static readonly bool ProcessGameCrashes = true;
        private const bool AllowUnknownVersions = true;


        private static DateTime? _lastCorruptInstallReport = null;

        private static readonly BugReportingManager _instance = new BugReportingManager();

        public static BugReportingManager Instance
        {
            get { return _instance; }
        }

        static BugReportingManager()
        {
            CrashReportFolderPath = ConfigurationManager.AppSettings["CrashReportsPath"];

            try
            {
                ProcessGameCrashes = bool.Parse(ConfigurationManager.AppSettings["ProcesGameCrashes"]);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to parse game crash config.");
            }

            try
            {
                if (!Directory.Exists(CrashReportFolderPath))
                {
                    Directory.CreateDirectory(CrashReportFolderPath);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to create crash report folder at '" + CrashReportFolderPath);
            }

            new Thread(ProcessQueueThread) { IsBackground = true }.Start();
        }

        public BugReportResponse SubmitBugReport(BugReport report)
        {
            try
            {
                report.IPAddress = ClientIPaddress;

                if (report.ClientVersion == null)
                {
                    return new BugReportResponse() { Status = BugReportStatus.InvalidData };
                }

                if (report.UsernameOrEmail == null)
                {
                    return new BugReportResponse() { Status = BugReportStatus.InvalidData };
                }               

                if (report.Type == BugReportType.AutoReport && report.Title.Equals("CorruptInstallException"))
                {
                    if (_lastCorruptInstallReport.HasValue &&
                        DateTime.UtcNow - _lastCorruptInstallReport.Value < TimeSpan.FromMinutes(30))
                    {
                        return new BugReportResponse() { Status = BugReportStatus.Successful };
                    }

                    _lastCorruptInstallReport = DateTime.UtcNow;
                }

                if (AllowUnknownVersions)
                {
                    report.BypassVersionFilter = true;
                }

                var status = BugReportStatus.Error;

                switch (report.Type)
                {
                    case BugReportType.CrashReport:
                        status = ProcessCrashReport(report);
                        break;
                    case BugReportType.GameCrash:
                        status = ProcessGameCrashes ? ProcessBugReport(report) : BugReportStatus.Successful;
                        break;
                    default:
                        status = ProcessBugReport(report);
                        break;
                }


                return new BugReportResponse() { Status = status };

            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to process bug report");
                return new BugReportResponse()
                {
                    Status = BugReportStatus.Error
                };
            }
        }
        

        private BugReportStatus ProcessCrashReport(BugReport report)
        {
            if (CrashReportQueue.Count > MaxQueueSize)
            {
                return BugReportStatus.Error;
            }

            CrashReportQueue.Enqueue(report);

            return BugReportStatus.Successful;
        }

        private BugReportStatus ProcessBugReport(BugReport report)
        {
            if (BugReportQueue.Count > MaxQueueSize)
            {
                return BugReportStatus.Error;
            }

            BugReportQueue.Enqueue(report);

            return BugReportStatus.Successful;
        }

        private static void ProcessQueueThread()
        {
            while (true)
            {                
                try
                {
                    Thread.Sleep(1000);

                    BugReport report = null;

                    while (BugReportQueue.TryDequeue(out report))
                    {
                        try
                        {
                            SaveBugReportToDatabase(report);
                        }
                        catch (Exception ex)
                        {
                            
                            Logger.Warn(ex, "Bug report was not saved to the database, due to invalid data or exception.", new { report.Platform, report.ClientVersion, report.UsernameOrEmail });
                        }
     
                        Thread.Sleep(10); // Caps this to 100 reports per second
                    }

                    while (CrashReportQueue.TryDequeue(out report))
                    {
                        try
                        {
                            SaveCrashReportToDatabase(report);
                        }
                        catch (Exception ex)
                        {
                            Logger.Warn(ex, "Crash report was not saved to the database, due to invalid data or exception.", new { report.Platform, report.ClientVersion, report.UsernameOrEmail });                                                        
                        }                        
                        Thread.Sleep(10); // Caps this to 100 reports per second
                    }

                    
                }
                catch (ThreadAbortException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to process bug report queue");
                }
            }
        }


        private static void SaveCrashReportToDatabase(BugReport report)
        {
            // Get the user ID
            var userID = AuthenticationProvider.GetUserID(report.UsernameOrEmail);


            // Get the client version ID
            var clientVersionID = GetOrCreateVersion(report.ClientVersion, report.Platform);

            if (clientVersionID == null)
            {
                Logger.Warn("Unable to save crash report. The client version supplied could not be retrieved or created!", report.ClientVersion);
                return;
            }


            // See if a crash report already exists with the same stack trace
            int? existingReportID = null;

            if (report.StackTraceHash.HasValue)
            {
                if (!TryGetExistingReportIDFromHash(report.StackTraceHash.Value, out existingReportID))
                {
                    return;
                }
            }

            if (existingReportID.HasValue)
            {
                // Update the existing report data                
                using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["ClientBugReporting"].ConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText =
                            "UPDATE [ClientBugReport] SET VersionID = @VersionID, StepsToReproduce = COALESCE(StepsToReproduce, @StepsToReproduce),  DateLastReported = GETUTCDATE(), ReportedCount = ReportedCount + 1 WHERE ID = @ID";
                        cmd.Parameters.AddWithValue("@VersionID", clientVersionID.Value);
                        cmd.Parameters.AddWithValue("@StepsToReproduce", report.StepsToReproduce.CoalesceToDBNull());
                        cmd.Parameters.AddWithValue("@ID", existingReportID.Value);
                        cmd.ExecuteNonQuery();
                    }
                }
            }
            else // Save a new client report
            {
                existingReportID = SaveReportToDatabase(report, userID, clientVersionID.Value);
            }

            // Save the incident
            SaveIncidentToDatabase(report, userID, existingReportID.Value, clientVersionID.Value);            
        }
        
        private static void SaveBugReportToDatabase(BugReport report)
        {
            // Get the user ID
            var userID = AuthenticationProvider.GetUserID(report.UsernameOrEmail);


            // Get the client version ID
            var clientVersionID = GetOrCreateVersion(report.ClientVersion, report.Platform);

            if (clientVersionID == null)
            {
                Logger.Warn("Unable to save bug report. The client version supplied could not be retrieved or created!", report.ClientVersion);
                return;
            }

            SaveReportToDatabase(report, userID, clientVersionID.Value);            
        }

        public System.Net.IPAddress ClientIPaddress
        {
            get
            {
                try
                {
                    // Start up the instance on the host
                    var context = OperationContext.Current;
                    var prop = context.IncomingMessageProperties;
                    var endpoint = prop[RemoteEndpointMessageProperty.Name] as RemoteEndpointMessageProperty;
                    return System.Net.IPAddress.Parse(endpoint.Address);
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, "Unable to parse IP address from request!");
                    return System.Net.IPAddress.None;
                }
            }
        }

        private static void SaveIncidentToDatabase(BugReport report, int userID, int reportID, int clientVersionID)
        {            
            // Update the existing report data                
            using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["ClientBugReporting"].ConnectionString))
            {
                conn.Open();

                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "INSERT INTO [ClientCrashIncident] (ReportID, VersionID, DateCreated, UserID, IPAddress)"
                                      + " OUTPUT INSERTED.ID"
                                      + " VALUES(@ReportID, @VersionID, GETUTCDATE(), @UserID, @IPAddress)";
                    cmd.Parameters.AddWithValue("@ReportID", reportID);
                    cmd.Parameters.AddWithValue("@VersionID", clientVersionID);
                    cmd.Parameters.AddWithValue("@UserID", userID);
                    cmd.Parameters.AddWithValue("@IPAddress", report.IPAddress.GetAddressBytes());
                    cmd.ExecuteNonQuery();
                }
            }
        }

        private static string GetHighLowPath(int id)
        {
            return Path.Combine((id / 1000).ToString(), (id % 1000).ToString());
        }

        private static string SaveBugReportAttachment(int attachmentID, string fileName, byte[] dumpData)
        {
            string relativePath = GetHighLowPath(attachmentID);
            string fullPath = Path.Combine(CrashReportFolderPath, relativePath);

            if (!Directory.Exists(fullPath))
            {
                Directory.CreateDirectory(fullPath);
            }

            string relativefilePath = Path.Combine(relativePath, fileName);
            string fullfilePath = Path.Combine(CrashReportFolderPath, relativefilePath);
            File.WriteAllBytes(fullfilePath, dumpData);
            return relativefilePath;
        }


        private static bool IsValidAttachment(BugReportAttachment attachment)
        {
            if (attachment.FileName.IsNullOrEmpty())
            {
                Logger.Warn("Skipping bug report attachment: Missing filename.");
                return false;
            }

            if (attachment.FileName.Length > 64)
            {
                Logger.Warn("Skipping bug report attachment: Filename '" + attachment.FileName + "' too long");
                return false;
            }

            if (attachment.FileContents == null)
            {
                Logger.Warn("Skipping bug report attachment: File contents are null.");
                return false;
            }

            if (attachment.FileContents.Length == 0)
            {
                Logger.Warn("Skipping bug report attachment: File contents are 0 bytes.");
                return false;
            }

            return true;
        }

        private static int SaveReportToDatabase(BugReport report, int userID, int clientVersionID)
        {
            // Reset last attempted host ot 0
            if (report.LastAttemptedVoiceHostID.HasValue && report.LastAttemptedVoiceHostID.Value == 0)
            {
                report.LastAttemptedVoiceHostID = null;
            }

            // Update the existing report data                
            using (
                var conn =
                    new SqlConnection(ConfigurationManager.ConnectionStrings["ClientBugReporting"].ConnectionString))
            {
                conn.Open();

                var bugReportID = 0;
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = string.Join(" ",
                        "INSERT INTO [ClientBugReport] (UserID, VersionID, Title, Type, Priority, StackTraceHash, UserSettings,",
                        "SystemInformation, IncompatibleProcesses, GameConfiguration, TimeConfiguration, LogData, ExceptionDetails,",
                        "ApplicationLocation, StepsToReproduce, LastAttemptedVoiceHostID, OverlayExitCode, RunningProcesses,",
                        "OverlayGameState, CallQuality)",
                        "OUTPUT INSERTED.ID",
                        "VALUES(@UserID, @VersionID, @Title, @Type, @Priority, @StackTraceHash, @UserSettings,",
                        "@SystemInformation, @IncompatibleProcesses, @GameConfiguration, @TimeConfiguration, @LogData, @ExceptionDetails,",
                        "@ApplicationLocation, @StepsToReproduce, @LastAttemptedVoiceHostID, @OverlayExitCode, @RunningProcesses,",
                        "@OverlayGameState, @CallQuality)");

                    cmd.Parameters.AddWithValue("@UserID", userID);
                    cmd.Parameters.AddWithValue("@VersionID", clientVersionID);
                    cmd.Parameters.AddWithValue("@Title", report.Title);
                    cmd.Parameters.AddWithValue("@Type", report.Type);
                    cmd.Parameters.AddWithValue("@Priority", report.Priority);
                    cmd.Parameters.AddWithValue("@StackTraceHash", report.StackTraceHash.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@UserSettings", report.UserSettings.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@SystemInformation", report.SystemInformation.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@IncompatibleProcesses", report.IncompatibleProcesses.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@GameConfiguration", report.GameConfiguration.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@TimeConfiguration", report.TimeConfiguration.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@LogData", report.LogData.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@ExceptionDetails", report.ExceptionDetails.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@ApplicationLocation", report.ApplicationLocation.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@StepsToReproduce", report.StepsToReproduce.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@LastAttemptedVoiceHostID", report.LastAttemptedVoiceHostID.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@OverlayExitCode", report.OverlayExitCode.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@OverlayGameState", report.OverlayGameState.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@CallQuality", report.CallQuality.CoalesceToDBNull());

                    string runningProcessesJson = null;
                    if (report.RunningProcesses != null && report.RunningProcesses.Any())
                    {
                        try
                        {
                            runningProcessesJson = JsonConvert.SerializeObject(report.RunningProcesses);
                        }
                        catch (Exception ex)
                        {
                            Logger.Warn(ex, "Failed to convert running process list to JSON. ");
                        }
                    }
                    cmd.Parameters.AddWithValue("@RunningProcesses", runningProcessesJson.CoalesceToDBNull());
                    bugReportID = (int)cmd.ExecuteScalar();
                }

                // Try to save each attachment file
                if (report.Attachments == null)
                {
                    return bugReportID;
                }

                foreach (var attachment in report.Attachments)
                {
                    if (!IsValidAttachment(attachment))
                    {
                        continue;
                    }

                    using (var cmd = conn.CreateCommand())
                    {
                        using (var transaction = conn.BeginTransaction())
                        {
                            cmd.Transaction = transaction;

                            // Now, save the record to the model
                            cmd.CommandText =
                                "INSERT INTO [ClientBugReportAttachment] (ReportID, FileName, FileLength) OUTPUT INSERTED.ID VALUES(@ReportID, @FileName, @FileLength)";
                            cmd.Parameters.AddWithValue("@ReportID", bugReportID);
                            cmd.Parameters.AddWithValue("@FileName", attachment.FileName);
                            cmd.Parameters.AddWithValue("@FileLength", attachment.FileContents.Length);
                            var attachmentID = (int)cmd.ExecuteScalar();

                            try
                            {
                                // First save the attachment to disk
                                SaveBugReportAttachment(attachmentID, attachment.FileName, attachment.FileContents);
                                transaction.Commit();

                            }
                            catch (Exception ex)
                            {
                                transaction.Rollback();
                                Logger.Error(ex, "Failed to save bug report attachment. " + ex.GetExceptionDetails());

                            }
                        }
                    }



                }

                return bugReportID;
            }
        }

        private static bool TryGetExistingReportIDFromHash(Int64 hash, out int? foundReportID)
        {
            int existingReportID;

            // Check our cash of hash code mappings
            if (StackTraceToReportID.TryGetValue(hash, out existingReportID))
            {
                foundReportID = existingReportID;
                return true;
            }

            try
            {
                // Check the database
                using (
                    SqlConnection conn =
                        new SqlConnection(ConfigurationManager.ConnectionStrings["ClientBugReporting"].ConnectionString)
                    )
                {
                    conn.Open();

                    using (SqlCommand cmd = conn.CreateCommand())
                    {
                        cmd.CommandText =
                            "SELECT TOP 1 [ID] from [ClientBugReport] where [StackTraceHash] = @StackTraceHash";
                        cmd.Parameters.AddWithValue("@StackTraceHash", hash);

                        using (SqlDataReader reader = cmd.ExecuteReader())
                        {
                            if (reader.Read())
                            {
                                foundReportID = reader.GetInt32(0);
                                StackTraceToReportID.TryAdd(hash, foundReportID.Value);
                            }
                            else
                            {
                                foundReportID = null;
                            }
                            return true;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "BugReporting: Unable to get existing report from hash!");
                foundReportID = null;
                return false;
            }
        }

        private static int? GetOrCreateVersion(Version version, BugReportPlatform platform)
        {
            int versionID;

            var cacheKey = version + "-" + platform;

            // Check our cache of hash code mappings
            if (VersionStringToID.TryGetValue(cacheKey, out versionID))
            {
                return versionID;                
            }

            try
            {
                // Check the database
                using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["ClientBugReporting"].ConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText =
                            "SELECT TOP 1 [ID] from [ClientVersion] where [VersionString] = @VersionString AND [Platform] = @Platform";
                        cmd.Parameters.AddWithValue("@VersionString", version.ToString());
                        cmd.Parameters.AddWithValue("@Platform", (int)platform);

                        using (var reader = cmd.ExecuteReader())
                        {
                            if (reader.Read())
                            {
                                versionID = reader.GetInt32(0);
                                VersionStringToID.TryAdd(cacheKey, versionID);
                                return versionID;                                
                            }

                            var newVersionID = CreateClientVersion(version, platform);

                            if (newVersionID > 0)
                            {
                                VersionStringToID.TryAdd(cacheKey, newVersionID);
                                return newVersionID;
                            }

                            return null;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex,"Unable to get client version from version string!");
                return null;
            }
        }

        private static int CreateClientVersion(Version version, BugReportPlatform platform)
        {
            try
            {
                // Check the database
                using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["ClientBugReporting"].ConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "INSERT INTO [ClientVersion]([Type], [VersionString], [Platform], [DateReleased]) OUTPUT INSERTED.ID VALUES(@Type, @VersionString, @Platform, GETUTCDATE())";
                        cmd.Parameters.AddWithValue("@Type", 1);
                        cmd.Parameters.AddWithValue("@VersionString", version.ToString());
                        cmd.Parameters.AddWithValue("@Platform", (int)platform);

                        return (int) cmd.ExecuteScalar();
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to get client version from version string");
                return 0;
            }
        }        

    }
}