﻿using System;
using System.Collections.Concurrent;
using System.Configuration;
using System.Data.SqlClient;
using System.Net;
using System.ServiceModel;
using System.Threading;
using Curse.ClientService.Models;
using Curse.CloudServices.Client;
using Curse.Logging;
using Curse.ServiceAuthentication;
using Curse.ClientService.FriendsService;

namespace Curse.ClientService.Managers
{
    public class UserAbuseReportingManager
    {        
        private static readonly ConcurrentQueue<UserAbuseReport> ReportQueue = new ConcurrentQueue<UserAbuseReport>();
        private const int MaxQueueLength = 100;

        public static void Initialize()
        {
            // Fetch user details from Friends Service
#if DEBUG
            ServicePointManager.ServerCertificateValidationCallback += (a, b, c, d) => true;
#endif
            new Thread(ProcessQueueThread) { IsBackground = true }.Start();
        }

        public static UserAbuseReportRequestStatus ReportUserAbuse(int reportingUserID, string reportingUsername, ClientPlatform reportingUserPlatform,
            int reportedUserID, UserAbuseReportReason reason, string description)
        {
            if (ReportQueue.Count >= MaxQueueLength)
            {
                return UserAbuseReportRequestStatus.Error;
            }            
            
            var report = new UserAbuseReport
            {
                ReporterUserID = reportingUserID,
                ReporterUsername = reportingUsername,
                ReporterPlatform = reportingUserPlatform,
                ReportedUserID = reportedUserID,                
                Timestamp = DateTime.UtcNow,
                Reason = reason,
                Description = description,
                Status = UserAbuseReportStatus.New
            };
             
            // Queue report
            ReportQueue.Enqueue(report);

            return UserAbuseReportRequestStatus.Successful;
        }

        private static void AddAdditionalData(UserAbuseReport report)
        {
            try
            {
                            
                using (var friendsClient = new FriendsServiceClient(BindingHelper.GetBinaryBinding(), new EndpointAddress(ConfigurationManager.AppSettings["FriendsUrl"])))
                {
                    var userProfileResponse = friendsClient.GetUserInfo(report.ReportedUserID, ConfigurationManager.AppSettings["LogServiceApiKey"]);

                    if (userProfileResponse.Status != BasicServiceResponseStatus.Successful)
                    {
                        Logger.Warn("Failed to request user info from web service: " + userProfileResponse.Status);
                        throw new Exception("Service called failed with: " + userProfileResponse.Status);
                    }

                    report.ReportedUserCity = userProfileResponse.City;
                    report.ReportedUserState = userProfileResponse.State;
                    report.ReportedUserCountry = userProfileResponse.CountryCode;
                    report.ReportedUsername = userProfileResponse.Username;
                    report.ReportedUserDisplayName = string.IsNullOrEmpty(userProfileResponse.Name) ? userProfileResponse.Username : userProfileResponse.Name;
                    report.ReportedUserAvatarUrl = userProfileResponse.AvatarUrl;
                    report.ReportedUserFriendCount = userProfileResponse.FriendCount;
                    report.ReportedUserAboutMeText = userProfileResponse.AboutMe;
                }
            }
            catch (Exception ex)
            {                                
                report.ReportedUsername = "Unknown";
                report.ReportedUserFriendCount = 0;                
                Logger.Warn(ex, "Failed to gather reported user information from friends web service");
            }

            var userInfo = AuthenticationProvider.GetUserProfile(report.ReportedUserID);
            report.ReportedUserRegistrationDate = userInfo.Profile.RegisteredOn;           
        }

        private static void ProcessQueueThread()
        {
            while (true)
            {
                try
                {
                    Thread.Sleep(1000);

                    UserAbuseReport report;

                    while (ReportQueue.TryDequeue(out report))
                    {
                        try
                        {
                            AddAdditionalData(report);
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex, "Failed to add additional data to report.");
                            continue;                                                                                    
                        }

                        try
                        {
                            SaveReportToDatabase(report);
                        }
                        catch (Exception ex)
                        {

                            Logger.Warn(ex, "User abuse report was not saved to the database, due to invalid data or exception.", new { report });
                        }

                        Thread.Sleep(10); // Caps this to 100 reports per second
                    }
                }
                catch (ThreadAbortException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to process user abuse report queue");
                }
            }
        }

        private static void SaveReportToDatabase(UserAbuseReport report)
        {
            using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["ClientUserAbuseReporting"].ConnectionString))
            {
                conn.Open();

                // TODO: Check for similar reports to avoid spam?
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = @"INSERT INTO [ClientUserAbuseReport] (ReporterUserID,ReporterUsername,ReporterPlatform,
                                      ReportedUserID,ReportedUserRegistrationDate,ReportedUserFriendCount,ReportedUserAvatarUrl,ReportedUsername,ReportedUserDisplayName,
                                      ReportedUserCity,ReportedUserState,ReportedUserCountry,ReportedUserAboutMeText,
                                      Timestamp,Reason,Description,Status)
                                      OUTPUT INSERTED.ID
                                      VALUES(@ReporterUserID,@ReporterUsername,@ReporterPlatform,
                                      @ReportedUserID,@ReportedUserRegistrationDate,@ReportedUserFriendCount,@ReportedUserAvatarUrl,@ReportedUsername,@ReportedUserDisplayName,
                                      @ReportedUserCity,@ReportedUserState,@ReportedUserCountry,@ReportedUserAboutMeText,
                                      @Timestamp,@Reason,@Description,@Status)";

                    cmd.Parameters.AddWithValue("@ReporterUserID", report.ReporterUserID);
                    cmd.Parameters.AddWithValue("@ReporterUsername", report.ReporterUsername);                    
                    cmd.Parameters.AddWithValue("@ReporterPlatform", (byte)report.ReporterPlatform);
                    cmd.Parameters.AddWithValue("@ReportedUserID", report.ReportedUserID);
                    cmd.Parameters.AddWithValue("@ReportedUserRegistrationDate", report.ReportedUserRegistrationDate);
                    cmd.Parameters.AddWithValue("@ReportedUserFriendCount", report.ReportedUserFriendCount);
                    cmd.Parameters.AddWithValue("@ReportedUserAvatarUrl", (object)report.ReportedUserAvatarUrl ?? DBNull.Value);
                    cmd.Parameters.AddWithValue("@ReportedUsername", report.ReportedUsername);
                    cmd.Parameters.AddWithValue("@ReportedUserDisplayName", report.ReportedUserDisplayName);
                    cmd.Parameters.AddWithValue("@ReportedUserCity", (object)report.ReportedUserCity ?? DBNull.Value);
                    cmd.Parameters.AddWithValue("@ReportedUserState", (object)report.ReportedUserState ?? DBNull.Value);
                    cmd.Parameters.AddWithValue("@ReportedUserCountry", (object)report.ReportedUserCountry ?? DBNull.Value);
                    cmd.Parameters.AddWithValue("@ReportedUserAboutMeText", (object)report.ReportedUserAboutMeText ?? DBNull.Value);
                    cmd.Parameters.AddWithValue("@Timestamp", report.Timestamp);
                    cmd.Parameters.AddWithValue("@Reason", (byte)report.Reason);
                    cmd.Parameters.AddWithValue("@Description", report.Description);
                    cmd.Parameters.AddWithValue("@Status", (byte)report.Status);

                    cmd.ExecuteNonQuery();
                }
            }
        }
    }
}