﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data.SqlClient;
using System.Configuration;
using System.Threading;
using System.Data;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;

using Curse.Extensions;
using Curse.AddOns;
using Curse.ClientService.Extensions;
using System.Drawing;
using System.Runtime.Serialization.Json;
using ICSharpCode.SharpZipLib.BZip2;
using System.Runtime.Serialization;

namespace Curse.ClientService
{
    public class CCategoryCache
    {
        // If the number of categories retrieved from the database is lower than this number, the category cache is not updated.
        public const int MinimumCategoryCount = 64;
        
        private int _updateThreadInterval;
        private Thread _updateThread = null;
        private string _databaseConnectionString = null;
        private string _staticFileDestination = null;
        private bool _createFeedFiles = false;
        private string _versionNumber = null;

        private static readonly CCategoryCache _instance = new CCategoryCache();

        public static CCategoryCache Instance
        {
            get
            {
                return _instance;
            }
        }

        private CCategoryCache()
        {
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["RoamingDBRadon"].ConnectionString;
            _staticFileDestination = ConfigurationManager.AppSettings["FeedPath"];
            _createFeedFiles = (System.Environment.MachineName.ToLower() == ConfigurationManager.AppSettings["JobMachineName"].ToLower());
            _versionNumber = ConfigurationManager.AppSettings["FeedFileVersion"];

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }        

        public DateTime FileDate
        {
            get;
            set;
        }        

        public void Initialize() {}        

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info,
                               null,
                               "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info,
                               null,
                               "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }

        private void UpdateCache()
        {

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }
                List<CCategory> categories = new List<CCategory>();

                SqlCommand command = new SqlCommand("select distinct ProjectCategory.*, Project.GameId"
                + " from ProjectCategory with(nolock)"
                + " inner join ProjectCategoryMap with(nolock)"
                + " on ProjectCategoryMap.ProjectCategoryId = ProjectCategory.ID"
                + " inner join Project with(nolock)"
                + " on Project.ID = ProjectCategoryMap.ProjectId"
                + " union"
                + " select distinct ProjectCategory.*, Project.GameID"
                + " from ProjectCategory with(nolock)"
                + " inner join Project with(nolock)"
                + " on ProjectCategory.ID = Project.PrimaryCategoryID"
                + " order by ProjectCategory.ID, GameId", conn);

                List<int> distinctGameIDs = new List<int>();

                using (SqlDataReader reader = command.ExecuteReader())
                {
                    
                    while (reader.Read())
                    {
                        int categoryID = (int)reader["ID"];
                        int gameID = (int)reader["GameId"];

                        if (!distinctGameIDs.Contains(gameID))
                        {
                            distinctGameIDs.Add(gameID);
                        }

                        CCategory category = categories.FirstOrDefault(p => p.Id == categoryID);
                        if (category == null)
                        {                        
                            category = new CCategory();
                            category.SetFromDataReader(reader);
                            categories.Add(category);
                        }
                        category.GameIDs.Add(gameID);                        
                    }

                    //Ensure that every game in the game cache is in the distinct list so it gets the "Any Category" thing
                    foreach (var game in CGameCache.Instance.Games) {
                        if (!distinctGameIDs.Contains(game.ID))
                            distinctGameIDs.Add(game.ID);
                    }//check for this
                    
                    categories.Add(new CCategory(0, null, "Any Category", null, null, distinctGameIDs));                    
                }

                if (distinctGameIDs.Count < CGameCache.MinimumGameCount)
                {
                    Logger.Log("Game ID list is incomplete. Skipping cache update, for this iteration.", ELogLevel.Info);
                    return;
                }
                

                if (categories.Count <= MinimumCategoryCount)
                {
                    Logger.Log("Category list is incomplete. Skipping cache update, for this iteration.", ELogLevel.Info);
                    return;
                }

                try
                {
                    SaveToDisk(categories);
                }
                catch (Exception ex)
                {
                    Logger.Log("Unable to create category feed file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }
            }
        }

        private void UpdateCacheDate()
        {
            string currentfile = null;
            try
            {
                currentfile = Path.Combine(_staticFileDestination, string.Format("Categories{0}.zip", _versionNumber));
                FileDate = new FileInfo(currentfile).LastWriteTimeUtc;
            }
            catch (Exception ex)
            {
                Logger.Log("Unable to update category cache file data at '{0}'! Details: {1}", ELogLevel.Error, currentfile, ex.GetExceptionDetails());
            }
        }

        private void SaveToDisk(List<CCategory> categories)
        {
            if (_createFeedFiles)
            {
                SaveBinaryToDisk(categories);
                SaveJsonToDisk(categories);
                //SaveXmlToDisk(categories);
            }

            UpdateCacheDate();
            
        }

        private void SaveBinaryToDisk(List<CCategory> categories)
        {
            using (MemoryStream ms = new MemoryStream())
            {
                BinaryFormatter bf = new BinaryFormatter();
                bf.Serialize(ms, categories);
                ms.Position = 0;

                byte[] uncompressed = ms.ToArray();
                byte[] compressed = Utility.GetCompressedBytes(uncompressed);

                string basePath = _staticFileDestination;
                string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string backupfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string currentfile = Path.Combine(basePath, string.Format("Categories{0}.zip", _versionNumber));
                using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
                {
                    binWriter.Write(compressed);
                }

                if (File.Exists(currentfile) && FileComparer.CompareFiles(currentfile, tempfile))
                {
                    try
                    {
                        File.Delete(tempfile);
                    }
                    catch(Exception ex)
                    {
                        Logger.Log("CGameCache - Unable to delete temp file! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                    }
                    return;
                }

                try
                {
                    if (File.Exists(currentfile))
                    {
                        File.Replace(tempfile, currentfile, backupfile);
                    }
                    else
                    {
                        File.Copy(tempfile, currentfile, true);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Log("CCategoryCache - SaveToDisk Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }
                finally
                {
                    File.Delete(tempfile);
                    if (File.Exists(backupfile))
                    {
                        File.Delete(backupfile);
                    }
                }
            }
        }

        private void SaveXmlToDisk(List<CCategory> categories)
        {
            byte[] uncompressedBytes = null;

            using (MemoryStream uncompressedStream = new MemoryStream())
            {
                DataContractSerializer xs = new DataContractSerializer(categories.GetType());
                xs.WriteObject(uncompressedStream, categories);
                uncompressedStream.Position = 0;
                uncompressedBytes = uncompressedStream.ToArray();
            }

            byte[] compressedBytes = null;

            using (MemoryStream ms = new MemoryStream())
            {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(ms))
                {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = ms.ToArray();
            }

            using (MemoryStream ms = new MemoryStream())
            {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(ms))
                {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = ms.ToArray();
            }

            string basePath = _staticFileDestination;
            string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".bz2");
            string currentfile = Path.Combine(basePath, "Categories.xml.bz2");
            using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
            {
                binWriter.Write(compressedBytes);
            }
            try
            {
                File.Delete(currentfile);
                File.Move(tempfile, currentfile);
            }
            catch
            {
                File.Delete(tempfile);
            }
        }

        private void SaveJsonToDisk(List<CCategory> categories)
        {
            byte[] uncompressedBytes = null;

            using (MemoryStream ms = new MemoryStream())
            {
                DataContractJsonSerializer js = new DataContractJsonSerializer(categories.GetType());
                js.WriteObject(ms, categories);
                uncompressedBytes = ms.ToArray();
            }
            
            byte[] compressedBytes = null;

            using (MemoryStream ms = new MemoryStream())
            {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(ms))
                {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = ms.ToArray();
            }

            using (MemoryStream ms = new MemoryStream())
            {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(ms))
                {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = ms.ToArray();
            }

            string basePath = _staticFileDestination;
            string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".bz2");
            string currentfile = Path.Combine(basePath, "Categories.json.bz2");
            using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
            {
                binWriter.Write(compressedBytes);
            }
            try
            {
                File.Delete(currentfile);
                File.Move(tempfile, currentfile);
            }
            catch
            {
                File.Delete(tempfile);
            }
        }

        public Dictionary<int, List<CAddOnCategory>> GetCategoryCache(SqlConnection conn, DateTime changeDate)
        {
            Dictionary<int, List<CAddOnCategory>> categoryCache = new Dictionary<int, List<CAddOnCategory>>();
            using (SqlCommand categoryCommand = new SqlCommand("curseService_GetAllAddOnCategories", conn))
            {
                categoryCommand.CommandType = CommandType.StoredProcedure;
                categoryCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                categoryCommand.Parameters["@LastUpdated"].Value = changeDate;
                categoryCommand.CommandTimeout = 300;

                using (SqlDataReader categoryReader = categoryCommand.ExecuteReader())
                {
                    while (categoryReader.Read())
                    {
                        int projectID = categoryReader.GetInt32(0);
                        string username = categoryReader.GetString(1);

                        if (!categoryCache.ContainsKey(projectID))
                        {
                            categoryCache.Add(projectID, new List<CAddOnCategory>());
                        }
                        CAddOnCategory category = new CAddOnCategory();
                        category.SetFromDataReader(categoryReader);
                        categoryCache[projectID].Add(category);
                    }
                }
            }
            return categoryCache;
        }
    }
}
