﻿using System;
using System.Linq;
using System.Configuration;
using System.Threading;
using System.Data.SqlClient;
using System.Collections.Generic;
using Curse.AddOns;
using System.Data;
using Curse.Extensions;

namespace Curse.ClientService
{
    public class CCategorySectionCache
    {
        private List<CCategorySection> _categorySections;
        private object _categorySectionLock = new object();

        private int _updateThreadInterval;
        private Thread _updateThread = null;
        private string _databaseConnectionString = null;

        private static readonly CCategorySectionCache _instance = new CCategorySectionCache();

        public static CCategorySectionCache Instance
        {
            get
            {
                return _instance;
            }
        }

        private CCategorySectionCache()
        {
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["RoamingDBRadon"].ConnectionString;
            _categorySections = new List<CCategorySection>();

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }

        public void Initialize() { }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info,
                               null,
                               "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info,
                               null,
                               "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }

        private void UpdateCache()
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }
                
                List<CCategorySection> sections = new List<CCategorySection>();

                var command = new SqlCommand("spGetGameSections", conn);
                command.CommandType = CommandType.StoredProcedure;
                using (var reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var section = new CCategorySection();
                        section.ID = reader.GetInt32(reader.GetOrdinal("ProjectCategorySectionID"));
                        section.GameID = reader.GetInt32(reader.GetOrdinal("GameID"));
                        section.Name = reader.GetString(reader.GetOrdinal("SectionName"));
                        section.PackageType = (EPackageType)reader.GetByte(reader.GetOrdinal("PackageType"));
                        section.Path = reader.GetNullableValue<string>("Path");
                        section.InitialInclusionPattern = reader.GetString(reader.GetOrdinal("InitialInclusionPattern"));
                        section.ExtraIncludePattern = reader.GetNullableValue<string>("ExtraInclusionPattern");

                        sections.Add(section);
                    }
                }

                lock (_categorySectionLock)
                {
                    _categorySections = sections;
                }
            }
        }

        public CCategorySection GetByID(int sectionID)
        {
            return _categorySections.FirstOrDefault(p => p.ID == sectionID);
        }
        
        public List<CCategorySection> GetAllSectionsByGameID(int gameId)
        {
            return _categorySections.Where(p => p.GameID == gameId).ToList();
        }
    }
}
