﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.ClientService.Models;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using System.Web.Caching;
using System.IO;

using Curse.Extensions;

namespace Curse.ClientService
{
    public class CSyncCache
    {
        private static readonly CSyncCache _instance = new CSyncCache();

        public List<CSyncedGameInstance> _syncCache = new List<CSyncedGameInstance>();
        public readonly string _databaseConnectionString;
        public readonly string _userBackupsPath;        
        private readonly Cache _distributedCache = null;
        private readonly DataTable _syncedAddonTableSchema = null;
        private readonly TimeSpan _defaultCacheDuration = TimeSpan.FromMinutes(5);

        private CSyncCache()
        {
            DataTable table = new DataTable();
            table.Columns.Add(new DataColumn("InstanceID", typeof(int)));
            table.Columns.Add(new DataColumn("AddonID", typeof(int)));
            table.Columns.Add(new DataColumn("FileID", typeof(int)));
            _syncedAddonTableSchema = table;

            _databaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
            _userBackupsPath = ConfigurationManager.AppSettings["UserBackupsPath"];            
            _distributedCache = HttpRuntime.Cache;
            _defaultCacheDuration = TimeSpan.Parse(ConfigurationManager.AppSettings["SyncCacheDefaultDuration"]);
        }

        public static CSyncCache Instance
        {
            get
            {
                return _instance;
            }
        }

        public void Initialize() {}

        private List<CSyncedGameInstance> GetProfileFromCache(int pUserID)
        {
            string cacheKey = string.Format("SyncProfile-{0}", pUserID);
            
            List<CSyncedGameInstance> list = _distributedCache.Get(cacheKey) as List<CSyncedGameInstance>;

            if (list == null)
            {
                list = GetProfileFromDatabase(pUserID);
                CacheProfile(list, pUserID);
            }

            return list;
        }

        public void SaveUserBackup(int userID, int instanceID, long pFingerprint, int pScreenWidth, int pScreenHeight, Stream stream)
        {
            List<CSyncedGameInstance> syncedInstances = GetProfileFromCache(userID);
            CSyncedGameInstance instance = syncedInstances.FirstOrDefault(p => p.InstanceID == instanceID);
            if (instance == null)
            {
                return;
            }
                        
            UpdateBackupSettings(instanceID, pFingerprint, pScreenWidth, pScreenHeight);            

            instance.LastBackupFingerprint = pFingerprint;
            instance.LastBackupScreenWidth = pScreenWidth;
            instance.LastBackupScreenHeight = pScreenHeight;
            instance.LastBackupDate = DateTime.UtcNow;

            DirectoryInfo di = new DirectoryInfo(Path.Combine(_userBackupsPath, userID.ToString()));
            if (!di.Exists)
            {
                di.Create();
            }
            string filename = Path.Combine(di.FullName, instanceID.ToString()) + ".zip";

            FileStream targetStream = null;
            using (targetStream = new FileStream(filename, FileMode.Create, FileAccess.Write, FileShare.None))
            {
                //read from the input stream in 4K chunks
                //and save to output stream
                const int bufferLen = 4096;
                byte[] buffer = new byte[bufferLen];
                int count = 0;
                while ((count = stream.Read(buffer, 0, bufferLen)) > 0)
                {
                    targetStream.Write(buffer, 0, count);
                }
                targetStream.Close();
            }

            return;

        }

        public void CacheProfile(List<CSyncedGameInstance> profile, int userID)
        {
            string cacheKey = string.Format("SyncProfile-{0}", userID);
            
            _distributedCache.Remove(cacheKey);

            if(profile != null)
            {
                _distributedCache.Add(cacheKey, profile, null, Cache.NoAbsoluteExpiration, _defaultCacheDuration, CacheItemPriority.High, null);                
            }
        }

        private List<CSyncedGameInstance> GetProfileFromDatabase(int userID)
        {
            Dictionary<int, CSyncedGameInstance> gameInstances = new Dictionary<int, CSyncedGameInstance>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                SqlCommand command = conn.CreateCommand();
                command.CommandType = CommandType.StoredProcedure;
                command.CommandText = "spGetSyncProfile";
                SqlParameter parameter = command.Parameters.Add("@UserID", SqlDbType.Int);
                parameter.Value = userID;                               

                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        int instanceID = (int)reader["InstanceID"];
                        int computerID = (int)reader["ComputerID"];
                        int addonID = (int)reader["AddonID"];
                        string instanceGuid = (string)reader["InstanceGUID"];


                        CSyncedGameInstance instance = null;

                        if (!gameInstances.ContainsKey(instanceID))
                        {
                            instance = new CSyncedGameInstance();
                            instance.SetFromDataReader(reader);
                            gameInstances.Add(instanceID, instance);
                        }
                        else
                        {
                            instance = gameInstances[instanceID];
                        }

                        // Computer
                        if (!instance.Computers.Exists(p => p.InstanceGuid == instanceGuid))
                        {
                            CSyncedComputer computer = new CSyncedComputer();
                            computer.SetFromDataReader(reader);
                            instance.Computers.Add(computer);
                        }

                        // Addon
                        if (addonID > 0 && !instance.Addons.Exists(p => p.AddonID == addonID))
                        {
                            CSyncedAddon addon = new CSyncedAddon();
                            addon.SetFromDataReader(reader);
                            instance.Addons.Add(addon);
                        }

                    }
                }             
            }
            return gameInstances.Values.ToList();            
        }        

        public CServiceResponse LeaveSyncGroup(int userID, int instanceID, int computerID, string instanceGUID)
        {
            List<CSyncedGameInstance> syncedInstances = GetProfileFromCache(userID);
            CSyncedGameInstance instance = syncedInstances.FirstOrDefault(p => p.InstanceID == instanceID);
            
            if (instance == null)
            {
                return new CServiceResponse(EServiceResponseStatus.LeaveSyncGroup_GroupNotFound);
            }

            if (!instance.Computers.Exists(p => p.ComputerID == computerID))
            {
                return new CServiceResponse(EServiceResponseStatus.LeaveSyncGroup_ComputerNotFound);                
            }
            
            RemoveComputerFromGroup(computerID, instanceID, instanceGUID);            
            CacheProfile(null, userID);

            return new CServiceResponse(EServiceResponseStatus.Successful);             
        }

        public CServiceResponse<CSyncedGameInstance> CreateSyncGroup(int userID, int gameID, string instanceName, string pComputerName, string pInstanceGUID, string pInstanceLabel)
        {
            List<CSyncedGameInstance> syncedInstances = GetProfileFromCache(userID);
            CSyncedGameInstance instance = syncedInstances.FirstOrDefault(p => p.InstanceName == instanceName);

            if (instance != null)
            {
                return new CServiceResponse<CSyncedGameInstance>(EServiceResponseStatus.CreateSyncGroup_GroupAlreadyExists); 
            }

            int instanceID = CreateNewGroup(userID, pComputerName, gameID, instanceName, pInstanceGUID, pInstanceLabel);            
            CacheProfile(null, userID);
            List<CSyncedGameInstance> instances = GetProfileFromCache(userID);
            instance = instances.FirstOrDefault(p => p.InstanceID == instanceID);

            return new CServiceResponse<CSyncedGameInstance>(EServiceResponseStatus.Successful, instance); 
        }

        public CServiceResponse<List<CSyncedGameInstance>> GetSyncProfile(int userID)
        {
            return new CServiceResponse<List<CSyncedGameInstance>>(EServiceResponseStatus.Successful, GetProfileFromCache(userID));
        }

        public CServiceResponse JoinSyncGroup(int pUserID, int pInstanceID, string pComputerName, string pInstanceGUID, string pInstanceLabel)
        {
            List<CSyncedGameInstance> syncedInstances = GetProfileFromCache(pUserID);
            
            CSyncedGameInstance instance = syncedInstances.FirstOrDefault(p => p.InstanceID == pInstanceID);

            if (instance == null)
            {
                return new CServiceResponse(EServiceResponseStatus.JoinSyncGroup_GroupNotFound);
            }

            if (instance.Computers.Any(p => p.InstanceGuid == pInstanceGUID))
            {
                return new CServiceResponse(EServiceResponseStatus.JoinSyncGroup_AlreadyInGroup);                
            }
            
            
            AddComputerToGroup(pUserID, pComputerName, pInstanceID, pInstanceGUID, pInstanceLabel);            
            CacheProfile(null, pUserID);

            return new CServiceResponse(EServiceResponseStatus.Successful);
        }

        public CServiceResponse SaveSyncTransactions(int pUserID, int pInstanceID, CSyncTransaction[] transactions)
        {
            List<CSyncedGameInstance> syncedInstances = GetProfileFromCache(pUserID);
            CSyncedGameInstance instance = syncedInstances.FirstOrDefault(p => p.InstanceID == pInstanceID);

            if (instance == null)
            {
                return new CServiceResponse(EServiceResponseStatus.SaveSyncTransactions_GroupNotFound);
            }

            List<CSyncedAddon> syncedAddons = instance.Addons;
            CSyncedAddon syncedAddon = null;
            foreach (CSyncTransaction transaction in transactions)
            {
                syncedAddon = syncedAddons.FirstOrDefault(p => p.AddonID == transaction.AddonID);

                switch (transaction.Type)
                {
                    case ESyncTransactionType.Delete:
                        syncedAddons.RemoveAll(p => p.AddonID == transaction.AddonID);
                        break;
                    case ESyncTransactionType.Install:
                        if (syncedAddon == null)
                        {
                            syncedAddon = new CSyncedAddon()
                            {
                                AddonID = transaction.AddonID,
                                FileID = transaction.FileID
                            };
                            syncedAddons.Add(syncedAddon);
                        }
                        break;
                    case ESyncTransactionType.Update:                        
                        if (syncedAddon != null)
                        {
                            syncedAddon.FileID = transaction.FileID;
                        }
                        break;
                }
            }
           
            SaveSyncSnapshot(pInstanceID, syncedAddons.ToArray());
            CacheProfile(null, pUserID);

            return new CServiceResponse(EServiceResponseStatus.Successful);
        }

        public CServiceResponse SaveSyncSnapshot(int userID, int instanceID, CSyncedAddon[] pSyncedAddons)
        {
            List<CSyncedGameInstance> syncedInstances = GetProfileFromCache(userID);
            
            if (!syncedInstances.Exists(p => p.InstanceID  == instanceID))
            {
                return new CServiceResponse(EServiceResponseStatus.SaveSyncSnapshot_GroupNotFound);
            }

            SaveSyncSnapshot(instanceID, pSyncedAddons);                            
            CacheProfile(null, userID);

            return new CServiceResponse(EServiceResponseStatus.Successful);
        }

        private void UpdateBackupSettings(int pInstanceID, long pFingerprint, int pScreenWidth, int pScreenHeight)
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "spUpdateBackupSettings";
                command.CommandType = CommandType.StoredProcedure;

                SqlParameter parameter = command.Parameters.Add("@InstanceID", SqlDbType.Int);
                parameter.Value = pInstanceID;

                parameter = command.Parameters.Add("@Fingerprint", SqlDbType.BigInt);
                parameter.Value = pFingerprint;

                parameter = command.Parameters.Add("@ScreenWidth", SqlDbType.Int);
                parameter.Value = pScreenWidth;

                parameter = command.Parameters.Add("@ScreenHeight", SqlDbType.Int);
                parameter.Value = pScreenHeight;

                command.ExecuteNonQuery();
            }
        }

        private int CreateNewGroup(int pUserID, string pComputerName, int pGameID, string pInstanceName, string pInstanceGUID, string pInstanceLabel)
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "spCreateSyncGroup";
                command.CommandType = CommandType.StoredProcedure;                

                // Input Params
                SqlParameter parameter = command.Parameters.Add("@ComputerName", SqlDbType.NVarChar);
                parameter.Value = pComputerName;

                parameter = command.Parameters.Add("@InstanceName", SqlDbType.NVarChar);
                parameter.Value = pInstanceName;

                parameter = command.Parameters.Add("@InstanceGUID", SqlDbType.NVarChar);
                parameter.Value = pInstanceGUID;

                parameter = command.Parameters.Add("@InstanceLabel", SqlDbType.NVarChar);
                parameter.Value = pInstanceLabel;

                parameter = command.Parameters.Add("@UserID", SqlDbType.Int);
                parameter.Value = pUserID;

                parameter = command.Parameters.Add("@GameID", SqlDbType.Int);
                parameter.Value = pGameID;

                // Output Params
                parameter = command.Parameters.Add("@ComputerID", SqlDbType.Int);
                parameter.Direction = ParameterDirection.Output;

                parameter = command.Parameters.Add("@InstanceID", SqlDbType.Int);
                parameter.Direction = ParameterDirection.Output;

                command.ExecuteNonQuery();

                return (int)command.Parameters["@InstanceID"].Value;
            }
        }

        private void AddComputerToGroup(int pUserID, string pComputerName, int pInstanceID, string pInstanceGUID, string pInstanceLabel)
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "spAddComputerToGroup";
                command.CommandType = CommandType.StoredProcedure;

                 //Create an output parameter for the new identity value.
                SqlParameter parameter = command.Parameters.Add("@ComputerID", SqlDbType.Int);
                parameter.Direction = ParameterDirection.Output;

                parameter = command.Parameters.Add("@ComputerName", SqlDbType.NVarChar);
                parameter.Value = pComputerName;

                parameter = command.Parameters.Add("@InstanceGUID", SqlDbType.NVarChar);
                parameter.Value = pInstanceGUID;

                parameter = command.Parameters.Add("@InstanceLabel", SqlDbType.NVarChar);
                parameter.Value = pInstanceLabel;                                                

                parameter = command.Parameters.Add("@UserID", SqlDbType.Int);
                parameter.Value = pUserID;

                parameter = command.Parameters.Add("@InstanceID", SqlDbType.Int);
                parameter.Value = pInstanceID;

        

                command.ExecuteNonQuery();
            }            
        }

        private void RemoveComputerFromGroup(int pComputerID, int pInstanceID, string pInstanceGUID)
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "spRemoveComputerFromGroup";
                command.CommandType = CommandType.StoredProcedure;

                //Create an output parameter for the new identity value.
                SqlParameter parameter = command.Parameters.Add("@ComputerID", SqlDbType.Int);
                parameter.Value = pComputerID;

                parameter = command.Parameters.Add("@InstanceID", SqlDbType.Int);
                parameter.Value = pInstanceID;

                parameter = command.Parameters.Add("@InstanceGUID", SqlDbType.NVarChar);
                parameter.Value = pInstanceGUID;

                command.ExecuteNonQuery();                
            }
        }               
        
        private void SaveSyncSnapshot(int pInstanceID, CSyncedAddon[] pSyncedAddons)
        {
            
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                DataTable table = _syncedAddonTableSchema.Clone();
                foreach (CSyncedAddon syncedAddon in pSyncedAddons)
                {
                    DataRow row = table.NewRow();
                    row["InstanceID"] = pInstanceID;
                    row["AddonID"] = syncedAddon.AddonID;
                    row["FileID"] = syncedAddon.FileID;
                    table.Rows.Add(row);
                }

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "delete from SyncedAddon where InstanceID = " + pInstanceID;
                command.ExecuteNonQuery();

                using (SqlBulkCopy bulk = new SqlBulkCopy(conn))
                {
                    bulk.DestinationTableName = "SyncedAddon";
                    bulk.WriteToServer(table);
                }
               
            }
        }

    }
}
