﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.CommunityTracker.Models;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using Curse.CommunityTracker.Extensions;

namespace Curse.CommunityTracker.Caching {
    public class CAuthorCache {
        private Dictionary<int, CAuthor> _authorCache = new Dictionary<int, CAuthor>();
        private Dictionary<int, List<CAuthor>> _authorsByDataSource = new Dictionary<int, List<CAuthor>>();

        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        public bool IsCacheUpdating { get { return _isCacheUpdating; } }
        public bool IsCacheBuilt { get { return _isCacheBuilt; } set { _isCacheBuilt = value; } }

        private Thread _updateThread = null;
        private int _updateThreadInterval;
        private DateTime _lastQueryTime = new DateTime(1979, 5, 17);
        private string _databaseConnectionString = string.Empty;

        private static readonly CAuthorCache _instance = new CAuthorCache();
        public static CAuthorCache Instance {
            get { return _instance; }
        }
        public DateTime LastQueryTime {
            get { return _lastQueryTime; }
        }

        public CAuthorCache() {
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }

        public static void Initialize() { }

        private void CacheThread() 
        {
            Boolean aborted = false;
            while (!aborted) {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                _isCacheUpdating = true;
                try {
                    UpdateCache();
                }
                catch (ThreadAbortException) {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex) {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                _isCacheUpdating = false;
            }
        }

        public void UpdateCache() 
        {
            Dictionary<int, CAuthor> authorCache = new Dictionary<int, CAuthor>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString)) {
                try {
                    conn.Open();
                }
                catch (Exception) {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                DateTime lastQueryTime = DateTime.UtcNow;
                SqlCommand command = new SqlCommand("select * from CommunityAuthor", conn);
                using (SqlDataReader reader = command.ExecuteReader()) {
                    while (reader.Read()) {
                        var author = new CAuthor();
                        author.SetFromDataReader(reader);

                        authorCache[author.ID] = author;
                    }
                }

                lock (_authorCache) {
                    _authorCache = authorCache;
                }

                // Build a source specific cache
                var authorsByDataSource = new Dictionary<int, List<CAuthor>>();
                foreach (CAuthor author in authorCache.Values) {
                    if (!authorsByDataSource.ContainsKey(author.DataSourceID)) {
                        authorsByDataSource[author.DataSourceID] = new List<CAuthor>();
                    }

                    authorsByDataSource[author.DataSourceID].Add(author);
                }

                lock (_authorsByDataSource) {
                    _authorsByDataSource = authorsByDataSource;
                }

                _lastQueryTime = lastQueryTime;
            }

            if (!IsCacheBuilt) {
                Logger.Log("Author Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;
            GC.Collect();
        }

        public void InvalidateCache() {
            _isCacheUpdating = true;
            try {
                UpdateCache();
            }
            catch (Exception exc) {
                Logger.Log("Failed to update Author Cache Details: ", ELogLevel.Error, exc.GetExceptionDetails());
            }
            _isCacheUpdating = false;
        }


        public CAuthor GetAuthorByID(int id) {
            if (!_isCacheBuilt) {
                return null;
            }

            if (_authorCache.ContainsKey(id)) {
                return _authorCache[id];
            }

            return null;
        }

        public CAuthor GetForumByID(int id, DateTime lastUpdated) {
            if (!_isCacheBuilt) {
                return null;
            }

            if (LastQueryTime < lastUpdated) {
                InvalidateCache();
            }

            if (_authorCache.ContainsKey(id)) {
                return _authorCache[id];
            }

            return null;
        }

        public IEnumerable<CAuthor> GetAuthorsByDataSourceID(int id, DateTime since)
        {
            if (!_isCacheBuilt)
            {
                return null;
            }


            List<CAuthor> authors = new List<CAuthor>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select ID from CommunityAuthor where exists (select top 1 1 from CommunityForumPost where CommunityAuthorID = CommunityAuthor.ID) and CommunityDataSourceID = @DataSourceID and DateModified >= @Since";
                cmd.Parameters.AddWithValue("@DataSourceID", id);
                cmd.Parameters.AddWithValue("@Since", since);

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        int authorID = reader.GetInt32(0);
                        CAuthor author = GetAuthorByID(authorID);
                        if (author != null)
                        {
                            authors.Add(author);
                        }
                    }
                }
            }                                    
            
            return authors;
        }
    }
}