﻿using System;
using System.Linq;
using Curse.CommunityTracker.Models;
using System.Collections.Generic;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using Curse.CommunityTracker.Extensions;

namespace Curse.CommunityTracker.Caching {
    public class CForumCache {
        private Dictionary<int, CForum> _forumCache = new Dictionary<int, CForum>();
        private Dictionary<int, List<CForum>> _dataSourceForums = new Dictionary<int, List<CForum>>();

        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        public bool IsCacheUpdating { get { return _isCacheUpdating; } }
        public bool IsCacheBuilt { get { return _isCacheBuilt; } set { _isCacheBuilt = value; } }

        private Thread _updateThread = null;
        private int _updateThreadInterval;
        private DateTime _lastQueryTime = new DateTime(1979, 5, 17);
        private string _databaseConnectionString = string.Empty;

        private static readonly CForumCache _instance = new CForumCache();
        public static CForumCache Instance {
            get { return _instance; }
        }
        public DateTime LastQueryTime {
            get { return _lastQueryTime; }
        }

        public CForumCache() {
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }

        public static void Initialize() { }

        private void CacheThread() {
            Boolean aborted = false;
            while (!aborted) {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                _isCacheUpdating = true;
                try {
                    UpdateCache();
                }
                catch (ThreadAbortException) {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex) {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                _isCacheUpdating = false;
            }
        }

        public void UpdateCache() {
            Dictionary<int, CForum> forumCache = new Dictionary<int, CForum>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString)) {
                try {
                    conn.Open();
                }
                catch (Exception) {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                DateTime lastQueryTime = DateTime.UtcNow;

                string query = "select"
                    + " CommunityForum.*,"
                    + " COALESCE(b.FullCultureCode, c.FullCultureCode, 'en-US') as LanguageCode"
                    + " from CommunityForum"
                    + " inner join CommunityDataSource on CommunityDataSource.ID = CommunityForum.CommunityDataSourceID"
                    + " left join Language as b on b.ID = CommunityForum.LanguageID"
                    + " left join Language as c on c.ID = CommunityDataSource.LanguageID";

                SqlCommand command = new SqlCommand(query, conn);
                using (SqlDataReader reader = command.ExecuteReader()) {
                    while (reader.Read()) {
                        var forum = new CForum();
                        forum.SetFromDataReader(reader);

                        forumCache[forum.ID] = forum;
                    }
                }

                lock (_forumCache) {
                    _forumCache = forumCache;
                }

                // Build a source specific cache
                var dataSourceForums = new Dictionary<int, List<CForum>>();
                foreach (CForum forum in forumCache.Values) {
                    if (!dataSourceForums.ContainsKey(forum.DataSourceID)) {
                        dataSourceForums[forum.DataSourceID] = new List<CForum>();
                    }

                    dataSourceForums[forum.DataSourceID].Add(forum);
                }

                lock (_dataSourceForums) {
                    _dataSourceForums = dataSourceForums;
                }

                _lastQueryTime = lastQueryTime;
            }

            if (!IsCacheBuilt) {
                Logger.Log("CommunityForum Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;
            GC.Collect();
        }

        public void InvalidateCache() {
            _isCacheUpdating = true;
            try {
                UpdateCache();
            }
            catch (Exception exc) {
                Logger.Log("Failed to update Forum Cache Details: ", ELogLevel.Error, exc.GetExceptionDetails());
            }
            _isCacheUpdating = false;
        }

        public CForum GetForumByID(int id) {
            if (!_isCacheBuilt) {
                return null;
            }

            if (_forumCache.ContainsKey(id)) {
                return _forumCache[id];
            }

            return null;
        }

        public CForum GetForumByID(int id, DateTime lastUpdated) {
            if (!_isCacheBuilt) {
                return null;
            }

            if (LastQueryTime < lastUpdated) {
                InvalidateCache();
            }

            if (_forumCache.ContainsKey(id)) {
                return _forumCache[id];
            }

            return null;
        }
        
        public IEnumerable<CForum> GetForumsBySourceId(int id) {
            if (!IsCacheBuilt) {
                return null;
            }

            if (_dataSourceForums.ContainsKey(id)) {
                return _dataSourceForums[id];
            }
            else {
                return null;
            }
        }
    }
}