﻿using System;
using System.Linq;
using Curse.CommunityTracker.Models;
using System.Collections.Generic;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using Curse.CommunityTracker.Extensions;

namespace Curse.CommunityTracker.Caching {
    public class CForumThreadCache {
        private Dictionary<int, CForumThread> _forumThreadCache = new Dictionary<int, CForumThread>();
        private Dictionary<int, List<CForumThread>> _threadsByForum = new Dictionary<int, List<CForumThread>>();

        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        public bool IsCacheUpdating { get { return _isCacheUpdating; } }
        public bool IsCacheBuilt { get { return _isCacheBuilt; } set { _isCacheBuilt = value; } }

        private Thread _updateThread = null;
        private int _updateThreadInterval;
        private DateTime _lastQueryTime = new DateTime(1979, 5, 17);
        private string _databaseConnectionString = string.Empty;

        private static readonly CForumThreadCache _instance = new CForumThreadCache();
        public static CForumThreadCache Instance {
            get { return _instance; }
        }
        public DateTime LastQueryTime {
            get { return _lastQueryTime; }
        }

        public CForumThreadCache() {
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }

        public static void Initialize() { }

        private void CacheThread() {
            Boolean aborted = false;
            while (!aborted) {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                _isCacheUpdating = true;
                try {
                    UpdateCache();
                }
                catch (ThreadAbortException) {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex) {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                _isCacheUpdating = false;
            }
        }

        public void InvalidateCache() {
            _isCacheUpdating = true;
            try {
                UpdateCache();
            }
            catch (Exception exc) {
                Logger.Log("Failed to update Thread Cache Details: ", ELogLevel.Error, exc.GetExceptionDetails());
            }
            _isCacheUpdating = false;
        }

        public void UpdateCache() {
            Dictionary<int, CForumThread> forumThreadCache = new Dictionary<int, CForumThread>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString)) {
                try {
                    conn.Open();
                }
                catch (Exception) {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                DateTime lastQueryTime = DateTime.UtcNow;

                string query = "select distinct CommunityForumThread.*, CommunityAuthor.Name as AuthorName, a.Url as LatestTrackedPostUrl, b.Url as StartingPostUrl from CommunityForumThread"
                                + " inner join CommunityAuthor on CommunityAuthor.ID = CommunityForumThread.CommunityAuthorID"
                                + " left join CommunityForumPost as a on a.ID = CommunityForumThread.LatestTrackedPostID"
                                + " left join CommunityForumPost as b on b.ID = CommunityForumThread.StartingPostID"
                                + " order by CommunityForumThread.ID desc";

                SqlCommand command = new SqlCommand(query, conn);
                using (SqlDataReader reader = command.ExecuteReader()) {
                    while (reader.Read()) {
                        var thread = new CForumThread();
                        thread.SetFromDataReader(reader);

                        forumThreadCache[thread.ID] = thread;
                    }
                }

                lock (_forumThreadCache) {
                    _forumThreadCache = forumThreadCache;
                }

                // Build a source specific cache
                var threadsByForum = new Dictionary<int, List<CForumThread>>();
                foreach (CForumThread thread in forumThreadCache.Values) {
                    if (!threadsByForum.ContainsKey(thread.ForumID)) {
                        threadsByForum[thread.ForumID] = new List<CForumThread>();
                    }

                    threadsByForum[thread.ForumID].Add(thread);
                }

                lock (_threadsByForum) {
                    _threadsByForum = threadsByForum;
                }

                _lastQueryTime = lastQueryTime;
            }

            if (!IsCacheBuilt) {
                Logger.Log("CommunityForumThread Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;
            GC.Collect();
        }

        public CForumThread GetThreadById(int threadId) {
            if (!_isCacheBuilt) {
                return null;
            }

            if (_forumThreadCache.ContainsKey(threadId)) {
                return _forumThreadCache[threadId];
            }
            else {
                return null;
            }
        }

        public CForumThread GetThreadById(int threadId, DateTime lastUpdated) {
            if (!_isCacheBuilt) {
                return null;
            }

            //Invalidate the cache if it has been longer that the last query time
            if (LastQueryTime < lastUpdated) {
                InvalidateCache();
            }

            if (_forumThreadCache.ContainsKey(threadId)) {
                return _forumThreadCache[threadId];
            }
            
            return null;
        }

        public IEnumerable<CForumThread> GetThreadsByForumId(int id, DateTime since) {
            if (!IsCacheBuilt) {
                return null;
            }

            if (_threadsByForum.ContainsKey(id)) {
                return _threadsByForum[id].Where(t => t.DateCreated >= since || t.DateModified >= since);
            }
            else {
                return null;
            }
        }

        public IEnumerable<UpdatedThread> GetRecentThreadActivity(int[] dataSourceIds, DateTime since) {
            Dictionary<int, UpdatedThread> updatedThreads = new Dictionary<int, UpdatedThread>();

            var query = "select CommunityForumThread.ID, CommunityForumPost.ID from CommunityForum inner join CommunityForumThread " +
               "on CommunityForumThread.CommunityForumID = CommunityForum.ID " +
               "inner join CommunityForumPost " +
               "on CommunityForumPost.CommunityForumThreadID = CommunityForumThread.ID " +
               "where CommunityForumThread.CommunityDataSourceID in({0}) and CommunityForumPost.DateModified >= @since";
            query = string.Format(query, string.Join(",", dataSourceIds));

            using (var conn = new SqlConnection(_databaseConnectionString)) {
                var cmd = new SqlCommand(query, conn);
                cmd.Parameters.Add("since", SqlDbType.DateTime).Value = since;

                conn.Open();
                using (SqlDataReader reader = cmd.ExecuteReader()) {
                    while (reader.Read()) {
                        var thread = CForumThreadCache.Instance.GetThreadById((int)reader[0], since);
                        var post = CForumThreadPostCache.Instance.GetPostById((int)reader[1], since);

                        if (updatedThreads.ContainsKey(thread.ID)) {
                            updatedThreads[thread.ID].Posts.Add(post);
                        }
                        else {
                            var update = new UpdatedThread {
                                ID = thread.ID,
                                Thread = thread,
                            };

                            update.Posts = new List<CForumThreadPost>();
                            update.Posts.Add(post);

                            updatedThreads.Add(update.ID, update);
                        }
                    }
                }
            }

            return updatedThreads.Values;
        }
    }
}