﻿using System;
using System.Linq;
using Curse.CommunityTracker.Models;
using System.Collections.Generic;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using Curse.CommunityTracker.Extensions;

namespace Curse.CommunityTracker.Caching {
    public class ForumDataSourceCache {
        private Dictionary<int, ForumDataSource> _forumDataSourceCache = new Dictionary<int, ForumDataSource>();
        private Dictionary<int, List<ForumDataSource>> _dataSourcesByGame = new Dictionary<int, List<ForumDataSource>>();

        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        public bool IsCacheUpdating { get { return _isCacheUpdating; } }
        public bool IsCacheBuilt { get { return _isCacheBuilt; } set { _isCacheBuilt = value; } }

        private Thread _updateThread = null;
        private int _updateThreadInterval;
        private DateTime _lastQueryTime = new DateTime(1979, 5, 17);              
        private string _databaseConnectionString = string.Empty;
        
        private static readonly ForumDataSourceCache _instance = new ForumDataSourceCache();
        public static ForumDataSourceCache Instance { 
            get { return _instance; } 
        }
        public DateTime LastQueryTime { get { return _lastQueryTime; } }

        public ForumDataSourceCache() {
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;

            UpdateCache();
            
            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }

        public static void Initialize() { }

        private void CacheThread() {
            Boolean aborted = false;
            while (!aborted) {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                _isCacheUpdating = true;
                try {
                    UpdateCache();
                }
                catch (ThreadAbortException) {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex) {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                _isCacheUpdating = false;
            }
        }

        public void UpdateCache() {
            Dictionary<int, ForumDataSource> forumDataSourceCache = new Dictionary<int, ForumDataSource>();
            
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString)) {
                try {
                    conn.Open();
                }
                catch (Exception) {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                DateTime changeTime = _lastQueryTime.AddMinutes(-5);
                SqlCommand command = new SqlCommand("SELECT CommunityDataSource.*, Language.FullCultureCode as 'Language' FROM CommunityDataSource with(nolock) inner join Language with(nolock) on CommunityDataSource.LanguageID = Language.ID", conn); // and CommunityDataSource.DateModified > @LastUpdated", conn);
                //command.Parameters.Add("LastUpdated", SqlDbType.DateTime).Value = changeTime;
                DateTime lastQueryTime = DateTime.UtcNow;

                using (SqlDataReader reader = command.ExecuteReader()) {
                    while (reader.Read()) {
                        var forumDataSource = new ForumDataSource();
                        forumDataSource.SetFromDataReader(reader);

                        forumDataSourceCache[forumDataSource.ID] = forumDataSource;
                    }
                }

                lock (_forumDataSourceCache) {
                    _forumDataSourceCache = forumDataSourceCache;
                }

                // Build a source specific cache
                var dataSourcesByGame = new Dictionary<int, List<ForumDataSource>>();
                foreach (ForumDataSource source in forumDataSourceCache.Values) {
                    if (!dataSourcesByGame.ContainsKey(source.GameID)) {
                        dataSourcesByGame[source.GameID] = new List<ForumDataSource>();
                    }

                    dataSourcesByGame[source.GameID].Add(source);
                }

                lock (_dataSourcesByGame) {
                    _dataSourcesByGame = dataSourcesByGame;
                }

                _lastQueryTime = lastQueryTime;
            }

            if (!IsCacheBuilt) {
                Logger.Log("ForumDataSource Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;
            GC.Collect();
        }

        public void InvalidateCache() {
            _isCacheUpdating = true;
            try {
                UpdateCache();
            }
            catch (Exception exc) {
                Logger.Log("Failed to update Data Source Cache Details: ", ELogLevel.Error, exc.GetExceptionDetails());
            }
            _isCacheUpdating = false;
        }

        public ForumDataSource GetDataSourceByID(int id) {
            if(!IsCacheBuilt){
                return null;
            }

            if (_forumDataSourceCache.ContainsKey(id)) {
                return _forumDataSourceCache[id];
            }
            else {
                return null;
            }
        }

        public ForumDataSource GetDataSourceByID(int id, DateTime lastUpdated) {
            if (!IsCacheBuilt) {
                return null;
            }

            if (LastQueryTime < lastUpdated) {
                InvalidateCache();
            }

            if (_forumDataSourceCache.ContainsKey(id)) {
                return _forumDataSourceCache[id];
            }
            else {
                return null;
            }
        }

        public IEnumerable<ForumDataSource> GetAllForumDataSources() {
            if (!IsCacheBuilt) {
                return null;
            }

            return _forumDataSourceCache.Values;
        }

        public IEnumerable<ForumDataSource> GetDataSourcesByGameId(int id) {
            if (!IsCacheBuilt) {
                return null;
            }

            if (_dataSourcesByGame.ContainsKey(id)) {
                return _dataSourcesByGame[id];
            }
            else {
                return null;
            }
        }

        public IEnumerable<ForumDataSource> GetDataSourcesByLanguage(int gameId, string language) {
            if (!IsCacheBuilt) {
                return null;
            }

            if (_dataSourcesByGame.ContainsKey(gameId)) {
                return _dataSourcesByGame[gameId].Where(s => s.Language == language);
            }
            else {
                return null;
            }
        }

        public IEnumerable<DataSourceActivity> GetAllRecentActivity(int[] dataSourceIds, DateTime since) {
            Dictionary<int, DataSourceActivity> activity = new Dictionary<int, DataSourceActivity>();

            var query = "select CommunityDataSource.ID as DataSourceID, CommunityForum.ID as ForumID, CommunityForumThread.ID as ThreadID, CommunityForumPost.ID as PostID " +
                "from CommunityDataSource " +
                "inner join CommunityForum on CommunityForum.CommunityDataSourceID = CommunityDataSource.ID " +
                "inner join CommunityForumThread on CommunityForumThread.CommunityForumID = CommunityForum.ID " +
                "inner join CommunityForumPost on CommunityForumPost.CommunityForumThreadID = CommunityForumThread.ID " +
                "where CommunityDataSource.ID in({0}) " +
                "and CommunityForumPost.DateModified >= @since";
            query = string.Format(query, string.Join(",", dataSourceIds));

            using (var conn = new SqlConnection(_databaseConnectionString)) {
                var cmd = new SqlCommand(query, conn);
                cmd.Parameters.Add("since", SqlDbType.DateTime).Value = since;

                conn.Open();
                using (SqlDataReader reader = cmd.ExecuteReader()) {
                    while (reader.Read()) {
                        var source = GetDataSourceByID((int)reader["DataSourceID"], since);
                        var forum = CForumCache.Instance.GetForumByID((int)reader["ForumID"], since);
                        var thread = CForumThreadCache.Instance.GetThreadById((int)reader["ThreadID"], since);
                        var post = CForumThreadPostCache.Instance.GetPostById((int)reader["PostID"], since);

                        if (activity.ContainsKey(source.ID)) {
                            var sourceUpdate = activity[source.ID];

                            // Add/Update the forums
                            if (forum == null) continue;
                            UpdatedForum forumUpdate = sourceUpdate.Forums.FirstOrDefault(p => p.ForumID == forum.ID);
                            
                            if (forumUpdate != null) {
                                if (thread == null) continue;
                                UpdatedThread threadUpdate = forumUpdate.Threads.FirstOrDefault(p => p.ID == thread.ID);
                                
                                if (threadUpdate != null) {
                                    if (post == null) continue;    
                                    threadUpdate.Posts.Add(post);
                                }
                                else { // create a new thread for the forum
                                    threadUpdate = new UpdatedThread() {
                                        ID = thread.ID,
                                        Thread = thread
                                    };
                                    forumUpdate.Threads.Add(threadUpdate);

                                    // Add the post to it
                                    threadUpdate.Posts = new List<CForumThreadPost>();
                                    if (post == null) continue;
                                    threadUpdate.Posts.Add(post);
                                }
                            }
                            else { // create a new forup for the sourceUpdate
                                if (forum == null) continue;
                                forumUpdate = new UpdatedForum() {
                                    ForumID = forum.ID,
                                    Forum = forum
                                };
                                sourceUpdate.Forums.Add(forumUpdate);

                                forumUpdate.Threads = new List<UpdatedThread>();

                                if (thread == null) continue;
                                var threadUpdate = new UpdatedThread() {
                                    ID = thread.ID,
                                    Thread = thread
                                };
                                forumUpdate.Threads.Add(threadUpdate);

                                // Add the post to it
                                threadUpdate.Posts = new List<CForumThreadPost>();
                                if (post == null) continue;
                                threadUpdate.Posts.Add(post);
                            }
                        }
                        else {
                            // Create the new data source activity
                            if (source == null) continue;
                            var sourceUpdate = new DataSourceActivity() {
                                DataSourceID = source.ID,
                                DataSource = source
                            };

                            // Add the forum to it
                            sourceUpdate.Forums = new List<UpdatedForum>();
                            if (forum != null)
                            {
                                var forumUpdate = new UpdatedForum()
                                {
                                    ForumID = forum.ID,
                                    Forum = forum
                                };
                                sourceUpdate.Forums.Add(forumUpdate);

                                // Add the thread to it
                                forumUpdate.Threads = new List<UpdatedThread>();
                                if (thread != null)
                                {
                                    var threadUpdate = new UpdatedThread()
                                    {
                                        ID = thread.ID,
                                        Thread = thread
                                    };
                                    forumUpdate.Threads.Add(threadUpdate);

                                    // Add the post to it
                                    threadUpdate.Posts = new List<CForumThreadPost>();
                                    if (post != null)
                                    {
                                        threadUpdate.Posts.Add(post);
                                    }
                                }
                            }

                            // Add it to the collection
                            activity.Add(source.ID, sourceUpdate);
                        }
                    }// while read
                }//using reader
            }// using conn

            return activity.Values;
        }
    }
}