﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;

using Curse.AddOns;
using Curse.CloudServices.Authentication;
using Curse.Extensions;
using System.Configuration;
using System.IO;
using System.ServiceModel.Web;

using System.Data.SqlClient;
using System.Data;
using Curse.AddOnService.Extensions;
using Curse.AddOnService.Requests;
using Curse.ClientService.Models;
using Curse.Logging;

namespace Curse.AddOnService
{
    [ServiceBehavior(Name = "AddOnService", Namespace = "http://addonservice.curse.com/", AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Multiple, InstanceContextMode = InstanceContextMode.PerSession)]
    public class AddOnService : IAddOnService
    {   
        static AddOnService()
        {
        }

        public AddOn GetAddOn(int id)
        {
            try
            {
                return AddOnCache.Instance.GetCachedAddOn(id);
            }
            catch
            {
                return null;
            }
        }

        public Stream GetAddOnDump(int id)
        {
            WebOperationContext.Current.OutgoingResponse.ContentType = "text/html";

            AddOn addon = AddOnCache.Instance.GetCachedAddOn(id);
            if (addon == null)
            {
                return new MemoryStream(System.Text.Encoding.UTF8.GetBytes("Unauthorized access"));

            }
            else
            {
                string response = addon.DumpAllProperties().Replace(Environment.NewLine, "<br>");
                return new MemoryStream(System.Text.Encoding.UTF8.GetBytes(response));
            }
        }

        public string ResetSingleAddonCache(int id)
        {

            if (!AddOnCache.Instance.IsCacheBuilt)
            {
                return "Addon cache cannot be reset: Cache is not yet built.";
            }

            if (AddOnCache.Instance.IsCacheUpdating)
            {
                return "Addon cache cannot be reset: Cache is currently updating.";
            }

            try
            {
                AddOnCache.Instance.ResetSingleAddonCache(id);
                return "Addon cache will be reset.";
            }
            catch (Exception ex)
            {
                return "Addon cache cannot be reset: Exception Occurred. Details: " + ex.GetExceptionDetails();
            }

        }

        public string ResetAllAddonCache()
        {

            if (!AddOnCache.Instance.IsCacheBuilt)
            {
                return "Addon cache cannot be reset: Cache is not yet built.";
            }

            if (AddOnCache.Instance.IsCacheUpdating)
            {
                return "Addon cache cannot be reset: Cache is currently updating.";
            }

            AddOnCache.Instance.ResetCache();

            return "Cache will now be reset.";
        }

        public string GetChangeLog(int addonID, int fileID)
        {
            AddOn addon = AddOnCache.Instance.GetCachedAddOn(addonID);
            if (addon == null)
            {
                return null;
            }

            AddOnFile file = null;
            addon.Files.TryGetValue(fileID, out file);

            if (file != null)
            {
                //return file.Changelog;
                return file.GetChangeLog();
            }
            else
            {
                return null;
            }

        }

        public string v2GetChangeLog(int addonID, int fileID)
        {
            return GetChangeLog(addonID, fileID);
        }

        public string GetAddOnDescription(int id)
        {
            AddOn addon = AddOnCache.Instance.GetCachedAddOn(id);
            if (addon == null)
            {
                return null;
            }
            return addon.FullDescription;
        }

        public string v2GetAddOnDescription(int id)
        {
            return GetAddOnDescription(id);
        }

        public AddOnFile GetAddOnFile(int addonID, int fileID)
        {
            AddOn addon = AddOnCache.Instance.GetCachedAddOn(addonID);
            if (addon == null)
            {
                return null;
            }

            AddOnFile file = null;
            addon.Files.TryGetValue(fileID, out file);
            return file;
        }

        public Dictionary<int, List<AddOnFile>> GetAddOnFiles(List<AddOnFileKey> addOnFileKeys)
        {
            var addonFiles = new Dictionary<int, List<AddOnFile>>();
            foreach (var keyGroup in addOnFileKeys.GroupBy(g=>g.AddOnID))
            {
                var addon = AddOnCache.Instance.GetCachedAddOn(keyGroup.Key);
                if (addon == null)
                {
                    continue;
                }

                var files = new List<AddOnFile>();
                foreach (var fileID in keyGroup.Select(g=>g.FileID))
                {
                    AddOnFile file;
                    if (addon.Files.TryGetValue(fileID, out file))
                    {
                        files.Add(file);
                    }
                }
                if (files.Count > 0)
                {
                    addonFiles.Add(keyGroup.Key, files);
                }
            }
            return addonFiles;
        } 

        public List<AddOn> GetAddOns(int[] ids)
        {
            var list = new List<AddOn>();
            foreach (var id in ids)
            {
                var addon = AddOnCache.Instance.GetCachedAddOn(id);
                if (addon != null)
                {
                    list.Add(addon);
                }
                else
                {
                    Logger.Debug("GetAddOns - Addon Not Found: " + id);
                }
            }
            return list;
        }

        public List<AddOn> v2GetAddOns(int[] ids)
        {
            return GetAddOns(ids);
        }

        public FingerprintMatchResult GetFingerprintMatches(long[] fingerprints)
        {
            return FingerprintCache.Instance.GetMatches(fingerprints);
        }

        public FingerprintMatchResult v2GetFingerprintMatches(long[] fingerprints)
        {
            return GetFingerprintMatches(fingerprints);
        }

        public List<FuzzyMatch> GetFuzzyMatches(int gameID, FolderFingerprint[] folderFingerprints)
        {

            return FingerprintCache.Instance.GetFuzzyMatches(gameID, folderFingerprints);
        }

        public DownloadToken GetDownloadToken(int fileID)
        {
            return new DownloadToken(OperationContext.Current.GetClientIPAddress());
        }

        public DownloadToken GetSecureDownloadToken(int fileID, int userID, bool hasPremium, int subscriptionToken)
        {           
            return new DownloadToken(OperationContext.Current.GetClientIPAddress());
        }

        public RepositoryMatch GetRepositoryMatchFromSlug(string gameSlug, string addonSlug)
        {
            return AddOnCache.Instance.GetRepositoryMatchFromSlug(gameSlug, addonSlug);
        }

        public List<AddOnFile> GetAddOnLatestFiles(int addonID)
        {
            AddOn addon = AddOnCache.Instance.GetCachedAddOn(addonID);
            if (addon == null)
            {
                return null;
            }

            return addon.LatestFiles;
        }

        public AddOnFile[] GetAllFilesForAddOn(int addOnID)
        {
            if (AddOnCache.Instance.IsCacheBuilt)
            {
                var addOnFiles = AddOnCache.Instance.FileCache[addOnID];
                return addOnFiles.Values.Where(f => f.FileStatus == FileStatus.Normal || f.FileStatus == FileStatus.SemiNormal).ToArray();
            }
            else
            {
                return null;
            }
        }

        #region Addon Sync (Premium Only)

        public ServiceResponse<List<SyncedGameInstance>> GetSyncProfile()
        {
            var session = AuthenticationContext.Current;
            try
            {
                return AddOnSyncCache.Instance.GetSyncProfile(session.UserID);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "CreateSyncGroup - Unhandled Exception!");
                return new ServiceResponse<List<SyncedGameInstance>>(ServiceResponseStatus.UnknownException, ex.Message);
            }
        }

        public ServiceResponse<SyncedGameInstance> CreateSyncGroup(string instanceName, int gameID, string computerName, string instanceGUID, string instanceLabel)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return AddOnSyncCache.Instance.CreateSyncGroup(session.UserID, gameID, instanceName, computerName, instanceGUID, instanceLabel);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "CreateSyncGroup - Unhandled Exception!");
                return new ServiceResponse<SyncedGameInstance>(ServiceResponseStatus.UnknownException, ex.Message);
            }

        }

        public ServiceResponse JoinSyncGroup(int instanceID, string computerName, string instanceGUID, string instanceLabel)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return AddOnSyncCache.Instance.JoinSyncGroup(session.UserID, instanceID, computerName, instanceGUID, instanceLabel);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "JoinSyncGroup - Unhandled Exception!");
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message);
            }

        }

        public ServiceResponse LeaveSyncGroup(int instanceID, int computerID, string instanceGUID)
        {
            var session = AuthenticationContext.Current;
            try
            {
                return AddOnSyncCache.Instance.LeaveSyncGroup(session.UserID, instanceID, computerID, instanceGUID);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "LeaveSyncGroup - Unhandled Exception!");
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message);
            }
        }

        public ServiceResponse SaveSyncSnapshot(int instanceID, SyncedAddon[] syncedAddons)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return AddOnSyncCache.Instance.SaveSyncSnapshot(session.UserID, instanceID, syncedAddons);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "SaveSyncSnapshot - Unhandled Exception!");
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message);
            }

        }

        public ServiceResponse SaveSyncTransactions(int instanceID, SyncTransaction[] transactions)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return AddOnSyncCache.Instance.SaveSyncTransactions(session.UserID, instanceID, transactions);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "SaveSyncTransactions - Unhandled Exception!");
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message);
            }
        }

        #endregion

        #region Saved Game Sync

        public ServiceResponse<List<SavedGame>> GetSavedGames()
        {
            var session = AuthenticationContext.Current;

            try
            {
                if (SavedGameSyncCache.Instance.RestrictionLevel == ESavedGameRestrictionLevel.Disabled)
                {
                    return new ServiceResponse<List<SavedGame>>(ServiceResponseStatus.Successful, new List<SavedGame>());
                }
                else
                {
                    return SavedGameSyncCache.Instance.GetSavedGames(session.UserID);
                }
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "GetSavedGames - Unhandled Exception!");
                return new ServiceResponse<List<SavedGame>>(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        public ServiceResponse DeleteSavedGame(int savedGameId)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return SavedGameSyncCache.Instance.DeleteSavedGame(session.UserID, savedGameId);
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "DeleteSavedGame - Unhandled Exception!");                
                return new ServiceResponse(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        public ServiceResponse DeleteSavedGameRevision(int savedGameId, int revisionId)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return SavedGameSyncCache.Instance.DeleteSavedGameRevision(session.UserID, savedGameId, revisionId);
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "DeleteSavedGameRevision - Unhandled Exception!");                
                return new ServiceResponse(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        public ServiceResponse SetSavedGameStatus(int savedGameId, ESavedGameStatus status)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return SavedGameSyncCache.Instance.SetSavedGameStatus(session.UserID, savedGameId, status);
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "SetSavedGameStatus - Unhandled Exception!");                
                return new ServiceResponse(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        public ServiceResponse SetSavedGameName(int savedGameId, string name)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return SavedGameSyncCache.Instance.SetSavedGameName(session.UserID, savedGameId, name);
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "SetSavedGameName - Unhandled Exception!");                
                return new ServiceResponse(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        public ServiceResponse SetSavedGameDescription(int savedGameId, string description)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return SavedGameSyncCache.Instance.SetSavedGameDescription(session.UserID, savedGameId, description);
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "SetSavedGameDescription - Unhandled Exception!");                
                return new ServiceResponse(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        public ServiceResponse SetSavedGameDefaultRevision(int savedGameId, int revisionId)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return SavedGameSyncCache.Instance.SetSavedGameDefaultRevision(session.UserID, savedGameId, revisionId);
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "SetSavedGameDefaultRevision - Unhandled Exception!");                
                return new ServiceResponse(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        public ServiceResponse SetSavedGameFavoriteRevision(int savedGameId, int revisionId)
        {
            var session = AuthenticationContext.Current;

            try
            {
                return SavedGameSyncCache.Instance.SetSavedGameFavoriteRevision(session.UserID, savedGameId, revisionId);
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "SetSavedGameFavoriteRevision - Unhandled Exception!");                
                return new ServiceResponse(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        public ServiceResponse<SavedGameConstraints> GetSavedGameConstraints()
        {
            return new ServiceResponse<SavedGameConstraints>(ServiceResponseStatus.Successful, new SavedGameConstraints(0, 0));
        }

        public ServiceResponse<ESavedGameRestrictionLevel> GetSavedGameRestrictionLevel()
        {
            try
            {
                return new ServiceResponse<ESavedGameRestrictionLevel>(ServiceResponseStatus.Successful, SavedGameSyncCache.Instance.RestrictionLevel);
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "GetSavedGameRestrictionLevel - Unhandled Exception!");                
                return new ServiceResponse<ESavedGameRestrictionLevel>(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        public ServiceResponse SetSavedGameRestrictionLevel(ESavedGameRestrictionLevel restrictionLevel)
        {
            try
            {
                SavedGameSyncCache.Instance.RestrictionLevel = restrictionLevel;
                return new ServiceResponse(ServiceResponseStatus.Successful);
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "SetSavedGameRestrictionLevel - Unhandled Exception!");                
                return new ServiceResponse(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        public ServiceResponse UploadAvailableForUser()
        {
            var session = AuthenticationContext.Current;

            try
            {
                return CloudUserCache.Instance.UserOrSlotsAvailable(session.UserID);
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "UploadAvailableForUser - Unhandled Exception!");                
                return new ServiceResponse(ServiceResponseStatus.UnknownException, exc.GetExceptionDetails());
            }
        }
        #endregion

        public string CacheHealthCheck()
        {
            if (!AddOnCache.Instance.IsCacheBuilt)
            {
                return "NoCache";
            }

            var addonID = 227724;
  
            var addon = AddOnCache.Instance.GetCachedAddOn(addonID);
            if (addon == null)
            {
                return "NoAddon";
            }

            return "Success";
        }
        
        public string HealthCheck()
        {
            using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["RoamingDBRadon"].ConnectionString))
            {
                conn.Open();
            }

            return "Success";
        }

#if DEBUG
        public string TestDatababaseReadSpeed()
        {
            string connectiongString = ConfigurationManager.ConnectionStrings["RoamingDBRadon"].ConnectionString;
            DateTime startTime = DateTime.UtcNow;
            int counter = 0;
            using (var conn = new SqlConnection(connectiongString))
            {
                conn.Open();
                var command = conn.CreateCommand();
                command.CommandText = "curseService_GetProjectFileGameVersions";
                command.CommandType = CommandType.StoredProcedure;
                command.Parameters.Add("LastUpdated", SqlDbType.DateTime).Value = new DateTime(1979, 5, 17); ;


                using (SqlDataReader gameVersionReader = command.ExecuteReader())
                {
                    while (gameVersionReader.Read())
                    {
                        ++counter;
                        var fileId = gameVersionReader.GetInt32(gameVersionReader.GetOrdinal("ProjectFileID"));
                        var versionName = gameVersionReader.GetString(gameVersionReader.GetOrdinal("Name"));

                    }
                }
            }
            return "Read " + counter.ToString("###,##0") + " addon file game version mappings in " + (DateTime.UtcNow - startTime).TotalSeconds.ToString("###,##0.00") + " seconds from: " + connectiongString;
        }
#endif
        
    }
}
