﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data.SqlClient;
using System.Threading;
using Curse.Logging;

namespace Curse.ClientService
{
    public abstract class BaseLookupCache
    {
        private const int RELOAD_LOOKUP_INTERVAL = 5000;
        protected string _databaseConnectionString;
        protected Dictionary<CustomKey, Int32> _lookupCache = new Dictionary<CustomKey, Int32>(new CustomKey.CustomKeyComparer());
        protected bool _lookupsLoaded = false;
        private string[] _lookupTables;
        private Thread _reloadLookupThread = null;

        protected BaseLookupCache(string connectionString, string[] lookupTables)
        {
            _databaseConnectionString = connectionString;
            _lookupTables = lookupTables;
            _lookupsLoaded = ReloadLookupCache();
            _reloadLookupThread = new Thread(ReloadLookupCacheThread) { IsBackground = true };
            _reloadLookupThread.Start();

        }
        private void ReloadLookupCacheThread()
        {
            var aborted = false;
            while (!aborted)
            {
                Thread.Sleep(RELOAD_LOOKUP_INTERVAL);
                try
                {
                    ReloadLookupCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _reloadLookupThread.Join(100);
                    Logger.Info(GetType().Name + " Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {
                    Logger.Info(GetType().Name +  " Thread Exception!");
                }
            }
        }

        private bool ReloadLookupCache()
        {
            try
            {
                using (var conn = new SqlConnection(_databaseConnectionString))
                {
                    conn.Open();
                    var command = conn.CreateCommand();
                    foreach (var tableName in _lookupTables)
                    {
                        command.CommandText = "select * from " + tableName;
                        using (SqlDataReader reader = command.ExecuteReader())
                        {
                            while (reader.Read())
                            {
                                _lookupCache[GetLookupKey(tableName, (string)reader["LookupLabel"])] = (int)reader["LookupID"];
                            }
                        }
                    }
                }
                return true;
            }
            catch (ThreadAbortException)
            {
                Logger.Debug("Failed to load lookup cache. Service is shutting down.");
                return false;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "CClientUsageCache - Failed to load lookup cache!");
                return false;
            }
        }

        protected CustomKey GetLookupKey(string tableName, string nameValue)
        {
            return new CustomKey(tableName.ToLowerInvariant(), nameValue.ToLowerInvariant());
        }

        protected int GetLookupID(SqlConnection conn, string tableName, string labelValue)
        {
            if (string.IsNullOrEmpty(labelValue))
            {
                labelValue = "Unknown";
            }

            labelValue = labelValue.Trim();

            var lookupKey = GetLookupKey(tableName, labelValue);
            if (_lookupCache.ContainsKey(lookupKey))
            {
                return _lookupCache[lookupKey];
            }

            using (var command = conn.CreateCommand())
            {
                command.CommandText = "insert into " + tableName +
                                      "(LookupLabel) output inserted.LookupID values(@LookupLabel)";
                var param = command.Parameters.Add("@LookupLabel", System.Data.SqlDbType.NVarChar, 256);
                param.Value = labelValue;
                var id = (int) command.ExecuteScalar();
                _lookupCache[lookupKey] = id;
                return id;
            }
        }
    }
}
