﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Configuration;
using System.Threading;
using System.Data.SqlClient;
using System.Data;
using System.Diagnostics;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using Curse.AddOns;
using Curse.AddOnService.Utilities;
using Curse.Extensions;
using Curse.Logging;
using ICSharpCode.SharpZipLib.BZip2;
using System.Runtime.Serialization.Json;
using Curse.AddOnService.Extensions;
using Newtonsoft.Json;

namespace Curse.AddOnService
{
    public class AddOnCache
    {
        private Dictionary<int, AddOn> _addOnCache = new Dictionary<int, AddOn>();
        private Dictionary<string, RepositoryMatch> _slugCache = new Dictionary<string, RepositoryMatch>();
        private DateTime _lastQueryTime = new DateTime(1979, 5, 17);
        private readonly int _updateThreadInterval;
        private Thread _updateThread = null;
        private readonly string _databaseConnectionString = null;
        private bool _isCacheUpdating = false;
        private readonly Dictionary<FeedTimespan, Int32> _feedLifespans = new Dictionary<FeedTimespan, Int32>();
        private readonly string _staticFileDestination = null;
        private readonly bool _createFeedFiles = false;
        private readonly Uri[] _cdnUrlFormats;

        private Dictionary<int, Dictionary<int, AddOnFile>> _fileCache = new Dictionary<int, Dictionary<int, AddOnFile>>();

        public Dictionary<int, Dictionary<int, AddOnFile>> FileCache
        {
            get { return _fileCache; }
            set { _fileCache = value; }
        }

        private static readonly AddOnCache _instance = new AddOnCache();

        public static AddOnCache Instance
        {
            get
            {
                return _instance;
            }
        }

        public AddOnCache()
        {
            IsCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["RoamingDBRadon"].ConnectionString;
            _staticFileDestination = ConfigurationManager.AppSettings["FeedPath"];
            _createFeedFiles = (Environment.MachineName.ToLower().Equals(ConfigurationManager.AppSettings["JobMachineName"], StringComparison.InvariantCultureIgnoreCase));
            
#if DEBUG
            _createFeedFiles = true;
#endif
            // Populate feed lifespans
            _feedLifespans.Add(FeedTimespan.Complete, (int)TimeSpan.FromHours(1).TotalSeconds);
            _feedLifespans.Add(FeedTimespan.Weekly, (int)TimeSpan.FromHours(1).TotalSeconds);
            _feedLifespans.Add(FeedTimespan.Daily, (int)TimeSpan.FromMinutes(30).TotalSeconds);
            _feedLifespans.Add(FeedTimespan.Hourly, (int)TimeSpan.FromMinutes(10).TotalSeconds);
        }

        public bool IsCacheUpdating
        {
            get
            {
                return _isCacheUpdating;
            }
        }

        public bool IsCacheBuilt { get; private set; }

        public void Initialize()
        {
            _updateThread = new Thread(CacheThread) { IsBackground = true, Priority = ThreadPriority.Normal };
            _updateThread.Start();
        }

        public void ResetCache()
        {
            _lastQueryTime = new DateTime(1979, 5, 17);
        }

        public RepositoryMatch GetRepositoryMatchFromSlug(string gameSlug, string addOnSlug)
        {
            string slug = GetSlugKey(gameSlug, addOnSlug);
            if (_slugCache.ContainsKey(slug))
            {
                return _slugCache[slug];
            }
            else
            {
                return null;
            }
        }

        public AddOn GetCachedAddOn(int id)
        {
            if (!IsCacheBuilt)
            {
                return null;
            }

            return _addOnCache.ContainsKey(id) ? _addOnCache[id] : null;
        }

        private void CacheThread()
        {
            while (true)
            {

                _isCacheUpdating = true;
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    Logger.Info("Thread Abort Exception. Service shutting down.");
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Update Thread Exception");
                }
                finally
                {
                    _isCacheUpdating = true;
                    Thread.Sleep(_updateThreadInterval);
                }

            }
        }

        public void ResetSingleAddonCache(int id)
        {
            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "update Project set DateSynced = GETUTCDATE() where Project.ID = @ProjectID";
                    var param = command.Parameters.Add("@ProjectID", SqlDbType.Int);
                    param.Value = id;
                    command.ExecuteNonQuery();
                }
            }
        }

        private bool _isInitialLoad = true;

        private DataTable GetGameTable(IEnumerable<Game> games)
        {
            var gameTable = new DataTable();

            gameTable.Columns.Add("ID", typeof(Int32));           

            // Add the user rows to the table
            foreach (var game in games)
            {
                gameTable.Rows.Add(game.ID);
            }

            return gameTable;
        }

        private void UpdateCache()
        {
            var wasInitialLoad = _isInitialLoad;
            if (_isInitialLoad)
            {
                _isInitialLoad = false;
            }

            var startTime = DateTime.UtcNow;
            var addOnCache = new Dictionary<int, AddOn>(_addOnCache);
            AddOn[] completeAddOnList = null;

            Action<string> logDelegate = (msg) =>
            {
                if (!wasInitialLoad) // Only log the initial load
                {
                    return;
                }

                Logger.Info(msg);
            };


            var changeDate = _lastQueryTime.AddMinutes(-5);
            var lastQueryTime = DateTime.UtcNow;
            
            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to establish connection to database!", _databaseConnectionString);
                    return;
                }
                
                logDelegate("Retrieving initial addon list...");

                using (var command = new SqlCommand("curseService_GetAddOnListv4", conn))
                {
                    command.CommandTimeout = 90; // This SP can be very slow
                    command.CommandType = CommandType.StoredProcedure;
                    command.Parameters.Add("LastUpdated", SqlDbType.DateTime).Value = changeDate;                    
                    
                    var allGames = GameCache.Instance.Games.Where(p => p.SupportsAddons);
                    if (ConfigurationManager.AppSettings["DebugGameID"] != null)
                    {
                        Logger.Info("Limiting addon cache to game: " + ConfigurationManager.AppSettings["DebugGameID"]);
                        allGames = allGames.Where(p => p.ID == int.Parse(ConfigurationManager.AppSettings["DebugGameID"]));
                    }

                    var gameTable = GetGameTable(allGames);

                    command.Parameters.Add("@GameIDs", SqlDbType.Structured).Value = gameTable;

                    // Populate the cache with recently modified addons:
                    logDelegate("Beginning Addon DB Call...");

                    using (var reader = command.ExecuteReader())
                    {
                        logDelegate("Completed Addon DB Call...");

                        // Get a cache of all authors
                        logDelegate("Building Author Cache...");
                        var authorCache = AuthorCache.Instance.GetAuthorCache(conn, changeDate);

                        // Get a cache of all authors
                        logDelegate("Building Attachment Cache...");
                        var attachmentCache = AttachmentCache.Instance.GetAttachmentCache(conn, changeDate);

                        // Get a cache of all categories
                        logDelegate("Building Category Cache...");
                        var categoryCache = CategoryCache.Instance.GetCategoryCache(conn, changeDate);

                        // Get a cache of the file dependenecies:
                        logDelegate("Building File Cache...");
                        var fileCache = Curse.AddOnService.FileCache.Instance.GetFileCache(conn, changeDate, gameTable);

                        lock (_fileCache)
                        {
                            foreach (var fileList in fileCache)
                            {
                                Dictionary<int, AddOnFile> list;
                                _fileCache.TryGetValue(fileList.Key, out list);

                                if (list == null)
                                {
                                    _fileCache.Add(fileList.Key, fileList.Value);
                                }
                                else
                                {
                                    _fileCache[fileList.Key] = fileList.Value;
                                }
                            }
                        }
                                                

                        var addonCount = 0;

                        while (reader.Read())
                        {
                            var addOnId = reader.GetInt32(reader.GetOrdinal("addon_id"));
                            var existingFiles = addOnCache.ContainsKey(addOnId) ? addOnCache[addOnId].Files : new Dictionary<int, AddOnFile>();
                            
                            var addon = new AddOn();
                            addon.SetFromDataReader(reader, authorCache, categoryCache, fileCache, existingFiles, attachmentCache);
                            addOnCache[addOnId] = addon;

                            if ((++addonCount % 1000) == 0)
                            {
                                logDelegate(string.Format("Processed 1,000 Addons, current total: {0}", addOnCache.Count.ToString("###,##0")));
                            }
                        }
                    }
                }

                logDelegate("Rebuilding fingerprint index...");
                FingerprintCache.Instance.Rebuild(addOnCache);

                // Construct the complete list:
                logDelegate("Constructing complete list...");
                completeAddOnList = addOnCache.Values.Where(p => p.IsAvailable && p.DefaultFileId > 0).ToArray();

                // Do this for complete addon count count
                logDelegate("Total Addons: " + addOnCache.Count.ToString("###,##0"));

                foreach (var game in GameCache.Instance.Games.Where(p => p.SupportsAddons))
                {
                    var totalCount = addOnCache.Values.Count(p => p.GameId == game.ID);
                    var availableCount = addOnCache.Values.Count(p => p.GameId == game.ID && p.IsAvailable);
                    logDelegate(string.Format("{0} - Total: {1}, Avaiable: {2}", game.Name, totalCount.ToString("###,##0"), availableCount.ToString("###,##0")));
                }

                var slugCache = new Dictionary<string, RepositoryMatch>();
                var gameStages = GameCache.Instance.Games.ToDictionary(game => game.ID, game => game.Slug.ToLower());

                // Construct the slug cache
                foreach (var kvp in addOnCache)
                {
                    var addon = kvp.Value;
                    var gameStage = gameStages[addon.GameId];
                    slugCache[GetSlugKey(gameStage, addon.Slug.ToLower())] = new RepositoryMatch()
                    {
                        Id = addon.Id,
                        LatestFiles = addon.LatestFiles
                    };
                }

                lock (_slugCache)
                {
                    _slugCache = slugCache;
                }

                lock (_addOnCache)
                {
                    _addOnCache = addOnCache;
                }

                _lastQueryTime = lastQueryTime;
            }


            IsCacheBuilt = true;

            logDelegate("Addon cache built in " + (DateTime.UtcNow - startTime).TotalSeconds + " seconds");
            
            // Build individual fingerprint cache
            CreateFeedFiles(completeAddOnList);

            // Get a cache of the individual fingerprints:

            if (ConfigurationManager.AppSettings["DebugDisableFingerprints"] == null || !bool.Parse(ConfigurationManager.AppSettings["DebugDisableFingerprints"]))
            {
                var fuzzyTimer = Stopwatch.StartNew();
                logDelegate("Building fuzzy fingerprint cache...");

                Dictionary<int, List<IndividualFileFingerprint>> fingerprintCache;

                var fuzzyMatchGames = GameCache.Instance.Games.Where(p => p.ID == 1); // Hard code to WoW for now

                if (ConfigurationManager.AppSettings["DebugGameID"] != null)
                {
                    Logger.Info("Limiting addon cache to game: " + ConfigurationManager.AppSettings["DebugGameID"]);
                    fuzzyMatchGames = fuzzyMatchGames.Where(p => p.ID == int.Parse(ConfigurationManager.AppSettings["DebugGameID"]));
                }

                var gameTable = GetGameTable(fuzzyMatchGames);

                using (var conn = new SqlConnection(_databaseConnectionString))
                {
                    conn.Open();
                    fingerprintCache = FingerprintCache.Instance.GetIndividualFingerprintCache(conn, changeDate, gameTable);
                }

                logDelegate("Completed fuzzy fingerprint cache in " + fuzzyTimer.Elapsed.TotalSeconds.ToString("F2") + " seconds.");

                foreach (var addon in _addOnCache.Values)
                {
                    addon.SetIndividualFileFingerprints(fingerprintCache);
                }

                fuzzyTimer.Restart();
                logDelegate("Rebuilding fuzzy fingerprint index...");
                FingerprintCache.Instance.RebuildFuzzy(addOnCache);
                logDelegate("Completed fuzzy fingerprint index in " + fuzzyTimer.Elapsed.TotalSeconds.ToString("F2") + " seconds.");
            }

            GC.Collect();

        }

        private readonly Dictionary<FeedTimespan, DateTime> _feedTimestamps = new Dictionary<FeedTimespan, DateTime>();

        private bool IsFeedExpired(FeedTimespan feedTimespan)
        {
            DateTime timestamp;
            var lifespan = _feedLifespans[feedTimespan];

            if (!_feedTimestamps.TryGetValue(feedTimespan, out timestamp))
            {
                Logger.Debug("Feed has not yet been created: " + feedTimespan);
                return true;
            }

            var age = (int)DateTime.UtcNow.Subtract(timestamp).TotalSeconds;

            if (age <= lifespan)
            {
                return false;
            }

            Logger.Debug("Feed has expired: " + feedTimespan);
            return true;
        }

        public string GetSlugKey(string gameSlug, string addonSlug)
        {
            return gameSlug + "_" + addonSlug;
        }

        private void CreateFeedFiles(AddOn[] completeAddOnList)
        {
            if (!_createFeedFiles)
            {
                return;
            }

            foreach (FeedTimespan timespan in Enum.GetValues(typeof(FeedTimespan)))
            {
                try
                {
                    CreateFeedFile(timespan, completeAddOnList);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to create feed file!");
                }
            }
        }

        private void CreateFeedFile(FeedTimespan feedTimespan, AddOn[] completeList)
        {
            if (completeList.Length == 0)
            {
                return;
            }

            if (!IsFeedExpired(feedTimespan))
            {
                return;
            }

            var startTime = DateTime.UtcNow;
            var hours = (int)feedTimespan;

            if (hours == 1) // Special case for the Hourly feed to prevent weird race conditions.
            {
                hours = 2;
            }

            var date = hours > 0 ? DateTime.UtcNow.AddHours(-hours) : new DateTime(2000, 1, 1);

            foreach (var game in GameCache.Instance.Games.Where(p => p.SupportsAddons))
            {
                var incrementalList = completeList.Where(p => p.DateModified >= date && p.GameId == game.ID).ToList();

                if (incrementalList.Count == 0)
                {
                    continue;
                }

                Logger.Info("Creating " + feedTimespan + " feed file for " + game.Name + " with a start date of " + date, new { AddonCount = incrementalList.Count });

                var feedFile = new AddonFeed(feedTimespan, game.ID) { Data = incrementalList.OrderBy(p => p.Id).ToArray() };
                feedFile.SaveToDisk(_staticFileDestination);                               
            }

            _feedTimestamps[feedTimespan] = startTime;
        }       
    }
}
