﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.ClientService.Models;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using System.Web.Caching;
using System.IO;
using Curse.AddOnService.Extensions;

namespace Curse.AddOnService
{
    public class AddOnSyncCache
    {
        private static readonly AddOnSyncCache _instance = new AddOnSyncCache();

        public List<SyncedGameInstance> _syncCache = new List<SyncedGameInstance>();
        public readonly string _databaseConnectionString;
        public readonly string _userBackupsPath;        
        private readonly Cache _distributedCache = null;
        private readonly DataTable _syncedAddonTableSchema = null;
        private readonly TimeSpan _defaultCacheDuration = TimeSpan.FromMinutes(5);

        private AddOnSyncCache()
        {
            var table = new DataTable();
            table.Columns.Add(new DataColumn("InstanceID", typeof(int)));
            table.Columns.Add(new DataColumn("AddonID", typeof(int)));
            table.Columns.Add(new DataColumn("FileID", typeof(int)));
            _syncedAddonTableSchema = table;

            _databaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
            _userBackupsPath = ConfigurationManager.AppSettings["UserBackupsPath"];            
            _distributedCache = HttpRuntime.Cache;
            _defaultCacheDuration = TimeSpan.Parse(ConfigurationManager.AppSettings["SyncCacheDefaultDuration"]);
        }

        public static AddOnSyncCache Instance
        {
            get
            {
                return _instance;
            }
        }

        public void Initialize() {}

        private List<SyncedGameInstance> GetProfileFromCache(int pUserID)
        {
            string cacheKey = string.Format("SyncProfile-{0}", pUserID);
            
            var list = _distributedCache.Get(cacheKey) as List<SyncedGameInstance>;

            if (list == null)
            {
                list = GetProfileFromDatabase(pUserID);
                CacheProfile(list, pUserID);
            }

            return list;
        }

        public void SaveUserBackup(int userID, int instanceID, long pFingerprint, int pScreenWidth, int pScreenHeight, Stream stream)
        {
            var syncedInstances = GetProfileFromCache(userID);
            SyncedGameInstance instance = syncedInstances.FirstOrDefault(p => p.InstanceID == instanceID);
            if (instance == null)
            {
                return;
            }
                        
            UpdateBackupSettings(instanceID, pFingerprint, pScreenWidth, pScreenHeight);            

            instance.LastBackupFingerprint = pFingerprint;
            instance.LastBackupScreenWidth = pScreenWidth;
            instance.LastBackupScreenHeight = pScreenHeight;
            instance.LastBackupDate = DateTime.UtcNow;

            var di = new DirectoryInfo(Path.Combine(_userBackupsPath, userID.ToString()));
            if (!di.Exists)
            {
                di.Create();
            }

            var filename = Path.Combine(di.FullName, instanceID.ToString()) + ".zip";

            FileStream targetStream = null;
            
            using (targetStream = new FileStream(filename, FileMode.Create, FileAccess.Write, FileShare.None))
            {
                stream.CopyTo(targetStream);                
            }            
        }

        public void CacheProfile(List<SyncedGameInstance> profile, int userID)
        {
            string cacheKey = string.Format("SyncProfile-{0}", userID);
            
            _distributedCache.Remove(cacheKey);

            if(profile != null)
            {
                _distributedCache.Add(cacheKey, profile, null, Cache.NoAbsoluteExpiration, _defaultCacheDuration, CacheItemPriority.High, null);                
            }
        }

        private List<SyncedGameInstance> GetProfileFromDatabase(int userID)
        {
            var gameInstances = new Dictionary<int, SyncedGameInstance>();

            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                var command = conn.CreateCommand();
                command.CommandType = CommandType.StoredProcedure;
                command.CommandText = "spGetSyncProfile";
                var parameter = command.Parameters.Add("@UserID", SqlDbType.Int);
                parameter.Value = userID;                               

                using (var reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var instanceID = (int)reader["InstanceID"];
                        var computerID = (int)reader["ComputerID"];
                        var addonID = (int)reader["AddonID"];
                        var instanceGuid = (string)reader["InstanceGUID"];


                        SyncedGameInstance instance = null;

                        if (!gameInstances.ContainsKey(instanceID))
                        {
                            instance = new SyncedGameInstance();
                            instance.SetFromDataReader(reader);
                            gameInstances.Add(instanceID, instance);
                        }
                        else
                        {
                            instance = gameInstances[instanceID];
                        }

                        // Computer
                        if (!instance.Computers.Exists(p => p.InstanceGuid == instanceGuid))
                        {
                            var computer = new SyncedComputer();
                            computer.SetFromDataReader(reader);
                            instance.Computers.Add(computer);
                        }

                        // Addon
                        if (addonID > 0 && !instance.Addons.Exists(p => p.AddonID == addonID))
                        {
                            var addon = new SyncedAddon();
                            addon.SetFromDataReader(reader);
                            instance.Addons.Add(addon);
                        }

                    }
                }             
            }
            return gameInstances.Values.ToList();            
        }        

        public ServiceResponse LeaveSyncGroup(int userID, int instanceID, int computerID, string instanceGUID)
        {
            List<SyncedGameInstance> syncedInstances = GetProfileFromCache(userID);
            SyncedGameInstance instance = syncedInstances.FirstOrDefault(p => p.InstanceID == instanceID);
            
            if (instance == null)
            {
                return new ServiceResponse(ServiceResponseStatus.LeaveSyncGroup_GroupNotFound);
            }

            if (!instance.Computers.Exists(p => p.ComputerID == computerID))
            {
                return new ServiceResponse(ServiceResponseStatus.LeaveSyncGroup_ComputerNotFound);                
            }
            
            RemoveComputerFromGroup(computerID, instanceID, instanceGUID);            
            CacheProfile(null, userID);

            return new ServiceResponse(ServiceResponseStatus.Successful);             
        }

        public ServiceResponse<SyncedGameInstance> CreateSyncGroup(int userID, int gameID, string instanceName, string pComputerName, string pInstanceGUID, string pInstanceLabel)
        {
            List<SyncedGameInstance> syncedInstances = GetProfileFromCache(userID);
            SyncedGameInstance instance = syncedInstances.FirstOrDefault(p => p.InstanceName == instanceName);

            if (instance != null)
            {
                return new ServiceResponse<SyncedGameInstance>(ServiceResponseStatus.CreateSyncGroup_GroupAlreadyExists); 
            }

            int instanceID = CreateNewGroup(userID, pComputerName, gameID, instanceName, pInstanceGUID, pInstanceLabel);            
            CacheProfile(null, userID);
            List<SyncedGameInstance> instances = GetProfileFromCache(userID);
            instance = instances.FirstOrDefault(p => p.InstanceID == instanceID);

            return new ServiceResponse<SyncedGameInstance>(ServiceResponseStatus.Successful, instance); 
        }

        public ServiceResponse<List<SyncedGameInstance>> GetSyncProfile(int userID)
        {
            return new ServiceResponse<List<SyncedGameInstance>>(ServiceResponseStatus.Successful, GetProfileFromCache(userID));
        }

        public ServiceResponse JoinSyncGroup(int pUserID, int pInstanceID, string pComputerName, string pInstanceGUID, string pInstanceLabel)
        {
            List<SyncedGameInstance> syncedInstances = GetProfileFromCache(pUserID);
            
            SyncedGameInstance instance = syncedInstances.FirstOrDefault(p => p.InstanceID == pInstanceID);

            if (instance == null)
            {
                return new ServiceResponse(ServiceResponseStatus.JoinSyncGroup_GroupNotFound);
            }

            if (instance.Computers.Any(p => p.InstanceGuid == pInstanceGUID))
            {
                return new ServiceResponse(ServiceResponseStatus.JoinSyncGroup_AlreadyInGroup);                
            }
            
            
            AddComputerToGroup(pUserID, pComputerName, pInstanceID, pInstanceGUID, pInstanceLabel);            
            CacheProfile(null, pUserID);

            return new ServiceResponse(ServiceResponseStatus.Successful);
        }

        public ServiceResponse SaveSyncTransactions(int pUserID, int pInstanceID, SyncTransaction[] transactions)
        {
            List<SyncedGameInstance> syncedInstances = GetProfileFromCache(pUserID);
            SyncedGameInstance instance = syncedInstances.FirstOrDefault(p => p.InstanceID == pInstanceID);

            if (instance == null)
            {
                return new ServiceResponse(ServiceResponseStatus.SaveSyncTransactions_GroupNotFound);
            }

            List<SyncedAddon> syncedAddons = instance.Addons;
            SyncedAddon syncedAddon = null;
            foreach (SyncTransaction transaction in transactions)
            {
                syncedAddon = syncedAddons.FirstOrDefault(p => p.AddonID == transaction.AddonID);

                switch (transaction.Type)
                {
                    case SyncTransactionType.Delete:
                        syncedAddons.RemoveAll(p => p.AddonID == transaction.AddonID);
                        break;
                    case SyncTransactionType.Install:
                        if (syncedAddon == null)
                        {
                            syncedAddon = new SyncedAddon()
                            {
                                AddonID = transaction.AddonID,
                                FileID = transaction.FileID
                            };
                            syncedAddons.Add(syncedAddon);
                        }
                        break;
                    case SyncTransactionType.Update:                        
                        if (syncedAddon != null)
                        {
                            syncedAddon.FileID = transaction.FileID;
                        }
                        break;
                }
            }
           
            SaveSyncSnapshot(pInstanceID, syncedAddons.ToArray());
            CacheProfile(null, pUserID);

            return new ServiceResponse(ServiceResponseStatus.Successful);
        }

        public ServiceResponse SaveSyncSnapshot(int userID, int instanceID, SyncedAddon[] pSyncedAddons)
        {
            List<SyncedGameInstance> syncedInstances = GetProfileFromCache(userID);
            
            if (!syncedInstances.Exists(p => p.InstanceID  == instanceID))
            {
                return new ServiceResponse(ServiceResponseStatus.SaveSyncSnapshot_GroupNotFound);
            }

            SaveSyncSnapshot(instanceID, pSyncedAddons);                            
            CacheProfile(null, userID);

            return new ServiceResponse(ServiceResponseStatus.Successful);
        }

        private void UpdateBackupSettings(int pInstanceID, long pFingerprint, int pScreenWidth, int pScreenHeight)
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "spUpdateBackupSettings";
                command.CommandType = CommandType.StoredProcedure;

                SqlParameter parameter = command.Parameters.Add("@InstanceID", SqlDbType.Int);
                parameter.Value = pInstanceID;

                parameter = command.Parameters.Add("@Fingerprint", SqlDbType.BigInt);
                parameter.Value = pFingerprint;

                parameter = command.Parameters.Add("@ScreenWidth", SqlDbType.Int);
                parameter.Value = pScreenWidth;

                parameter = command.Parameters.Add("@ScreenHeight", SqlDbType.Int);
                parameter.Value = pScreenHeight;

                command.ExecuteNonQuery();
            }
        }

        private int CreateNewGroup(int pUserID, string pComputerName, int pGameID, string pInstanceName, string pInstanceGUID, string pInstanceLabel)
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "spCreateSyncGroup";
                command.CommandType = CommandType.StoredProcedure;                

                // Input Params
                SqlParameter parameter = command.Parameters.Add("@ComputerName", SqlDbType.NVarChar);
                parameter.Value = pComputerName;

                parameter = command.Parameters.Add("@InstanceName", SqlDbType.NVarChar);
                parameter.Value = pInstanceName;

                parameter = command.Parameters.Add("@InstanceGUID", SqlDbType.NVarChar);
                parameter.Value = pInstanceGUID;

                parameter = command.Parameters.Add("@InstanceLabel", SqlDbType.NVarChar);
                parameter.Value = pInstanceLabel;

                parameter = command.Parameters.Add("@UserID", SqlDbType.Int);
                parameter.Value = pUserID;

                parameter = command.Parameters.Add("@GameID", SqlDbType.Int);
                parameter.Value = pGameID;

                // Output Params
                parameter = command.Parameters.Add("@ComputerID", SqlDbType.Int);
                parameter.Direction = ParameterDirection.Output;

                parameter = command.Parameters.Add("@InstanceID", SqlDbType.Int);
                parameter.Direction = ParameterDirection.Output;

                command.ExecuteNonQuery();

                return (int)command.Parameters["@InstanceID"].Value;
            }
        }

        private void AddComputerToGroup(int pUserID, string pComputerName, int pInstanceID, string pInstanceGUID, string pInstanceLabel)
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "spAddComputerToGroup";
                command.CommandType = CommandType.StoredProcedure;

                 //Create an output parameter for the new identity value.
                SqlParameter parameter = command.Parameters.Add("@ComputerID", SqlDbType.Int);
                parameter.Direction = ParameterDirection.Output;

                parameter = command.Parameters.Add("@ComputerName", SqlDbType.NVarChar);
                parameter.Value = pComputerName;

                parameter = command.Parameters.Add("@InstanceGUID", SqlDbType.NVarChar);
                parameter.Value = pInstanceGUID;

                parameter = command.Parameters.Add("@InstanceLabel", SqlDbType.NVarChar);
                parameter.Value = pInstanceLabel;                                                

                parameter = command.Parameters.Add("@UserID", SqlDbType.Int);
                parameter.Value = pUserID;

                parameter = command.Parameters.Add("@InstanceID", SqlDbType.Int);
                parameter.Value = pInstanceID;

        

                command.ExecuteNonQuery();
            }            
        }

        private void RemoveComputerFromGroup(int pComputerID, int pInstanceID, string pInstanceGUID)
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "spRemoveComputerFromGroup";
                command.CommandType = CommandType.StoredProcedure;

                //Create an output parameter for the new identity value.
                SqlParameter parameter = command.Parameters.Add("@ComputerID", SqlDbType.Int);
                parameter.Value = pComputerID;

                parameter = command.Parameters.Add("@InstanceID", SqlDbType.Int);
                parameter.Value = pInstanceID;

                parameter = command.Parameters.Add("@InstanceGUID", SqlDbType.NVarChar);
                parameter.Value = pInstanceGUID;

                command.ExecuteNonQuery();                
            }
        }               
        
        private void SaveSyncSnapshot(int pInstanceID, SyncedAddon[] pSyncedAddons)
        {
            
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                conn.Open();

                DataTable table = _syncedAddonTableSchema.Clone();
                foreach (SyncedAddon syncedAddon in pSyncedAddons)
                {
                    DataRow row = table.NewRow();
                    row["InstanceID"] = pInstanceID;
                    row["AddonID"] = syncedAddon.AddonID;
                    row["FileID"] = syncedAddon.FileID;
                    table.Rows.Add(row);
                }

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "delete from SyncedAddon where InstanceID = " + pInstanceID;
                command.ExecuteNonQuery();

                using (SqlBulkCopy bulk = new SqlBulkCopy(conn))
                {
                    bulk.DestinationTableName = "SyncedAddon";
                    bulk.WriteToServer(table);
                }
               
            }
        }

    }
}
