﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data.SqlClient;
using System.Configuration;
using System.Threading;
using System.Data;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using Curse.AddOns;
using System.Drawing;
using System.Runtime.Serialization.Json;
using Curse.Logging;
using ICSharpCode.SharpZipLib.BZip2;
using Curse.AddOnService.Extensions;

namespace Curse.AddOnService
{
    public class CategoryCache
    {
        // If the number of categories retrieved from the database is lower than this number, the category cache is not updated.
        public const int MinimumCategoryCount = 64;

        private int _updateThreadInterval;
        private Thread _updateThread = null;
        private string _databaseConnectionString = null;
        private string _staticFileDestination = null;
        private bool _createFeedFiles = false;

        private readonly  ConcurrentDictionary<int, Category> _categories = new ConcurrentDictionary<int, Category>();

        private static readonly CategoryCache _instance = new CategoryCache();

        public static CategoryCache Instance
        {
            get
            {
                return _instance;
            }
        }

        public Category GetByID(int id)
        {
            Category found;
            if (_categories.TryGetValue(id, out found))
            {
                return found;
            }

            return null;
        }    

        private CategoryCache()
        {
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["RoamingDBRadon"].ConnectionString;
            _staticFileDestination = ConfigurationManager.AppSettings["FeedPath"];
            _createFeedFiles = (System.Environment.MachineName.ToLower() == ConfigurationManager.AppSettings["JobMachineName"].ToLower());            

#if DEBUG
            _createFeedFiles = true;
#endif
            if (_createFeedFiles)
            {
                _updateThread = new Thread(CacheThread) { IsBackground = true };
                _updateThread.Priority = ThreadPriority.Lowest;
                _updateThread.Start();
            }
        }

        public DateTime FileDate
        {
            get;
            set;
        }

        public void Initialize() { }

        private void CacheThread()
        {

            while (true)
            {
                Thread.Sleep(_updateThreadInterval);
                try
                {
                    CreateCategoryFeed();
                }
                catch (ThreadAbortException)
                {
                    Logger.Info("Thread Abort Exception. Service shutting down.");
                    break;
                }
                catch (Exception ex)
                {

                    Logger.Error(ex, "Update Thread Exception!");
                }
            }
        }

        private void CreateCategoryFeed()
        {

            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to establish connection to database");
                    return;
                }
                var categories = new List<Category>();

                var command = new SqlCommand("select distinct ProjectCategory.*, Project.GameId"
                + " from ProjectCategory with(nolock)"
                + " inner join ProjectCategoryMap with(nolock)"
                + " on ProjectCategoryMap.ProjectCategoryId = ProjectCategory.ID"
                + " inner join Project with(nolock)"
                + " on Project.ID = ProjectCategoryMap.ProjectId"
                + " union"
                + " select distinct ProjectCategory.*, Project.GameID"
                + " from ProjectCategory with(nolock)"
                + " inner join Project with(nolock)"
                + " on ProjectCategory.ID = Project.PrimaryCategoryID"
                + " order by ProjectCategory.ID, GameId", conn);

                var distinctGameIDs = new List<int>();

                using (var reader = command.ExecuteReader())
                {

                    while (reader.Read())
                    {
                        var categoryID = (int)reader["ID"];
                        var gameID = (int)reader["GameId"];

                        if (!distinctGameIDs.Contains(gameID))
                        {
                            distinctGameIDs.Add(gameID);
                        }

                        var category = categories.FirstOrDefault(p => p.Id == categoryID);

                        if (category == null)
                        {
                            category = new Category();
                            category.SetFromDataReader(reader);
                            categories.Add(category);
                        }

                        category.GameIDs.Add(gameID);
                    }

                    //Ensure that every game in the game cache is in the distinct list so it gets the "Any Category" thing
                    foreach (var game in GameCache.Instance.Games)
                    {
                        if (!distinctGameIDs.Contains(game.ID))
                            distinctGameIDs.Add(game.ID);
                    }

                    categories.Add(new Category(0, null, "Any Category", null, null, distinctGameIDs));
                }

                foreach (var category in categories)
                {
                    _categories[category.Id] = category;
                }

                if (distinctGameIDs.Count < GameCache.MinimumGameCount)
                {
                    Logger.Info("Game ID list is incomplete. Skipping cache update, for this iteration!");
                    return;
                }


                if (categories.Count <= MinimumCategoryCount)
                {
                    Logger.Info("Category list is incomplete. Skipping cache update, for this iteration.");
                    return;
                }

                if (_createFeedFiles)
                {
                    var feedFile = new CategoryFeed() { Data = categories.ToArray() };
                    feedFile.SaveToDisk(_staticFileDestination);
                }

            }
        }
        
        public Dictionary<int, List<AddOnCategory>> GetCategoryCache(SqlConnection conn, DateTime changeDate)
        {
            var categoryCache = new Dictionary<int, List<AddOnCategory>>();
            using (var categoryCommand = new SqlCommand("curseService_GetAllAddOnCategories", conn))
            {
                categoryCommand.CommandType = CommandType.StoredProcedure;
                categoryCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                categoryCommand.Parameters["@LastUpdated"].Value = changeDate;
                categoryCommand.CommandTimeout = 300;

                using (var categoryReader = categoryCommand.ExecuteReader())
                {
                    while (categoryReader.Read())
                    {
                        var projectID = categoryReader.GetInt32(0);

                        if (!categoryCache.ContainsKey(projectID))
                        {
                            categoryCache.Add(projectID, new List<AddOnCategory>());
                        }
                        var category = new AddOnCategory();
                        category.SetFromDataReader(categoryReader);
                        categoryCache[projectID].Add(category);
                    }
                }
            }
            return categoryCache;
        }
    }
}
