﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.Extensions;
using System.Configuration;
using System.Data.SqlClient;
using System.Threading;
using System.Data;
using Curse.ClientService.Models;
using Curse.Logging;

namespace Curse.AddOnService
{
    public class CloudUserCache
    {
        private List<int> _userIds;
        private readonly object _userLock = new object();

        private readonly string _databaseConnectionString = null;
        private readonly int _updateThreadInterval;

        private int _maxCloudUsers;

        static CloudUserCache _instance = new CloudUserCache();
        public static CloudUserCache Instance { get { return _instance; } }

        public CloudUserCache()
        {
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
            _maxCloudUsers = Int32.Parse(ConfigurationManager.AppSettings["MaxCloudUsers"]);

            _userIds = new List<int>();

            UpdateCache();

            new Thread(CacheThread) { IsBackground = true, Priority = ThreadPriority.Lowest }.Start();            
        }
        public void Initialize() { }

        #region Caching
        private void CacheThread()
        {            
            while (true)
            {
                Thread.Sleep(_updateThreadInterval);
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {                                        
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Update Thread Exception");
                }
            }
        }
        private void UpdateCache()
        {
            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to establish connection to database!", _databaseConnectionString);
                    return;
                }

                var users = new List<int>();

                using (var command = new SqlCommand("SELECT * FROM CloudUsage", conn))
                {
                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            var userId = reader.GetInt32(reader.GetOrdinal("UserID"));
                            users.Add(userId);
                        }
                    }
                }

                lock (_userLock)
                {
                    _userIds = users;
                }
            }
        }
        #endregion

        public ServiceResponse UserOrSlotsAvailable(int userId)
        {
            if (_userIds.Contains(userId))
            {
                return new ServiceResponse(ServiceResponseStatus.Successful);
            }
            else if ((_maxCloudUsers - _userIds.Count) > 0)
            {
                return ReserveSlot(userId);
            }
            return new ServiceResponse(ServiceResponseStatus.UserOrSlotsAvailable_NoCloudSlotsAvailable);
        }

        public ServiceResponse ReserveSlot(int userId)
        {
            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                var cmd = new SqlCommand("spAddCloudUser", conn);
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.Parameters.Add("UserID", SqlDbType.Int).Value = userId;
                cmd.Parameters.Add("MaxNumUsers", SqlDbType.Int).Value = _maxCloudUsers;

                conn.Open();
                try
                {
                    cmd.ExecuteNonQuery(); // Procedure returns 1 for success & 0 for failure
                    lock (_userLock)
                    {
                        _userIds.Add(userId);
                    }
                    return new ServiceResponse(ServiceResponseStatus.Successful);
                }
                catch (Exception exc)
                {
                    Logger.Error(exc, "Unable to reserve slot for user!", new { UserID = userId });
                    return new ServiceResponse(ServiceResponseStatus.UnknownException, exc.GetExceptionDetails());
                }
            }
        }
    }
}
