﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Threading;
using System.Data;
using System.Data.SqlClient;
using Curse.Logging;

namespace Curse.AddOnService
{
    public class DownloadTracker
    {
        private static readonly LogCategory Logger = new LogCategory("DownloadTracker");

        private const int UpdateThreadInterval = 1000*60*5; // 5 mins by default.
        private static string _connectionString = null;
        private static Thread _writeToDatabaseThread = null;

        private static Dictionary<int, int> _fileDownloads = null;
        private static readonly object SyncRoot = new object();

        public static void Initialize()
        {
            _connectionString = ConfigurationManager.ConnectionStrings["RoamingDBRadon"].ConnectionString;
            _fileDownloads = new Dictionary<int, int>();
            _writeToDatabaseThread = new Thread(WriteToDatabaseThread) { IsBackground = true };
            _writeToDatabaseThread.Start();
        }

        public static void LogFileDownload(int fileID)
        {
            lock (SyncRoot)
            {
                if (_fileDownloads.ContainsKey(fileID))
                {
                    _fileDownloads[fileID] += 1;
                }
                else
                {
                    _fileDownloads.Add(fileID, 1);
                }
            }
        }

        public static void WriteToDatabaseThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                //Every minute for now:            
                Thread.Sleep(UpdateThreadInterval);
                try
                {
                    WriteToDatabase();
                    Logger.Info("Project download statistics saved.");
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "AddOnDownloadTracker Exception");
                }
            }

        }

        private static void WriteToDatabase()
        {

            // File Downloads
            Dictionary<int, int> fileDownloads = null;
            lock (SyncRoot)
            {
                fileDownloads = new Dictionary<int, int>(_fileDownloads);
                _fileDownloads.Clear();
            }

            var projectDownloads = new Dictionary<int,int>();
            foreach (var kvp in fileDownloads)
            {
                var projectID = FileCache.Instance.GetAddonIDByFileID(kvp.Key);
                if (projectID > 0)
                {
                    projectDownloads[projectID] = kvp.Value;
                }
                
            }

            SaveProjectDownloads(projectDownloads);            
        }

        private static void SaveProjectDownloads(Dictionary<int, int> projectDownloads)
        {
            if (projectDownloads.Count == 0)
            {
                return;
            }

            var fileDownloadsTable = new DataTable();
            fileDownloadsTable.Columns.Add("ID", typeof(int));
            fileDownloadsTable.Columns.Add("DownloadCount", typeof(int));
            fileDownloadsTable.BeginLoadData();
            foreach (var kvp in projectDownloads)
            {
                var row = fileDownloadsTable.NewRow();
                row["ID"] = kvp.Key;
                row["DownloadCount"] = kvp.Value;
                fileDownloadsTable.Rows.Add(row);
            }

            using (var conn = new SqlConnection(_connectionString))
            {
                var cmd = conn.CreateCommand();
                cmd.CommandText = "CREATE TABLE [#tmp]([ID] [int] NOT NULL, [DownloadCount] [int] NOT NULL)";

                try 
                {
                    conn.Open();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to establish connection to database!", _connectionString);
                    return;
                }

                cmd.ExecuteNonQuery();
                using (var bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, null)) 
                {
                    bulk.DestinationTableName = "#tmp";
                    bulk.WriteToServer(fileDownloadsTable);
                }

                using (var trans = conn.BeginTransaction()) 
                {
                    cmd.Transaction = trans;
                    
                    //First update the download count
                    cmd.CommandText = "UPDATE [a] SET [a].[DownloadCount] = [a].[DownloadCount] + [b].[DownloadCount] FROM dbo.Project AS [a] INNER JOIN [#tmp] AS [b] ON [b].[ID] = [a].[ID]";
                    cmd.ExecuteNonQuery();

                    //Now update the daily count
                    cmd.CommandText = "UPDATE [a] SET [a].[DownloadCount] = [a].[DownloadCount] + [b].[DownloadCount] FROM dbo.ProjectDailyDownloadCount AS [a] INNER JOIN [#tmp] AS [b] ON [b].[ID] = [a].[ID]";
                    cmd.ExecuteNonQuery();

                    trans.Commit();
                }
            }
        }        

    }
}
