﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Data.SqlClient;
using Curse.AddOns;
using System.Data;
using Curse.AddOnService.Extensions;
using Curse.Logging;

namespace Curse.AddOnService
{
    public class FileCache
    {
        private static readonly LogCategory Logger = new LogCategory("FileCache");

        private Dictionary<int, string> _changeLogs = new Dictionary<int, string>();
        public Dictionary<int, string> ChangeLogs
        {
            get { return _changeLogs; }
            set { _changeLogs = value; }
        }

        private static readonly FileCache _instance = new FileCache();

        private readonly Dictionary<int, int> _addonIDByFileID = new Dictionary<int, int>();

        public static FileCache Instance
        {
            get
            {
                return _instance;
            }
        }

        public int GetAddonIDByFileID(int fileID)
        {
            var projectID = 0;
            if (_addonIDByFileID.TryGetValue(fileID, out projectID))
            {
                return projectID;
            }
            else
            {
                return 0;
            }
        }

        public Dictionary<int, Dictionary<int, AddOnFile>> GetFileCache(SqlConnection conn, DateTime changeDate, DataTable gameTable)
        {

            var fileCache = new Dictionary<int, Dictionary<int, AddOnFile>>();
            var fileDependencyCache = new Dictionary<int, List<AddOnFileDependency>>();
            var fileGameVersionCache = new Dictionary<int, List<string>>();

            //Get the file dependancies
            Logger.Debug("Getting File Dependencies...");

            using (var dependencyCommand = new SqlCommand("curseService_GetAllFileDependenciesv2", conn))
            {
                dependencyCommand.CommandType = CommandType.StoredProcedure;
                dependencyCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                dependencyCommand.Parameters["@LastUpdated"].Value = changeDate;
                dependencyCommand.Parameters.Add("@GameIDs", SqlDbType.Structured).Value = gameTable;
                dependencyCommand.CommandTimeout = 300;

                Logger.Debug("Beginning Dependency Call");

                using (var dependencyReader = dependencyCommand.ExecuteReader())
                {
                    Logger.Debug("Dependency Call Complete");

                    while (dependencyReader.Read())
                    {
                        int fileId = dependencyReader.GetInt32(dependencyReader.GetOrdinal("addon_file_id"));
                        int dependencyId = dependencyReader.GetInt32(dependencyReader.GetOrdinal("addon_dependency_id"));
                        byte dependencyType = dependencyReader.GetByte(dependencyReader.GetOrdinal("addon_dependency_type"));

                        if (!fileDependencyCache.ContainsKey(fileId))
                        {
                            fileDependencyCache.Add(fileId, new List<AddOnFileDependency>());
                        }//if there isnt an entry for this file ID already

                        fileDependencyCache[fileId].Add(new AddOnFileDependency(dependencyId, dependencyType));
                    }
                }
            }

            Logger.Debug("File Cache: Getting GameVersions...");

            using (var command = conn.CreateCommand())
            {
                command.CommandText = "curseService_GetProjectFileGameVersions";
                command.CommandType = CommandType.StoredProcedure;
                command.Parameters.Add("LastUpdated", SqlDbType.DateTime).Value = changeDate;
                command.CommandTimeout = 300;

                Logger.Debug("Executing game version query...");
                using (var gameVersionReader = command.ExecuteReader())
                {
                    Logger.Debug("Query completed. Building dictionary...");
                    while (gameVersionReader.Read())
                    {
                        int fileId = gameVersionReader.GetInt32(gameVersionReader.GetOrdinal("ProjectFileID"));
                        string versionName = gameVersionReader.GetString(gameVersionReader.GetOrdinal("Name"));

                        if (!fileGameVersionCache.ContainsKey(fileId))
                        {
                            fileGameVersionCache.Add(fileId, new List<string>());
                        }

                        fileGameVersionCache[fileId].Add(versionName);
                    }
                }
            }
            

            // Get a cache of all files
            Logger.Debug("File Cache: Getting File Cache...");
            using (var fileCommand = new SqlCommand("curseService_GetAllAddOnFiles", conn))
            {
                fileCommand.CommandType = CommandType.StoredProcedure;
                fileCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                fileCommand.Parameters["@LastUpdated"].Value = changeDate;
                fileCommand.CommandTimeout = 300;

                Logger.Debug("Beginning File Call");
                using (var fileReader = fileCommand.ExecuteReader())
                {
                    Logger.Debug("File Call Complete");
                    while (fileReader.Read())
                    {
                        int projectID = fileReader.GetInt32(fileReader.GetOrdinal("addon_id"));
                        if (!fileCache.ContainsKey(projectID))
                        {
                            fileCache.Add(projectID, new Dictionary<int, AddOnFile>());
                        }

                        var file = new AddOnFile();
                        file.SetFromDataReader(fileReader, fileDependencyCache, fileGameVersionCache);
                        fileCache[projectID].Add(file.Id, file);
                        _addonIDByFileID[file.Id] = projectID;
                    }
                }
            }


            // Get the addon fingerprints
            Logger.Debug("Getting File Fingerprints...");
            using (var fingerprintCommand = new SqlCommand("curseService_GetAllAddOnFileFingerprints", conn))
            {
                fingerprintCommand.CommandType = CommandType.StoredProcedure;
                fingerprintCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                fingerprintCommand.Parameters["@LastUpdated"].Value = changeDate;
                fingerprintCommand.CommandTimeout = 600;

                Logger.Debug("File Cache: Beginning Fingerprint Call...");
                using (var fingerprintReader = fingerprintCommand.ExecuteReader())
                {
                    Logger.Debug("File Cache: Fingerprint Call Complete...");
                    while (fingerprintReader.Read())
                    {
                        var projectID = fingerprintReader.GetInt32(fingerprintReader.GetOrdinal("addon_id"));

                        if (!fileCache.ContainsKey(projectID))
                        {
                            continue;
                        }

                        var fileId = fingerprintReader.GetInt32(fingerprintReader.GetOrdinal("file_id"));
                        var fingerprint = fingerprintReader.GetInt64(fingerprintReader.GetOrdinal("fingerprint"));

                        string foldername = null;
                        if (fingerprintReader["folder"] != System.DBNull.Value)
                        {
                            foldername = fingerprintReader.GetString(fingerprintReader.GetOrdinal("folder"));
                        }

                        var files = fileCache[projectID];
                        AddOnFile file;

                        if (!files.TryGetValue(fileId, out file))
                        {
                            continue;
                        }
                        
                        if (file == null) { continue; }

                        file.Fingerprints.Add(fingerprint);
                        if (foldername != null)
                        {
                            var module = new AddOnModule()
                            {
                                Fingerprint = fingerprint,
                                Foldername = foldername
                            };
                            file.Modules.Add(module);
                        }
                    }
                }
            }

            return fileCache;
        }

    }
}
