﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.AddOns;
using System.Data.SqlClient;
using System.Data;
using Curse.Logging;

namespace Curse.AddOnService
{
    public class FingerprintCache
    {
        private Dictionary<long, List<FingerprintMatch>> _fingerprints;
        private Dictionary<int, Dictionary<string, List<FuzzyMatch>>> _fuzzyIndexFingerprints;
        private bool _isCacheBuilt = false;
        private bool _isFuzzyCacheBuilt = false;

        private static readonly FingerprintCache _instance = new FingerprintCache();

        public static FingerprintCache Instance
        {
            get
            {
                return _instance;
            }
        }

        private FingerprintCache()
        {
            _isCacheBuilt = false;
            _fingerprints = new Dictionary<long, List<FingerprintMatch>>();
            _fuzzyIndexFingerprints = new Dictionary<int, Dictionary<string, List<FuzzyMatch>>>();
        }

        public void Rebuild(Dictionary<int, AddOn> addonCache)
        {
            var fingerprints = new Dictionary<long, List<FingerprintMatch>>();
            
            foreach (var addonEntry in addonCache)
            {
                var addon = addonEntry.Value;

                if (!GameCache.Instance.IsValidGame(addon.GameId))
                {
                    continue;
                }

                if (!addon.IncludeInFingerprinting)
                {
                    continue;
                }

                foreach (var fileEntry in addon.Files)
                {
                    var file = fileEntry.Value;

                    if (addon.PackageType == PackageTypes.SingleFile || addon.PackageType == PackageTypes.Mod)
                    {
                        file.Fingerprints.Clear();
                        file.Fingerprints.Add(file.PackageFingerprint);
                        if (file.PackageFingerprint != 0)
                        {
                            if (!fingerprints.ContainsKey(file.PackageFingerprint))
                            {
                                fingerprints.Add(file.PackageFingerprint, new List<FingerprintMatch>());
                            }
                            fingerprints[file.PackageFingerprint].Add(new FingerprintMatch(addon, file));
                        }
                    }
                    else
                    {
                        // Addon-Level Fingerprints
                        foreach (var fingerprint in file.Fingerprints)
                        {
                            if (!fingerprints.ContainsKey(fingerprint))
                            {
                                fingerprints.Add(fingerprint, new List<FingerprintMatch>());
                            }

                            var match = new FingerprintMatch(addon, file);
                            fingerprints[fingerprint].Add(match);
                        }                       
                    }
                }
            }

            lock (_fingerprints)
            {
                _fingerprints = fingerprints;
            }
            

            _isCacheBuilt = true;
        }

        public void RebuildFuzzy(Dictionary<int, AddOn> addonCache)
        {
            var folderFingerprints = new Dictionary<int, Dictionary<string, List<FuzzyMatch>>>();

            // Setup Individual Fingerprints                  
            foreach (var game in GameCache.Instance.Games)
            {
                folderFingerprints.Add(game.ID, new Dictionary<string, List<FuzzyMatch>>());
            }

            foreach (var addonEntry in addonCache)
            {
                var addon = addonEntry.Value;

                if (!GameCache.Instance.IsValidGame(addon.GameId))
                {
                    continue;
                }

                if (!addon.IncludeInFingerprinting)
                {
                    continue;
                }

                foreach (var fileEntry in addon.Files)
                {
                    var file = fileEntry.Value;

                    // Module-Level Fingerprints
                    foreach (var folder in file.FolderFingerprints)
                    {
                        var foldername = folder.Key;

                        if (!folderFingerprints[addon.GameId].ContainsKey(foldername))
                        {
                            folderFingerprints[addon.GameId].Add(foldername, new List<FuzzyMatch>());
                        }

                        folderFingerprints[addon.GameId][foldername].Add(new FuzzyMatch(foldername, addon, file));
                    }
                }
            }

            lock (_fuzzyIndexFingerprints)
            {
                _fuzzyIndexFingerprints = folderFingerprints;
            }

            _isFuzzyCacheBuilt = true;

        }

        public List<FuzzyMatch> GetFuzzyMatches(int gameID, FolderFingerprint[] folderFingerprints)
        {
            List<FuzzyMatch> matches = new List<FuzzyMatch>();

            if (gameID == 0 || !_fuzzyIndexFingerprints.ContainsKey(gameID) || !_isFuzzyCacheBuilt)
            {
                return matches;
            }

            FuzzyMatch bestMatch = null;
            int matchCount = 0;
            int potentialMatchCount = 0;

            foreach (FolderFingerprint folderFingerprint in folderFingerprints)
            {

                List<FuzzyMatch> potentialMatches;

                if (!_fuzzyIndexFingerprints[gameID].TryGetValue(folderFingerprint.Foldername, out potentialMatches))
                {
                    continue;
                }

                int lastId = 0;
                bool singleAddonMatch = true;
                bestMatch = null;

                foreach (FuzzyMatch potentialMatch in potentialMatches)
                {
                    if (lastId > 0 && lastId != potentialMatch.Id)
                    {
                        singleAddonMatch = false;
                    }
                    lastId = potentialMatch.Id;
                    if (potentialMatch.Fingerprints.Count != folderFingerprint.Fingerprints.Count)
                    {
                        continue;
                    }
                    potentialMatchCount = potentialMatch.FuzzyMatchCount(folderFingerprint.Fingerprints);

                    if (bestMatch == null ||
                        potentialMatchCount > matchCount ||
                        bestMatch.File.FileStatus == FileStatus.Deleted && potentialMatch.File.FileStatus != FileStatus.Deleted)
                    {

                        bestMatch = potentialMatch;
                        matchCount = potentialMatchCount;
                        if (potentialMatchCount == folderFingerprint.Fingerprints.Count)
                        {
                            break;
                        }
                    }
                }

                if (bestMatch == null && singleAddonMatch)
                {
                    bestMatch = potentialMatches.FirstOrDefault();
                }

                // No match found, continue to next folder
                if (bestMatch == null)
                {
                    continue;
                }
                // Ensure we don't add the same match twice
                if (matches.Exists(p => p.Id == bestMatch.Id))
                {
                    continue;
                }
                matches.Add(bestMatch);
            }
            return matches;
        }

        public FingerprintMatchResult GetMatches(long[] pFileFingerprints)
        {
            var result = new FingerprintMatchResult();
            result.IsCacheBuilt = _isCacheBuilt;
            result.InstalledFingerprints = new List<long>(pFileFingerprints);
            
            if (!_isCacheBuilt)
            {
                return result;
            }

            FindExactMatches(result);

            if (result.ExactFingerprints.Count < result.InstalledFingerprints.Count)
            {
                FindPartialMatches(result);
            }

            return result;
        }

        public void FindExactMatches(FingerprintMatchResult pResult)
        {
            var processedFingerprints = new List<long>();
            var matches = new List<FingerprintMatch>();

            foreach (var installedFingerprint in pResult.InstalledFingerprints)
            {
                if (processedFingerprints.Contains(installedFingerprint))
                {
                    continue;
                }

                if (!_fingerprints.ContainsKey(installedFingerprint))
                {
                    continue;
                }

                var potentialMatches = _fingerprints[installedFingerprint];

                if (potentialMatches.Count == 0)
                {
                    continue;
                }

                FingerprintMatch currentMatch = null;

                foreach (var potentialMatch in potentialMatches)
                {
                    // Match must be perfect, and consume more files, be the latest matching release date.
                    if (potentialMatch.IsExactMatch(pResult.InstalledFingerprints) && IsBetterMatch(currentMatch, potentialMatch))
                    {
                        currentMatch = potentialMatch;
                    }
                }

                if (currentMatch != null)
                {
                    processedFingerprints.AddRange(currentMatch.File.Fingerprints);
                    matches.Add(currentMatch);
                }
            }

            pResult.ExactMatches = matches;
            pResult.ExactFingerprints = processedFingerprints;
        }

        public void FindPartialMatches(FingerprintMatchResult pResult)
        {
            List<long> processedFingerprints = new List<long>();
            List<FingerprintMatch> matches = new List<FingerprintMatch>();
            Dictionary<int, List<long>> matchFingerprints = new Dictionary<int, List<long>>();

            foreach (long installedFingerprint in pResult.UnmatchedFingerprints)
            {
                if (processedFingerprints.Contains(installedFingerprint))
                {
                    continue;
                }
                if (!_fingerprints.ContainsKey(installedFingerprint))
                {
                    continue;
                }
                List<FingerprintMatch> potentialMatches = _fingerprints[installedFingerprint];

                FingerprintMatch currentMatch = null;

                foreach (FingerprintMatch potentialMatch in potentialMatches)
                {
                    if (IsBetterMatch(currentMatch, potentialMatch))
                    {
                        currentMatch = potentialMatch;
                    }
                }

                if (currentMatch != null)
                {
                    processedFingerprints.AddRange(currentMatch.File.Fingerprints);
                    matchFingerprints[currentMatch.Id] = currentMatch.GetPartialMatchFingerprints(pResult.InstalledFingerprints);
                    matches.Add(currentMatch);
                }
            }
            pResult.PartialMatchFingerprints = matchFingerprints;
            pResult.PartialMatches = matches;
        }

        public bool IsBetterMatch(FingerprintMatch pCurrentMatch, FingerprintMatch pPotentialMatch)
        {
            // If there is no current match, then it is always better.
            if (pCurrentMatch == null)
            {
                return true;
            }

            // If this potential match consumes less fingerprints, then it isn't a better match
            if (pPotentialMatch.File.Fingerprints.Count < pCurrentMatch.File.Fingerprints.Count)
            {
                return false;
            }

            // if the current match isn't normal status, and the current one is, it is better
            if (pCurrentMatch.File.FileStatus != FileStatus.Normal && pPotentialMatch.File.FileStatus == FileStatus.Normal)
            {
                return true;
            }

            // if the potential match isn't normal status, and the current one is, discard it.
            if (pPotentialMatch.File.FileStatus != FileStatus.Normal && pCurrentMatch.File.FileStatus == FileStatus.Normal)
            {
                return false;
            }

            // If this is an earlier release, discard it.
            if (pPotentialMatch.File.FileDate < pCurrentMatch.File.FileDate)
            {
                return false;
            }

            return true;
        }

        public Dictionary<int, List<IndividualFileFingerprint>> GetIndividualFingerprintCache(SqlConnection conn, DateTime changeDate, DataTable gameTable)
        {
            var individualFingerprintCache = new Dictionary<int, List<IndividualFileFingerprint>>();

            // Get a cache of the individual fingerprints:                                        
            using (var fingerprintCommand = new SqlCommand("curseService_GetAllIndividualFingerprintsV2", conn))
            {
                fingerprintCommand.CommandType = CommandType.StoredProcedure;
                fingerprintCommand.Parameters.Add(new SqlParameter("@LastUpdated", SqlDbType.DateTime));
                fingerprintCommand.Parameters["@LastUpdated"].Value = changeDate;
                fingerprintCommand.Parameters.Add("@GameIDs", SqlDbType.Structured).Value = gameTable;
                fingerprintCommand.CommandTimeout = 300;

                int idOrdinal = 0;
                int fileIDOrdinal = 0;
                int fingerprintOrdinal = 0;
                int folderOrdinal = 0;
                int fingerprintCount = 0;

                using (var fingerprintReader = fingerprintCommand.ExecuteReader())
                {
                    var hasOrdinals = false;
                    while (fingerprintReader.Read())
                    {
                        if (!hasOrdinals)
                        {
                            idOrdinal = fingerprintReader.GetOrdinal("addon_id");
                            fileIDOrdinal = fingerprintReader.GetOrdinal("file_id");
                            fingerprintOrdinal = fingerprintReader.GetOrdinal("fingerprint");
                            folderOrdinal = fingerprintReader.GetOrdinal("folder");
                            hasOrdinals = true;
                        }

                        var id = fingerprintReader.GetInt32(idOrdinal);
                        var fileId = fingerprintReader.GetInt32(fileIDOrdinal);
                        var fingerprint = fingerprintReader.GetInt64(fingerprintOrdinal);
                        var folder = fingerprintReader.GetString(folderOrdinal);

                        // This is because there are paths in the DB and not Folders as previously done
                        if (folder.Contains('/'))
                        {
                            // if the folder contains a slash cut it down to it's minor component
                            folder = folder.Substring(0, folder.IndexOf('/'));
                        }
                        else
                        {
                            folder = string.Empty;
                        }

                        // Add the addon to the list if it doesnt exist
                        if (!individualFingerprintCache.ContainsKey(id))
                        {
                            individualFingerprintCache.Add(id, new List<IndividualFileFingerprint>());
                        }

                        individualFingerprintCache[id].Add(new IndividualFileFingerprint(id, fileId, fingerprint, folder));

                        const int logInterval = 500000;
                        if ((++fingerprintCount % logInterval) == 0)
                        {
                            Logger.Debug("Processed " + logInterval + " Fingerprints. Current Total: " + fingerprintCount.ToString("###,###"));
                        }
                    }
                }

            }

            return individualFingerprintCache;
        }
    }
}
